#include <QMap>

#include "Path.h"
#include "TileHandler.h"
#include "Tile.h"

Path::Path()
	: m_source(nullptr)
	, m_tileHandler(nullptr)
	, m_score(SIZE * SIZE)
{

}

Path::Path(Tile *target, Tile *source, TileHandler *tileHandler)
	: m_source(source)
	, m_tileHandler(tileHandler)
	, m_score(SIZE * SIZE)
{
	m_entries =
	{
		PathEntry(target, 0)
	};

	m_strategies =
	{
		PathFindingStrategy::left(),
		PathFindingStrategy::right(),
		PathFindingStrategy::top(),
		PathFindingStrategy::bottom()
	};

	for (int i = 0; i < m_entries.count(); i++)
	{
		const PathEntry &entry = m_entries[i];

		if (source == entry.first)
		{
			m_score = qMin(m_score, entry.second);
		}

		populate(entry);
	}
}

QList<Tile *> Path::untangle() const
{
	QList<Tile *> result;

	if (m_source)
	{
		untangle(m_source, result);
	}

	return result;
}

bool Path::operator <(const Path &other) const
{
	return m_score < other.m_score;
}

bool Path::operator >(const Path &other) const
{
	return m_score > other.m_score;
}

void Path::populate(const PathEntry &entry)
{
	const Tile *current = entry.first;

	const int score = entry.second + 1;
	const int x = current->x();
	const int y = current->y();

	const QList<Tile *> candidates =
	{
		m_tileHandler->tileAt(x + 1, y),
		m_tileHandler->tileAt(x - 1, y),
		m_tileHandler->tileAt(x, y + 1),
		m_tileHandler->tileAt(x, y - 1)
	};

	for (Tile *candidate : candidates)
	{
		for (const PathFindingStrategy &strategy : m_strategies)
		{
			if (strategy.canEnter(current, candidate) && !contains(candidate, score))
			{
				m_entries << PathEntry(candidate, score);
			}
		}
	}
}

void Path::untangle(Tile *source, QList<Tile *> &output) const
{
	const int x = source->x();
	const int y = source->y();

	QMap<int, PathEntry> neighbours;

	for (const PathEntry &entry : m_entries)
	{
		Tile *tile = entry.first;

		if ((qAbs(tile->x() - x) == 1 && tile->y() == y) ||
			(qAbs(tile->y() - y) == 1 && tile->x() == x))
		{
			neighbours.insertMulti(entry.second, entry);
		}
	}

	if (!neighbours.isEmpty())
	{
		const PathEntry &entry = neighbours.first();

		output << entry.first;

		if (entry.second)
		{
			untangle(entry.first, output);
		}
	}
}

bool Path::contains(Tile *tile, int score) const
{
	for (const PathEntry &entry : m_entries)
	{
		if (entry.first == tile && entry.second <= score)
		{
			return true;
		}
	}

	return false;
}
