#include <QDebug>
#include <QLineF>

#include "Game.h"
#include "Tile.h"
#include "TileItem.h"
#include "Player.h"
#include "StepCounter.h"
#include "TileCollection.h"

Game::Game()
	: m_started(false)
	, m_entrance(Tile::None)
	, m_timer(0)
	, m_player(Player::none())
	, m_pathFinder(m_tileHandler)
	, m_playerHandler(m_players, m_pathFinder, m_tileHandler)
	, m_scoreHandler(m_tileHandler, m_players)
	, m_server(m_playerHandler, *this)
{
	startTimer(10);
}

PlayerHandler *Game::playerHandler()
{
	return &m_playerHandler;
}

TileHandler *Game::tileHandler()
{
	return &m_tileHandler;
}

ActionVisualizer *Game::actionVisualizer()
{
	return &m_actionVisualizer;
}

ScoreHandler *Game::scoreHandler()
{
	return &m_scoreHandler;
}

bool Game::started() const
{
	return m_started;
}

bool Game::gameOver() const
{
	for (Player *player : m_players)
	{
		if (!player->dead() && player->connected())
		{
			return false;
		}
	}

	return m_started;
}

int Game::timer() const
{
	return m_timer;
}

void Game::place(Tile *tile)
{
	if (tile->candidate())
	{
		if (m_entrance != Tile::None)
		{
			if (tile->highlight() < 2)
			{
				StepCounter stepCounter;
				TileCollection tileCollection;

				m_tileHandler.updateCandidates(m_player, m_entrance);
				m_pathFinder.walkThePath(tile, m_player, stepCounter, tileCollection);

				for (Tile *tile : tileCollection)
				{
					const QList<Tile *> &neighbours = m_tileHandler.neighbours(tile);

					for (Tile *neighbour : neighbours)
					{
						const bool isPath = tileCollection
							.contains(neighbour);

						neighbour->setHighlight(1 + isPath);
					}
				}
			}
			else
			{
				tile->setEntrance(m_entrance);

				m_player->move(tile);
				m_player->populateHand(m_tileStack);

				m_entrance = Tile::None;

				m_outputHandler.refresh(m_player);
				m_tileHandler.updateCandidates(m_player, m_entrance);
				m_scoreHandler.invalidate(m_player);

				skipTurn();
			}
		}
	}

	emit gameOverChanged();
}

void Game::select(Player *player, const Reservation &reservation)
{
	if (player == m_player)
	{
		const int entrance = reservation.entrance();
		const int action = reservation.action();

		m_tileHandler.updateCandidates(m_player, entrance);
		m_actionVisualizer.visualize(action);

		m_entrance = entrance;
	}
	else
	{
		m_outputHandler.refresh(player);
	}
}

void Game::begin()
{
	for (Player *player : m_players)
	{	
		if (player->connected())
		{
			player->populateHand(m_tileStack);

			m_outputHandler.refresh(player);
		}
	}

	skipTurn();

	m_started = true;
	m_scoreHandler.invalidate(m_player);

	emit startedChanged();
}

void Game::skipTurn()
{
	Player *first = nullptr;
	Player *baseline = nullptr;
	Player *candidate = nullptr;

	for (Player *player : m_players)
	{
		if (player->connected() && !player->dead())
		{
			if (!first)
			{
				first = player;
			}

			if (player != m_player)
			{
				if (baseline && !candidate)
				{
					candidate = player;
				}
			}
			else
			{
				baseline = player;
			}
		}
	}

	m_player = candidate ?: first;
	m_timer = TIMER;

	emit playerChanged();
}

void Game::timerEvent(QTimerEvent *)
{
	if (m_started && (m_timer -= 10) <= 0)
	{
		skipTurn();
	}

	emit timerChanged();
}
