import sys
import random
from itertools import product
from astar_grid import AStarGrid, AStarGridNode
from coord import Coord

class World(object):
    debug = False
    tiles = []
    bombs = []
    bombRange = 2
    width = 0
    height = 0
    nodes = None
    graph = None
    enemies = []
    directions = { 'UP': Coord(x=0, y=-1), 'LEFT': Coord(x=-1, y=0), 'RIGHT': Coord(x=1, y=0), 'DOWN': Coord(x=0, y=1) }

    def __init__(self, size, tiles):
        if self.debug:
            print "Constructing world object. Size: {0}".format(size)
        self.update(size, tiles)

    def update(self, size, tiles):
        if self.debug:
            print "Updating world object. Size: {0}".format(size)
        self.width = size.x
        self.height = size.y
        self.tiles = [[square for square in row] for row in tiles]

    def updateBombs(self, bombs):
        self.bombs = bombs
        for bomb in bombs:
            others = self.inRangeOfBomb(bomb)
            others.append(bomb)
            lowest = self.lowestState(others)
            for b in others:
                b["state"] = lowest

    def lowestState(self, bombs):
        lowest = 100
        for bomb in bombs:
            if bomb["state"] < lowest:
                lowest = bomb["state"]
        return lowest

    def inRangeOfBomb(self, bomb):
        others = []
        for other in self.bombs:
            if bomb != other:
                if other["x"] >= bomb["x"] - 2 and other["x"] <= bomb["x"] + 2 and other["y"] == bomb["y"]:
                    others.append(other)
                if other["y"] >= bomb["y"] - 2 and other["y"] <= bomb["y"] + 2 and other["x"] == bomb["x"]:
                    others.append(other)
        return others
            
    def moveRequired(self, fromTile, toTile):
        for direction, move in self.directions.iteritems():
            if (fromTile + move) == toTile:
                if self.isRockTile(toTile):
                    return "BOMB"
                return direction
        return None

    def isInsideMap(self, p):
        return (p.x < self.width and p.y < self.height and p.x > -1 and p.y > -1)

    def isRockTile(self, tile):
        return (self.tiles[tile.y][tile.x] == "#")

    def isPlayer(self, tile):
        for player in self.enemies:
            if player["x"] == tile.x and player["y"] == tile.y:
                return True
        return False

    def passableTile(self, tile):
        if self.tiles[tile.y][tile.x] == "+":
            return False
        for bomb in self.bombs:
            if bomb["x"] == tile.x and bomb["y"] == tile.y:
                return False
        # for player in self.enemies:
            # if player["x"] == tile.x and player["y"] == tile.y:
                # return False
        return (self.tiles[tile.y][tile.x] != "+")

    def isPassable(self, tile):
        if self.tiles[tile.y][tile.x] != ".":
            return False
        for bomb in self.bombs:
            if bomb["x"] == tile.x and bomb["y"] == tile.y:
                return False
        for player in self.enemies:
            if player["x"] == tile.x and player["y"] == tile.y:
                return False
        return True

    def isDeadly(self, tile):
        for bomb in self.bombs:
            if (tile.x >= bomb["x"] - 2 and tile.x <= bomb["x"] + 2 and tile.y == bomb["y"]):
                return True
            if (tile.y >= bomb["y"] - 2 and tile.y <= bomb["y"] + 2 and tile.x == bomb["x"]):
                return True
        return False

    def deadlyWithin(self, tile, turns):
        for bomb in self.bombs:
            if (tile.x >= bomb["x"] - 2 and tile.x <= bomb["x"] + 2 and tile.y == bomb["y"] and bomb["state"] <= turns):
                return True

            if (tile.y >= bomb["y"] - 2 and tile.y <= bomb["y"] + 2 and tile.x == bomb["x"] and bomb["state"] <= turns):
                return True                

        return False

    def escapeDangerMap(self, start, level, visited, firstDir=None):
        for tile, direction in self.getNeighbours(start):
            if tile in visited:
                pass
            else:
                visited.append(tile)
                if self.isPassable(tile):
                    if not self.isDeadly(tile):
                        if None == firstDir:
                            yield (direction, level)
                        else:
                            yield (firstDir, level)
                    else:
                        if None == firstDir:
                            for dir, lvl in self.escapeDangerMap(tile, level+1, visited, direction):
                                yield (dir, lvl)
                        else:
                            for dir, lvl in self.escapeDangerMap(tile, level+1, visited, firstDir):
                                yield (dir, lvl)

    def escapeDanger(self, start):
        minLevel = 100000
        rightWay = {}
        for direction, level in self.escapeDangerMap(start, 0, [start], None):
            if level <= minLevel:
                minLevel = level
                if level not in rightWay:
                    rightWay[level] = []
                rightWay[level].append(direction)
        if minLevel in rightWay:
            return random.choice(rightWay[minLevel])
        else:
            return None

    def escapeDangerOld(self, start):
        unsafe = []
        nextUnsafe = []
        for tile, direction in self.getNeighbours(start):
            if self.isPassable(tile):
                return direction
            else:
                unsafe.append((tile, direction))
        for tile, direction in unsafe:
            for nextTile, nextMove in self.getNeighbours(tile):
                if self.isPassable(nextTile):
                    return direction
                else:
                    nextUnsafe.append((nextTile, direction))
        for tile, direction in nextUnsafe:
            for nextTile, nextMove in self.getNeighbours(tile):
                if self.isPassable(nextTile):
                    return direction
        return random.choice(["UP", "DOWN", "LEFT", "RIGHT"])

    def getNeighbours(self, pos):
        for direction, move in self.directions.iteritems():
            tile = pos + move
            if self.isInsideMap(tile):
                yield (tile, direction)

    def possibleMoves(self, pos):
        moves = []
        # Iterates over all possible directions
        for direction, move in self.directions.iteritems():
            tile = pos + move

            # Check if the direction puts us outside map
            if self.isInsideMap(tile):

                # Check if the direction puts us on a passable tile
                if self.passableTile(tile):

                    # Check if the tile is deadly within the next x turns
                    if not self.deadlyWithin(tile, 2):

                        # Add direction as possibility
                        moves.append(direction)

        return moves

    def createGraph(self):
        nodes = [[AStarGridNode(x, y, self.tiles[y][x], self.isPlayer(Coord(x=x, y=y)), self.isDeadly(Coord(x=x, y=y))) for x in range(self.width)] for y in range(self.height)]
        graph = {}

        for y, x in product(range(self.height), range(self.width)):
            currentTile = Coord(x=x, y=y)
            node = nodes[y][x]
            graph[node] = []
            moves = self.possibleMoves(currentTile)
            # print "{0} possible moves for {1}".format(moves, currentTile)
            for move in moves:
                target = currentTile + self.directions[move]
                graph[node].append(nodes[target.y][target.x])

        return graph, nodes

    def search(self, start, end):
        if self.debug:
            print "Looking for path from {0} to {1}".format(start, end)

        if (end == None):
            return []

        if (start == end):
            return []

        #if(self.graph == None or self.nodes == None):
        self.graph, self.nodes = self.createGraph()

        astar = AStarGrid(self.graph)
        start, end = self.nodes[start.y][start.x], self.nodes[end.y][end.x]
        path = astar.search(start, end)

        if path is None:
            return None

        # Convert found path to sequence of coordinates
        path = [Coord(x=tile.x, y=tile.y) for tile in path]

        # Remove first tile since this is our starting point and is not needed
        return path[1::]
