#include "FleeToSafePos.h"
#include <vector>
#include <queue>
#include "GameManager.h"
#include "FollowPathGoal.h"
#include "MapEvent.h"
#include "FleeGoal.h"
#include "FollowPathGoal.h"

void FleeToSafePos::Activate()
{
    m_status = OdinAI::GOAL_ACTIVE;
    
    for(int i = 5;i > 0;--i)
    {
        std::list<int> path = Search(i);
        if(path.size() > 0)
        {
            AddGoal(new FollowPathGoal(m_pOwner, path));
            break;
        }
    }
}
    
int FleeToSafePos::Process()
{
    ActivateIfInactive();

    m_status = ProcessSubgoals();

    return m_status;
}

void FleeToSafePos::OnMapChanged(const OdinAI::Event *pEvent)
{
    Clear();
    m_map = *dynamic_cast<const MapChangedEvent*>(pEvent)->GetMap();
    Activate();
}

std::list<int> FleeToSafePos::Search(int minRange)
{
    struct Edge
    {
        int from;
        int to;
        int cost;
    };

    const auto &curMap = gGameMgr.GetMap();
    const iVec2 &pos = m_pOwner->GetPosition();
    int startNode = gGameMgr.GetMap().GetNodeIdx(pos);

    int numNodes = curMap.GetSizeX() * curMap.GetSizeY();
    std::vector<bool> visited(numNodes);
    std::vector<int> route(numNodes);
    std::queue<Edge> queue;
    
    Edge dummy = {startNode, startNode, 0};
    queue.push(dummy);
    
    int targetNode = -1;
    while(!queue.empty())
    {
        Edge edge = queue.front();
        queue.pop();

        route[edge.to] = edge.from;

        iVec2 pos = curMap.GetPosFromNodeIdx(edge.to);
        char tile = m_map[pos.x][pos.y];
        if(tile < 32 && (tile-1) == edge.cost)
            continue;

        if(curMap.IsTileStandAble(tile))
        {
            bool isSafe = true;
            for(int x = pos.x-minRange;x < pos.x+minRange;++x)
            {
                for(int y = pos.y-minRange;y < pos.y+minRange;++y)
                {
                    if(x >= 0 && x < curMap.GetSizeX() && y >= 0 && y < curMap.GetSizeY())
                    {
                        if(m_map[x][y] == static_cast<char>(Tile::Wall))
                        {
                            isSafe = false;
                            break;
                        }
                    }
                }
            }

            if(isSafe)
            {
                // We found an empty positions which are not in range of any bomb, this maybe our escape path.
                targetNode = edge.to;
                break;
            }
        }

        Edge edges[4];
        int curEdge = 0;

        // Left edge
        if(pos.x > 0 && (curMap.IsTileStandAble(m_map[pos.x-1][pos.y]) || m_map[pos.x-1][pos.y] < 32))
        {
            edges[curEdge].from = edge.to;
            edges[curEdge].to = curMap.GetNodeIdx(iVec2(pos.x-1, pos.y));
            ++curEdge;
        }

        // Right edge
        if(pos.x + 1 < curMap.GetSizeX() && (curMap.IsTileStandAble(m_map[pos.x+1][pos.y]) || m_map[pos.x+1][pos.y] < 32))
        {
            edges[curEdge].from = edge.to;
            edges[curEdge].to = curMap.GetNodeIdx(iVec2(pos.x+1, pos.y));
            ++curEdge;
        }

        // Top edge
        if(pos.y > 0 && (curMap.IsTileStandAble(m_map[pos.x][pos.y-1]) || m_map[pos.x][pos.y-1] < 32))
        {
            edges[curEdge].from = edge.to;
            edges[curEdge].to = curMap.GetNodeIdx(iVec2(pos.x, pos.y-1));
            ++curEdge;
        }

        // Bottom edge
        if(pos.y + 1 < curMap.GetSizeY() && (curMap.IsTileStandAble(m_map[pos.x][pos.y+1]) || m_map[pos.x][pos.y+1] < 32))
        {
            edges[curEdge].from = edge.to;
            edges[curEdge].to = curMap.GetNodeIdx(iVec2(pos.x, pos.y+1));
            ++curEdge;
        }

        for(int i = 0;i < curEdge;++i)
        {
            edges[i].cost = edge.cost + 1;
            if(!visited[edges[i].to])
            {
                queue.push(edges[i]);

                visited[edges[i].to] = true;
            }
        }
    }

    std::list<int> path;
    if(targetNode < 0)
        return path;

    int node = targetNode;
    path.push_front(node);

    while(node != startNode)
    {
        node = route[node];
        path.push_front(node);
    }

    return path;
}

void FleeToSafePos::End()
{
    gGameMgr.GetEventMgr().RemoveEventListener("mapChanged", std::bind(&FleeToSafePos::OnMapChanged, this, std::placeholders::_1), this);
}