/*******************************************************************************
 * ________       .___.__           _____  .___ 
 * \_____  \    __| _/|__| ____    /  _  \ |   |
 *  /   |   \  / __ | |  |/    \  /  /_\  \|   |
 * /    |    \/ /_/ | |  |   |  \/    |    \   |
 * \_______  /\____ | |__|___|  /\____|__  /___|
 *         \/      \/         \/         \/     
 *
 * Copyright (c) Emil Sandst 2012
 *******************************************************************************/
#ifndef ODINAI_EVENT_MANAGER_H_
#define ODINAI_EVENT_MANAGER_H_

#include <functional>
#include <map>
#include <string>
#include <vector>
#include <tuple>
#include <memory>
#include "OdinAI/Event.h"
#include "OdinAI/Timer.h"
#include "SharedDefs.h"

namespace OdinAI
{

typedef std::function<void(const Event*)> EventCallback;

/**
 *  Event manager class manages events that are fired by different sub systems.
 */
class EventManager
{
public:
	EventManager() {};

	/**
	 *  Binds a callback function with a specific event.
	 *  @param eventName The name of the event
	 *  @param callback Points to a callback function, which is called when the specific event fired.
	 */
	void AddEventListener(const char *eventName, EventCallback callBackfunction, void *pOwner);

	/**
	 * Fire an event.
	 * @param event The event object that are sent to all listeners
	 * @param delay Wait delay ms before sending out the event
	 */
	void FireEvent(std::shared_ptr<Event> event, uint delay = 0);

    /**
     * Remove a callback function and prevent it from listening to an event.
     */
    void RemoveEventListener(const char *eventName, EventCallback callback, void *pOwner);

	/**
	 * Update the event manager, note it should only be done once.
	 */
	void Update();

	/**
	 * Release all dynamic memory.
	 */
	void Release();
private:
	struct EventRecord
	{
		std::shared_ptr<Event> event;
		uint delay; /*< Delay, as given in the FireEvent() call */
		uint sent; /*< Time point for when the record was created */
	};

    struct EventListener
    {
        EventCallback callback;
        void *pOwner;
    };

    std::map< std::string, std::vector<EventListener> > m_eventListeners;
	std::vector<EventRecord> m_eventStore;
};

}
#endif
