#ifndef ODINAI_DIJKSTRA_GRAPH_SEARCH_H_
#define ODINAI_DIJKSTRA_GRAPH_SEARHC_H_

#include "PriorityQueue.h"
#include <list>

namespace OdinAI
{
/** 
 * @example TargetCondition.h
 * Example of a TargetCondition.
 */

/**
 * @brief Given a graph, and an optional target, it calculates the shortest path from the source node to a node that fills your requirements.
 *
 * Below is an example: @include TargetCondition.h
 */
template <class GraphType, class TargetCondition, class PathValidation=DefaultPathValidation>
class DijkstraGraphSearch
{
private:
    typedef typename GraphType::EdgeType Edge;
  
private:
    const GraphType& m_Graph; //!< The graph to search.

    std::vector<const Edge*> m_ShortestPathTree; //!< Current shortest path tree. In other words shortest path from any node to any other node.

    std::vector<int> m_CostToThisNode; //!< Stores the cost to this node.

    std::vector<const Edge*> m_SearchFrontier; //!< Our search frontier.

    int m_iSource; //!< Source node.
    int m_iTarget; //!< Will after the search store the node found during the search.

    /**
     * Do the search.
     */
    void Search();
public:

    DijkstraGraphSearch(const GraphType &graph,
                         int source) : 
                                        m_Graph(graph),
                                        m_ShortestPathTree(graph.NumNodes()),                              
                                        m_SearchFrontier(graph.NumNodes()),
                                        m_CostToThisNode(graph.NumNodes()),
                                        m_iSource(source),
                                        m_iTarget(-1)
    {                                           
        Search();
    }
 
    /**
     * Get our shortest path tree, in other words the shortest path from any node to any node.
     * @return The shortest path tree.
     */
    std::vector<const Edge*> GetSPT() const{ return m_ShortestPathTree; }

    /**
     * Get the path. The list is empty if no path found.
     * @return The path represented as a list with the nodes that you must traverse between.
     */
    std::list<int> GetPath() const;

    /**
     * Get cost to our target.
     * @return Cost to our target.
     */
    int GetCostToTarget()const{ return m_CostToThisNode[m_iTarget]; }

    /**
     * Get cost to any node. NOTE it will only work if the node is on the SPT tree.
     * @return Cost from source node to this node.
     */
    int GetCostToNode(unsigned int nd) const { return m_CostToThisNode[nd]; }
};

template <class GraphType, class TargetCondition, class PathValidation>
void DijkstraGraphSearch<GraphType, TargetCondition, PathValidation>::Search()
{
    IndexedPriorityQLow<int> pq(m_CostToThisNode, m_Graph.NumNodes());

    pq.Insert(m_iSource);

    // While the queue is not empty
    while(!pq.Empty())
    {
        int nextClosestNode = pq.Pop();

        m_ShortestPathTree[nextClosestNode] = m_SearchFrontier[nextClosestNode];

        if (TargetCondition::IsTarget<GraphType>(m_Graph, nextClosestNode)) 
        {
            m_iTarget = nextClosestNode;
            return;
        }

        GraphType::ConstEdgeIterator ConstEdgeItr(m_Graph, nextClosestNode);
        for (const Edge* pE=ConstEdgeItr.Begin();
                !ConstEdgeItr.End();
                pE=ConstEdgeItr.Next())
        {
            int NewCost = m_CostToThisNode[nextClosestNode] + pE->Cost();

            bool valid = PathValidation::IsValid(m_Graph, pE->To(), NewCost);
            if(!valid)
                continue;

            if (m_SearchFrontier[pE->To()] == nullptr)
            {
                m_CostToThisNode[pE->To()] = NewCost;

                pq.Insert(pE->To());

                m_SearchFrontier[pE->To()] = pE;
            }
            else if ((NewCost < m_CostToThisNode[pE->To()]) &&
                     (m_ShortestPathTree[pE->To()] == nullptr))
            {
                m_CostToThisNode[pE->To()] = NewCost;

                pq.ChangePriority(pE->To());

                m_SearchFrontier[pE->To()] = pE;
            }
        }
    }
}

//-----------------------------------------------------------------------------
template <class GraphType, class TargetCondition, class PathValidation>
std::list<int> DijkstraGraphSearch<GraphType, TargetCondition, PathValidation>::GetPath() const
{
    std::list<int> path;

    //just return an empty path if no target or no path found
    if (m_iTarget < 0)  return path;

    int nd = m_iTarget;
    path.push_front(nd);

    while ((nd != m_iSource) && (m_ShortestPathTree[nd] != nullptr))
    {
        nd = m_ShortestPathTree[nd]->From();

        path.push_front(nd);
    }

    return path;
}
}
#endif