package skyport.ais;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import skyport.Coordinate;
import skyport.PlayerResources;
import skyport.Resources;
import skyport.SkyportException;
import skyport.WeaponResource;
import skyport.algorithms.astar.Node;
import skyport.algorithms.astar.Nodemap;
import skyport.constants.GameConstants;
import skyport.enums.Direction;
import skyport.enums.Tile;
import skyport.enums.WeaponType;
import skyport.json.PlayerBean;
import skyport.json.WeaponBean;

import com.fasterxml.jackson.core.JsonProcessingException;

public class Kaporder extends Ai {

	private static final String AINAME = "kaporder";

	private int actionsLeft;

	public Kaporder() {
		super(AINAME);
		actionsLeft = 0;
	}

	@Override
	protected void processTurn() throws JsonProcessingException {
		PlayerBean currentPlayer = game.getCurrentPlayer();

		actionsLeft = GameConstants.ACTIONS_PER_TURN;
		List<Direction> dirSequence = null;
		int index = 0;
		Coordinate target = null;
		PlayerResources playerResources = game.getResources()
				.getPlayerResources(aiName);

		System.out.println("My turn! Turn: " + game.getTurn() + " Scrap: "
				+ playerResources.getResourceCount(Tile.SCRAP) + " Rubidium: "
				+ playerResources.getResourceCount(Tile.RUBIDIUM)
				+ " Explosium: "
				+ playerResources.getResourceCount(Tile.EXPLODIUM));
		Coordinate pos = currentPlayer.getPosition();

		PlayerBean closestPlayer = game.getClosestPlayerFrom(pos);

		while (actionsLeft > 0) {
			if (closestPlayer != null) {
				target = closestPlayer.getPosition();
			} else {
				// System.out.println("I can't find any other players!");
				List<List<Node>> resourceRoutes = new ArrayList<List<Node>>();
				Nodemap nodemap = game.getNodemap();

				for (Tile tile : Tile.values()) {
					if (needResource(tile)) {
						Coordinate resourceCoord = game
								.getCoordinateToClosestTileFrom(pos, tile);

						if (resourceCoord != null) {
							resourceRoutes.add(nodemap.getRoute(pos,
									resourceCoord));
						}
					}
				}

				int minDistance = Integer.MAX_VALUE;
				List<Node> shortestRoute = null;

				for (List<Node> route : resourceRoutes) {
					int nodes = route.size();

					if (nodes < minDistance) {
						minDistance = nodes;
						shortestRoute = route;
					}
				}

				if (shortestRoute != null) {
					List<Direction> directions = getDirectionsFrom(pos,
							shortestRoute);
					Iterator<Direction> it = directions.iterator();

					while (it.hasNext() && actionsLeft > 0) {
						Direction dir = it.next();
						client.send(message.move(dir));
						pos.changeToCoordinateInDirection(dir);
						--actionsLeft;
					}
				} else {
					System.out
							.println("Can't find any valuable resource on the map");
					Direction dir = getSafeDirectionFrom(pos);

					if (dir != null) {
						client.send(message.move(dir));
						actionsLeft = 0;
					} else {
						// System.out.println("I can't move!");
						actionsLeft = 0;
					}
				}
			}

			if (actionsLeft > 0) {
				upgradeWeaponsIfPossible();
			}

			if (actionsLeft > 0 && target != null) {
				shootIfInRange(target);
			}

			if (actionsLeft > 0) {
				mineCurrentTileIfNeeded();
			}

			if (actionsLeft > 0) {
				if (dirSequence == null && target != null) {
					dirSequence = getDirectionSequenceTowards(actionsLeft, pos,
							target);
				}

				if (dirSequence != null && index < dirSequence.size()) {
					Direction dir = dirSequence.get(index);
					client.send(message.move(dir));
					pos.changeToCoordinateInDirection(dir);
					--actionsLeft;
					++index;
				} else {
					// System.out.println("Not enough directions");
					actionsLeft = 0;
				}
			}
		}
	}

	private void upgradeWeaponsIfPossible() throws JsonProcessingException {
		PlayerBean currentPlayer = game.getCurrentPlayer();
		WeaponBean[] weapons = { currentPlayer.getPrimaryWeapon(),
				currentPlayer.getSecondaryWeapon() };
		PlayerResources playerResources = game.getResources()
				.getPlayerResources(aiName);

		for (int i = 0; i < weapons.length; ++i) {
			if (playerResources.upgradeWeaponIfPossible(weapons[i])) {
				WeaponType type = weapons[i].getType();
				client.send(message.upgrade(type));
				weapons[i].setLevel(weapons[i].getLevel() + 1);
				System.out.println("upgraded " + type.toString());
				--actionsLeft;
			}
		}
	}

	private void shootIfInRange(final Coordinate target)
			throws JsonProcessingException {
		PlayerBean currentPlayer = game.getCurrentPlayer();
		Coordinate pos = currentPlayer.getPosition();
		WeaponBean[] weapons = weaponsSortedByDamage(currentPlayer);

		for (int i = 0; i < weapons.length && actionsLeft > 0; ++i) {
			WeaponType weaponType = weapons[i].getType();

			switch (weaponType) {
			case DROID:
				int droidRange = GameConstants.DROID_RANGE[weapons[i]
						.getLevel() - 1];

				if (pos.hasRangeTo(target, droidRange)) {
					List<Direction> sequence = getDirectionSequence(droidRange,
							pos, target);

					if (sequence != null) {
						client.send(message.droidAttack(sequence));
						actionsLeft = 0;
					}
				}
				break;
			case LASER:
				int laserRange = GameConstants.LASER_RANGE[weapons[i]
						.getLevel() - 1];

				if (pos.hasDirectRangeTo(target, laserRange)) {
					Direction dir = pos.getDirectionTowards(target);
					client.send(message.laserAttack(dir));
					actionsLeft = 0;
				}
				break;
			case MORTAR:
				int mortarRange = GameConstants.MORTAR_RANGE[weapons[i]
						.getLevel() - 1];

				if (pos.hasRangeTo(target, mortarRange)) {
					Coordinate mortarCoord = pos
							.getRelativeCoordinateTo(target);
					client.send(message.mortarAttack(mortarCoord));
					actionsLeft = 0;
				}
				break;
			default:
				break;
			}
		}
	}

	private void mineCurrentTileIfNeeded() throws JsonProcessingException {
		PlayerBean currentPlayer = game.getCurrentPlayer();
		Coordinate pos = currentPlayer.getPosition();
		Tile tile = game.getTilemap().getTileAt(pos);
		//System.out.println("Currently on tile " + tile.toString());

		if (tile != null && tile.isMineable()) {
			if (needResource(tile)) {
				// System.out.println("Tile can be mined");

				Resources resources = game.getResources();
				int resourceCount = resources.getResourceCountAt(pos);
				// System.out.println("Resources left on tile: " +
				// resourceCount);

				while (actionsLeft > 0 && resourceCount > 0) {
					// System.out.println("Let's mine!");
					client.send(message.mine());
					resources.decrementResourceCountAt(pos);
					resources.getPlayerResources(aiName)
							.incrementResourceCount(tile);
					--resourceCount;
					--actionsLeft;
				}
			} else {
				// System.out.println("I don't need this resource");
			}
		}
	}

	private boolean needResource(Tile resource) {
		PlayerBean currentPlayer = game.getCurrentPlayer();
		WeaponBean weapon1 = currentPlayer.getPrimaryWeapon();
		WeaponBean weapon2 = currentPlayer.getSecondaryWeapon();
		WeaponResource weaponResource = WeaponResource.getInstance();
		Tile resource1 = weaponResource.getResource(weapon1.getType());
		Tile resource2 = weaponResource.getResource(weapon2.getType());
		int maxLevel = GameConstants.MAX_WEAPON_LEVEL;

		return (resource == resource1 && weapon1.getLevel() < maxLevel)
				|| (resource == resource2 && weapon2.getLevel() < maxLevel);
	}

	public static void main(final String[] args) {
		Kaporder ai = new Kaporder();

		if (args.length == 2) {
			try {
				ai.setIp(args[0]);
				int port = Integer.parseInt(args[1]);
				ai.setPort(port);
			} catch (IllegalArgumentException e) {
				System.out.println("Usage: Kaporder <ip> <port_number>");
			}
		}

		try {
			ai.start();
		} catch (IOException | SkyportException e) {
			System.out.println(e.getMessage());
		}
	}
}
