package skyport;

import skyport.enums.Direction;

public class Coordinate {

	private int x;
	private int y;

	public Coordinate(int x, int y) {
		this.x = x;
		this.y = y;
	}

	public Coordinate(Coordinate other) {
		this.x = other.x;
		this.y = other.y;
	}

	public Coordinate(String string) {
		String[] items = string.split(",");

		try {
			x = Integer.parseInt(items[0].trim());
			y = Integer.parseInt(items[1].trim());
		} catch (Exception e) {
			x = -1;
			y = -1;
		}
	}

	public int getX() {
		return x;
	}

	public int getY() {
		return y;
	}

	public void changeToCoordinateInDirection(Direction dir) {
		switch (dir) {
		case LEFT_DOWN:
			++x;
			break;
		case DOWN:
			++x;
		case RIGHT_DOWN:
			++y;
			break;
		case RIGHT_UP:
			--x;
			break;
		case UP:
			--x;
		case LEFT_UP:
			--y;
			break;
		default:
			break;
		}
	}

	public Coordinate getCoordinateInDirection(Direction dir) {
		return getCoordinateInDirection(dir, 1);
	}

	public Coordinate getCoordinateInDirection(Direction dir, int range) {
		int newX = x;
		int newY = y;

		switch (dir) {
		case LEFT_DOWN:
			newX += range;
			break;
		case DOWN:
			newX += range;
		case RIGHT_DOWN:
			newY += range;
			break;
		case RIGHT_UP:
			newX -= range;
			break;
		case UP:
			newX -= range;
		case LEFT_UP:
			newY -= range;
			break;
		default:
			break;
		}

		return new Coordinate(newX, newY);
	}

	public Coordinate getRelativeCoordinateTo(Coordinate coord) {
		int diffX = coord.x - x;
		int diffY = coord.y - y;

		return new Coordinate(diffX, diffY);
	}

	public Direction getDirectionTowards(Coordinate coord) {
		int diffX = coord.x - x;
		int diffY = coord.y - y;
		int factor = diffX != 0 ? Math.abs(diffX) : Math.abs(diffY);

		if (factor != 0) {
			diffX /= factor;
			diffY /= factor;
		}

		if (diffX == -1) {
			if (diffY == -1) {
				return Direction.UP;
			} else if (diffY == 0) {
				return Direction.RIGHT_UP;
			}
		} else if (diffX == 0) {
			if (diffY == -1) {
				return Direction.LEFT_UP;
			} else if (diffY == 1) {
				return Direction.RIGHT_DOWN;
			}
		} else if (diffX == 1) {
			if (diffY == 0) {
				return Direction.LEFT_DOWN;
			} else if (diffY == 1) {
				return Direction.DOWN;
			}
		}

		return null;
	}

	public boolean hasDirectRangeTo(Coordinate coord, int range) {
		Direction dir = getDirectionTowards(coord);
		int distance;

		if (dir == null) {
			return false;
		}

		switch (dir) {
		case UP:
		case RIGHT_UP:
		case DOWN:
		case LEFT_DOWN:
			distance = Math.abs(coord.x - x);
			break;
		case LEFT_UP:
		case RIGHT_DOWN:
			distance = Math.abs(coord.y - y);
			break;
		default:
			distance = Integer.MAX_VALUE;
			break;
		}

		return range >= distance;
	}

	public boolean hasRangeTo(Coordinate coord, int range) {
		boolean withinRange = false;
		int x1 = x - range;
		int x2 = x + range;

		for (int currentX = x1; currentX <= x2 && !withinRange; ++currentX) {
			if (currentX == coord.x) {
				int diffX = x - currentX;

				int y1 = y - range - (diffX > 0 ? 0 : diffX);
				int y2 = y + range - (diffX < 0 ? 0 : diffX);

				for (int currentY = y1; currentY <= y2 && !withinRange; ++currentY) {
					withinRange = currentY == coord.y;
				}
			}
		}

		return withinRange;
	}

	public String getString() {
		return x + "," + y;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		Coordinate other = (Coordinate) obj;
		if (x != other.x)
			return false;
		if (y != other.y)
			return false;
		return true;
	}
}
