/*
 * Functions for loading and saving bitmap images
 * Programmed and designed by Matti 'ccr' Hamalainen
 * (C) Copyright 2012 Tecnic Software productions (TNSP)
 *
 * Please read file 'COPYING' for information on license and distribution.
 */
#ifndef LIBMGFX_H
#define LIBMGFX_H 1

#include "dmlib.h"
#include "dmfile.h"

#ifdef __cplusplus
extern "C" {
#endif


enum
{
    IMGFMT_PNG,
    IMGFMT_PPM,
    IMGFMT_PCX,
    IMGFMT_ILBM,
    IMGFMT_RAW,
    IMGFMT_ARAW,

    IMGFMT_LAST
};


enum
{
    DM_IFMT_PALETTE,
    DM_IFMT_RGB,
    DM_IFMT_RGBA,
    DM_IFMT_RGB_PLANE,
};


// RGBx color struct
typedef struct
{
    Uint8 r, g, b, a;
} DMColor;


// Bitmapped image struct (can be one of types specified by DM_IFMT_*)
typedef struct
{
    int width, height, pitch;
    BOOL constpal;
    int ncolors, ctransp;
    DMColor *pal;
    Uint8 *data;
} DMImage;


typedef struct
{
    int scale, nplanes, format;
    BOOL interleave, paletted;
} DMImageSpec;


typedef struct
{
    char *fext;
    char *desc;
    int  (*probe)(const Uint8 *buf, const size_t len);
    int  (*read)(const char *filename, DMImage **pimg);
    int  (*readFILE)(FILE *fp, DMImage **pimg);
    int  (*write)(const char *filename, DMImage *pimg, DMImageSpec *spec);
    int  (*writeFILE)(FILE *fp, DMImage *pimg, DMImageSpec *spec);
} DMImageFormat;


// Probe scores
#define DM_PROBE_SCORE_MAX     1000
#define DM_PROBE_SCORE_GOOD    750
#define DM_PROBE_SCORE_AVG     500
#define DM_PROBE_SCORE_MAYBE   250
#define DM_PROBE_SCORE_FALSE   0


extern DMImageFormat dmImageFormatList[IMGFMT_LAST];


DMImage * dmImageAlloc(int width, int height);
void      dmImageFree(DMImage *img);
int       dmImageGetBytesPerPixel(int format);
int       dmImageProbeGeneric(const Uint8 *buf, const size_t len, DMImageFormat **fmt, int *index);

BOOL dmCompareColor(const DMColor *c1, const DMColor *c2, BOOL alpha);


int dmWriteImageData(DMImage *img, void *cbdata, BOOL (*writeRowCB)(void *, Uint8 *, size_t), const DMImageSpec *spec);

int dmWriteIFFMasterRAWPalette(FILE *fp, DMImage *img, int ncolors, const char *indent, const char *type);
int dmWriteRAWImageFILE(FILE *fp, DMImage *img, DMImageSpec *spec);
int dmWriteRAWImage(const char *filename, DMImage *img, DMImageSpec *spec);

int dmWritePPMImageFILE(FILE *fp, DMImage *img, DMImageSpec *spec);
int dmWritePPMImage(const char *filename, DMImage *img, DMImageSpec *spec);

#ifdef DM_USE_LIBPNG
int dmWritePNGImageFILE(FILE *fp, DMImage *img, DMImageSpec *spec);
int dmWritePNGImage(const char *filename, DMImage *img, DMImageSpec *spec);
#endif

int dmWritePCXImageFILE(FILE *fp, DMImage *img, DMImageSpec *spec);
int dmWritePCXImage(const char *filename, DMImage *img, DMImageSpec *spec);
int dmReadPCXImageFILE(FILE *fp, DMImage **pimg);
int dmReadPCXImage(const char *filename, DMImage **pimg);


#ifdef __cplusplus
}
#endif

#endif // LIBMGFX_H
