/*
 * DMLib
 * -- Resource management
 * Programmed and designed by Matti 'ccr' Hamalainen
 * (C) Copyright 2011-2012 Tecnic Software productions (TNSP)
 */
#ifndef DMRES_H
#define DMRES_H

#include "dmlib.h"

#ifdef DM_USE_PACKFS
#include "dmpack.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif


/* Constants
 */
enum
{
    DMPRUNE_ATIME   = 0x0001,
    DMPRUNE_MTIME   = 0x0002,
};

enum
{
    DRF_USE_PACK    = 0x0001,
    DRF_USE_STDIO   = 0x0002,
    DRF_PRELOAD_RAW = 0x0004,
    DRF_PRELOAD_RES = 0x0008,
};

enum
{
    DMF_PERSIST     = 0x0001, // Persist loaded RAW resource
    DMF_TEMPORARY   = 0x0002,
    DMF_UNALLOCATED = 0x0004, // The raw data is not allocated, so do not free it
    DMF_LOADED_RAW  = 0x1000, // Raw data has been loaded
    DMF_LOADED_RES  = 0x2000, // Resource has been loaded
};


/* Typedefs and structures
 */
struct DMResourceLib;
struct DMResourceDataOps;
struct DMResourceOps;
struct DMResource;

typedef struct DMResource
{
    // Timestamps (in seconds from time())
    int    mtime,              // When resource was loaded
           atime;              // Last accessed (dmResourceRef()/unref)
    int    refcount;           // Reference count

    int    flags;              // Resource flags (DMF_*)
    char   *filename;

    // Raw data (or mem data)
    size_t rawSize;            // Size of data
    off_t  rawOffset;          // Current offset in data
    Uint8 *rawData;            // Pointer to data

    // Decoded resource data
    void   *resData;
    size_t resSize;
    struct DMResourceDataOps *rops;

    int    error;              // Error code

#ifdef DM_USE_STDIO
    FILE * fh;                 // File handle for stdio
#endif

    struct DMResourceOps *fops;    // Pointer to file handling functions struct
    struct DMResourceLib *lib;     // Pointer to the resource library
    struct DMResource *next, *prev;
} DMResource;


typedef struct DMResourceDataOps
{
    BOOL   (*probe)(DMResource *res, const char *fext);
    int    (*load)(DMResource *res);
    void   (*free)(DMResource *res);
} DMResourceDataOps;


typedef struct DMResourceLib
{
    int  flags;
    char *resPath;

    DMResource *resources, *preload;
    DMMutex *mutex;

#ifdef DM_USE_PACKFS
    char *packFilename;
    DMPackFile *packFile;
#endif
} DMResourceLib;


typedef struct DMResourceOps
{
    char    *name;

    int     (*freset)(DMResource *);
    int     (*ferror)(DMResource *);
    int     (*fseek)(DMResource *, const off_t, const int);
    off_t   (*fsize)(DMResource *);
    off_t   (*ftell)(DMResource *);
    BOOL    (*feof)(DMResource *);
    int     (*fgetc)(DMResource *);
    int     (*fputc)(int, DMResource *);
    size_t  (*fread)(void *, const size_t, const size_t, DMResource *);
    size_t  (*fwrite)(void *, const size_t, const size_t, DMResource *);

    int     (*fopen)(DMResource *);
    void    (*fclose)(DMResource *);
    int     (*preload)(DMResource *);
} DMResourceOps;


/* Functions
 */
int          dmResourcesInit(DMResourceLib **lib, const char *filename, const char *path, const int flags, int (*classifier)(DMResource *));
int          dmResourcesClose(DMResourceLib *lib);

void         dmResourcesPrune(DMResourceLib *lib, const int agems, int const flags);
int          dmResourcesPreload(DMResourceLib *lib, BOOL start, int *loaded, int *total);

DMResource * dmResourceNew(DMResourceLib *lib, const char *filename, const size_t size);
void         dmResourceFree(DMResource *node);
void         dmResourceInsert(DMResourceLib *lib, DMResource * node);
void         dmResourceDelete(DMResourceLib *lib, DMResource * node);
DMResource * dmResourceFind(DMResourceLib *lib, const char *filename);
int          dmResourceRef(DMResource *);
int          dmResourceUnref(DMResource *);


// Opening and closing resources
int          dmf_open(DMResourceLib *lib, const char *, DMResource **handle);
int          dmf_create_memio(DMResourceLib *lib, const char *, Uint8 *buf, size_t len, DMResource **phandle);
#ifdef DM_USE_STDIO
int          dmf_create_stdio(const char *filename, const char *mode, DMResource **phandle);
int          dmf_create_stdio_stream(FILE *, DMResource **phandle);
#endif
void         dmf_close(DMResource *);


// Basic resource access functions
int          dmfreset(DMResource *);
int          dmferror(DMResource *);
int          dmfseek(DMResource *, const off_t, const int);
off_t        dmfsize(DMResource *);
off_t        dmftell(DMResource *);
BOOL         dmfeof(DMResource *);
int          dmfgetc(DMResource *);
int          dmfputc(int, DMResource *);
size_t       dmfread(void *, const size_t, const size_t, DMResource *);
size_t       dmfwrite(void *, const size_t, const size_t, DMResource *);
char *       dmfgets(char *s, int size, DMResource * f);


// Helper functions for endianess based reading etc
int          dmf_read_str(DMResource *, void *, size_t);
BOOL         dmf_read_byte(DMResource *, Uint8 *);

BOOL         dmf_read_be16(DMResource *, Uint16 *);
BOOL         dmf_read_be32(DMResource *, Uint32 *);
BOOL         dmf_read_le16(DMResource *, Uint16 *);
BOOL         dmf_read_le32(DMResource *, Uint32 *);

#ifdef DM_HAVE_64BIT
BOOL         dmf_read_be64(DMResource *, Uint64 *);
BOOL         dmf_read_le64(DMResource *, Uint64 *);
#endif


#ifdef __cplusplus
}
#endif

#endif // DMRES_H
