# $Id: Tests.pm,v 1.15 2012/03/01 18:11:33 je Exp $

package Test::Tests;

use Fuzz;
use Generator::Random;
use Ly::Mode;
use Router qw(router);
use Test::Simple tests => 25;
use Test::Song;

sub test_chords {
  my $ly_mode = Ly::Mode->new(modes => {});
  my $r;

  $r = $Test::Song::Chords{a}->expr;
  ok($r->blessed eq 'Expr::Sequence::WithDuration::Chords',
     'Return type is for chords');
  ok(to_ly($r, $ly_mode) eq '\chordmode { | e2 a4:m d8:maj7 g8:6 }',
     'Lilypond output for some basic chords');

  $r = $Test::Song::Chords{b}->expr;
  ok(to_ly($r, $ly_mode) eq '\chordmode { | e2 a4:m d8:m g8 | a4 }',
     'Lilypond output for chords modifiers');

  $r = $Test::Song::Chords{c}->expr;
  ok(to_ly($r, $ly_mode)
       eq '\chordmode { | a4.:m \times 5/8 { d1:m } }',
     'Lilypond output for times notation');

  # p [ map { $_->can('noteset') ? $_->noteset : () }
          # $Test::Song::Chords{a}->expr->complete->joined_elements ];
  # exit;
}

sub test_generators {
  # XXX nothing yet
}

sub test_joining {
  my $ly_mode = Ly::Mode->new(modes => {});
  my @r;
  my @expected_durations = (0.125, 0.125, 0.3125, 0.0625, 0.0625, 0.25, 0.0625);

  @r = map { $_->duration }
           $Test::Song::Melody{a}->expr->complete->joined_elements;
  ok((all { $_ }
          pairwise { $b->blessed eq 'Expr::Duration'  &&  $a == $b->length }
                   @expected_durations,
                   @r),
     'Joining returns durations of expected length');
}

sub test_lyrics {
  my $ly_mode = Ly::Mode->new(modes => {});
  my $r;

  $r = $Test::Song::Lyrics->expr;
  ok($r->blessed eq 'Expr::Sequence::Lyrics',
     'Return type for lyrics');
  ok(to_ly($r, $ly_mode)
       eq '\\lyricmode { tur -- nip -- sit tu -- le -- vat tur -- nip -- sit '
          . 'tu -- le -- vat tur -- nip -- sit o -- ve -- lat '
          . 'ne tu -- le -- vat }',
     'Lilypond output for lyrics');
}

sub test_matching {
  # XXX not yet
  # p $Test::Song::Melody{b}->expr ~~ $Test::Song::Chords{c}->expr;
}

sub test_parts {
  my $ly_mode = Ly::Mode->new(modes => {});
  my $r;

  $r = $Test::Song::Parts{chords}->expr(section => 'verse');
  ok($r->blessed eq 'Expr::Sequence::WithDuration::Chords',
     'Verse part returns correct expression type');
  ok(to_ly($r, $ly_mode) eq '\chordmode { | e2 a4:m d8:maj7 g8:6 }',
     'Lilypond output for verse part');

  $r = $Test::Song::Parts{chords}->expr(section => 'bridge');
  ok(to_ly($r, $ly_mode) eq '\chordmode { | e2 a4:m d8:m g8 | a4 }',
     'Lilypond output for bridge part');

  $r = $Test::Song::Parts{chords} % 'ab';
  ok($r->blessed eq 'Expr::Delayed',
     'Parts with tags but without section returns Expr::Delayed type');
  $r = $r->expr(section => 'verse');
  ok($r->blessed eq 'Expr::Sequence::WithDuration::Chords',
     'Parts with preset tags and section returns expression type');
}

sub test_random {
  my $rand = Generator::Random->seed(49);
  my @got_numbers      = map { $rand->next } (1 .. 10);
  my @expected_numbers = (0.841492570471019,
			  0.0180743800010532,
			  0.84045874979347,
			  0.606050330447033,
			  0.087010654155165,
			  0.0422492183279246,
			  0.542025910690427,
			  0.415849933633581,
			  0.946597171481699,
			  0.98637643433176);
  ok((all { $_ }
       (pairwise { abs($a - $b) < 0.001 } @got_numbers, @expected_numbers)),
     'Random numbers are predictable');
}

sub test_rhythms {
  my $ly_mode = Ly::Mode->new(modes => {});
  my $r;

  $r = $Test::Song::Rhythms{a}->expr;
  ok($r->blessed eq 'Expr::Sequence::Rhythm',
     'Return type for rhythms');

  $r = $Test::Song::Melody{b}->expr;
  ok($r->blessed eq 'Expr::Sequence::WithDuration::Melody',
     'Return type for combined notes and rhythm');
  
  ok(to_ly($r, $ly_mode) eq "{ b8 g'16 c,16 f8 c16 e16 }",
     'Lilypond output for combines notes and rhythm');

  $r = $Test::Song::Rhythms{b}->expr;
  ok(to_ly($r, $ly_mode) eq '{ 16 16 16 16 16 16 16 16 4 8. 16 }',
     '(Pseudo) lilypond output for rhythm vector with default base');

  $r = $Test::Song::Rhythms{c}->expr;
  ok(to_ly($r, $ly_mode) eq '{ 8 32 32 32 32 16. 32 32 32 32 32 }',
     '(Pseudo) lilypond output for rhythm vector with non-default base');
}

sub test_router {
  my $router = router(Test::Song->make_router);
  ok($router->blessed eq 'Router',
     'Router can be made');
}

sub test_tagsequence {
  my $ly_mode = Ly::Mode->new(modes => {});
  my $r;

  $r = $Test::Song::Tagsequences->expr(tags => 'bcd');
  ok($r->blessed eq 'Expr::Sequence::WithDuration::Chords',
     'Return type in tagged sequence is correct expression type');
  ok(to_ly($r, $ly_mode)
       eq '\chordmode { | d1:m | c1 | d1:m | a1:m | c1 | c1 }',
     'Lilypond output for tag sequences');
}

sub test_tracks {
  my $ly_mode = Ly::Mode->new(modes => {});
  my $song    = Test::Song->new;

  my ($chordnames_track, $voice_track) = $song->tracks;
  ok($chordnames_track->blessed eq 'Track::ChordNames',
     'Songtracks include Track::ChordNames object');
  ok($voice_track->blessed eq 'Track::Voice',
     'Songtracks include Track::Voice object');

  ok(to_ly($chordnames_track, $ly_mode)
       =~ m/^\\new \s ChordNames           \s = \s "chords" \s { 
             \\set \s Staff.midiinstrument \s = \s " .* " .*
             \\chordmode \s { \s { \s | \s e2 \s a4:m
            .* $/gmx,
     'Chords track has sensible lilypond output');
}

sub test_vocals {
  my $ly_mode = Ly::Mode->new(modes => {});

  my $r = $Test::Song::Vocals->expr;
  ok($r->blessed eq 'Expr::Vocal',
     'Expr::Vocal object made');
}

sub run_tests {
  test_chords;
  test_generators;
  test_joining;
  test_lyrics;
  test_matching;
  test_parts;
  test_random;
  test_rhythms;
  test_router;
  test_tagsequence;
  test_tracks;
  test_vocals;
}

sub flatten { map { (ref($_) eq 'ARRAY') ? flatten(@$_) : $_ } @_ }

sub to_ly {
  my ($r, $ly_mode) = @_;
  (join ' ' => (flatten $r->to_ly($ly_mode)));
}

1;
