/**
   Graphic.c 
   Source file for all graphics related stuff.
   This is where the magic happens.
*/

#ifndef NO_GL
#include "graphic.h"
#include "GL/glx.h"
#include "GL/glext.h"
#include "shader.h"

#define NUMFUNCTIONS 10    //number of functions in *strs function array

//#define DEBUG2

#define oglCreateProgram	            ((PFNGLCREATEPROGRAMPROC)myglfunc[0])
#define oglCreateShader		            ((PFNGLCREATESHADERPROC)myglfunc[1])
#define oglShaderSource                 ((PFNGLSHADERSOURCEPROC)myglfunc[2])
#define oglCompileShader                ((PFNGLCOMPILESHADERPROC)myglfunc[3])
#define oglAttachShader                 ((PFNGLATTACHSHADERPROC)myglfunc[4])
#define oglLinkProgram                  ((PFNGLLINKPROGRAMPROC)myglfunc[5])
#define oglUseProgram                   ((PFNGLUSEPROGRAMPROC)myglfunc[6])
#define oglUniform4fv                   ((PFNGLUNIFORM4FVPROC)myglfunc[7])
#define oglUniform1i                    ((PFNGLUNIFORM1IPROC)myglfunc[8])
#define oglGetUniformLocation           ((PFNGLGETUNIFORMLOCATIONARBPROC)myglfunc[9])

static char *strs[] = {
	"glCreateProgram",
	"glCreateShader",
	"glShaderSource",
	"glCompileShader",
	"glAttachShader",
	"glLinkProgram",
	"glUseProgram",
	"glUniform4fv",
	"glUniform1i",
	"glGetUniformLocation",
    };

void *myglfunc[NUMFUNCTIONS];
static int pid;
static float fparams[4*5];


#ifdef DEBUG2
void printShaderInfoLog(GLuint obj)
{
  int infologLength = 0;
  int charsWritten  = 0;
  char *infoLog;
  
  glGetShaderiv(obj, GL_INFO_LOG_LENGTH,&infologLength);
  
  if (infologLength > 0)
    {
      infoLog = (char *)malloc(infologLength);
      glGetShaderInfoLog(obj, infologLength, &charsWritten, infoLog);
      printf("%s\n",infoLog);
      free(infoLog);
    }
}

void printProgramInfoLog(GLuint obj)
{
  int infologLength = 0;
  int charsWritten  = 0;
  char *infoLog;
  
  glGetProgramiv(obj, GL_INFO_LOG_LENGTH,&infologLength);
  
  if (infologLength > 0)
    {
      infoLog = (char *)malloc(infologLength);
      glGetProgramInfoLog(obj, infologLength, &charsWritten, infoLog);
      printf("%s\n",infoLog);
      free(infoLog);
    }
}
#endif

/**
   Init shaders.
*/
static void initShaders(int *pid, const char *vs, const char *fs)
{
  pid[0] = oglCreateProgram();
  const int vsId = oglCreateShader( GL_VERTEX_SHADER );
  const int fsId = oglCreateShader( GL_FRAGMENT_SHADER );
  oglShaderSource( vsId, 1, &vs, 0 );
  oglShaderSource( fsId, 1, &fs, 0 );
  oglCompileShader( vsId );
  oglCompileShader( fsId );
  oglAttachShader( pid[0], fsId );
  oglAttachShader( pid[0], vsId );
  oglLinkProgram( pid[0] );
}

/**
   Init function for graphics.
*/
int ginit()
{
  // init external gl commands
  int i;
  for(i=0; i<NUMFUNCTIONS;i++)
    {
      myglfunc[i] = glXGetProcAddress( (const unsigned char *)strs[i] );
      if( !myglfunc[i] )
	return(0);
    }

  // init shaders
  initShaders( &pid, vsh, fsh);

#ifdef DEBUG2
  printShaderInfoLog(pid);  
  printProgramInfoLog(pid);
#endif

    // camera position
    fparams[ 0] = 0.0f;
    fparams[ 1] = 0.0f;
    fparams[ 2] = 2.0f;
    fparams[ 3] = 0.0f; //global alpha
    // params
    fparams[ 4] = 0.0f;  // time
    fparams[ 5] = 0.0f; // dist set
    fparams[ 6] = 50.0f; //ray marching distance
    fparams[ 7] = 0.0f; //displace factor
    // light pos
    fparams[ 8] = 10.0f;
    fparams[ 9] = 30.0f;
    fparams[10] = 0.0f;
    fparams[11] = 0.1f;  //sphere size
    // color dif
    fparams[12] = 0.5f;   
    fparams[13] = 0.4f;   
    fparams[14] = 0.3f;   
    fparams[15] = 0.0f;   
    // color amp
    fparams[16] = 0.5f;
    fparams[17] = 0.5f;
    fparams[18] = 0.5f;
    fparams[19] = 1.0f;

  return(1);
}

static float stepsize = 0.1f;

/*insert random colors to the scene (diffuse,ambient)*/
void randCol(float time)
{
   // color dif
  fparams[12] = fparams[12]+(rand()%2000-1000)/(stepsize*1000.0f)-stepsize/2;   
  fparams[13] = fparams[12]+(rand()%2000-1000)/(stepsize*1000.0f)-stepsize/2;
  fparams[14] = fparams[12]+(rand()%2000-1000)/(stepsize*1000.0f)-stepsize/2;
  // color amp
  fparams[16] = fparams[12]+(rand()%2000-1000)/(stepsize*1000.0f)-stepsize/2;
  fparams[17] = fparams[12]+(rand()%2000-1000)/(stepsize*1000.0f)-stepsize/2;
  fparams[18] = fparams[12]+(rand()%2000-1000)/(stepsize*1000.0f)-stepsize/2;
}

/**
   Update by time the graphics.
*/
void gupdate(int time)
{
    //--- update parameters -----------------------------------------
  const float t = 0.001f*(float)time; 

  float f = t/10;
  fparams[4] = t;
  float send=2.5f;

  // phase one: fade in
  if (f<1.0f)
    fparams[3] = f;
  // second phase: translate camera weirdly around a ball
  else if(f<send && f>1.0f)
    {
      fparams[1] = sinf(1.0f*t+0.1f);
      fparams[2] = cosf(1.0f*t)+2;
      fparams[0] = cosf(1.0f*t);
      fparams[6] = 25.0f;
	
      if(fparams[1] < 0.0f)
	fparams[1] = 0.0f;
      if(fparams[2] < 0.0f)
	fparams[1] = 0.0f;
      if((send-0.3f)<f)
	fparams[3] = (send-f)*3.3333f;
    }
  // third phase: a ball
  else if(f<8.0f)
    {
      fparams[5] = 1.0f; //change distance set

      fparams[0] = 0.0f; //x
      fparams[1] = 0.0f; //y
      fparams[2] = 2.0f; //z
      
      if(f<(send+0.3f))
	fparams[3] = (f-send)*3.3333f;
      if(fparams[3] >= 0.98888)
	fparams[3] = 1.0f;
      
      float s = 29.0f;
      if(fparams[7]>=19.8f)
	s = 89.0f;

      // start displace
      if(f>(send+0.4f))
	{
	  fparams[7] = fmod((t-(s)),30.0f);
	}
    }
  
  // final fadeout
  if(t > 59.0f)
    fparams[3] = (60.0f-t);


    //--- render -----------------------------------------

    oglUseProgram( pid );
    oglUniform4fv( oglGetUniformLocation( pid, "fp" ),  5, fparams );
    glRects( -1, -1, 1, 1 );

#ifdef DEBUG2
    //  printShaderInfoLog(pid);  
    //  printProgramInfoLog(pid);
#endif

}

#endif
