/**
   Audio.c 
   This file contains the music player functions.
*/

#ifndef NO_AUDIO
#include "audio.h"

#ifndef TINY
//#include "sys/time.h"
//struct timeval start,current;
#endif

#define SIN    0
#define SAW    1
#define SQUARE 2
//#define TRI    4
#define NOISE  3

#define CHANNELS 15   //channel amount
#define TRACK_MAX 30  //track max length

#define END -2
#define LOOP -3  
#define OFF -4   //off
#define LOOPSTART -5
#define LOOPUNTIL -6
#define VOL -7

#define BIT_MULTI 44.1f      // convert ms to bits
#define FLOAT_MULTI 10922   //this is the float multiplicator of when converting to signed short. Comes from SHORT_MAX/3

unsigned int counter;

typedef struct _Channel
{
  int track[TRACK_MAX][2]; // music track
  int place;               // place on track (eg. next event)
  int loopstart[2];        // loop start pos. [0]==millis, [1]==track.place
  float vol;               // volume
  int a;                   // attack of the adsr
  int r;                   // release of the adsr
  int wave;                // waveform in use
} Channel;

static Channel channels[CHANNELS];

// 8bit short syna
static Channel ch = 
{
  //track
  {
    {LOOPSTART,0},
    {1000,554},
    {1500,785},
    {2000,785},
    {2500,554},
    {3200,262},
    {LOOPUNTIL,60000},
    {VOL,500},
    {OFF,0},
  },
  //place
  {0},
  //loopstart
  {0,0},
  //vol
  {0.7f}, //0.4
  //a
  {2},
  //r
  {10000},
  //wave
  {SAW},
  };

// tsirp
static Channel ch1 = 
{
  //track
  {
    {LOOPSTART,26000},
    {VOL,1000},
    {350,659},
    {500,880},
    {VOL,0},
    {1000,1},
    // {900,662},
    //   {1150,662},
    //    {1300,659},
    {LOOPUNTIL,60000},
    {OFF,0},
  },
  //place
  {0},
  //loopstart
  {0,0},
  //vol
  {0.0f},
  //a
  {4000},
  //r
  {400},
  //wave
  {SAW},
  };

// BEEP
static Channel ch2 = 
{
  //track
  {
    {LOOPSTART,49000},
    {VOL,10},
    {350,659},
    {500,880},
    {VOL,0},
    {1000,1},
    {LOOPUNTIL,60000},
    {OFF,0},
  },
  //place
  {0},
  //loopstart
  {0,0},
  //vol
  {0.1f},
  //a
  {2000},
  //r
  {40},
  //wave
  {NOISE},
  };

// KOHINA
static Channel ch3 = 
{
  //track
  {
    {LOOPSTART,0},
    //{VOL,200},
    {500,659},
    //{600,880},
    //{VOL,0},
    //{2000,1},
    {LOOPUNTIL,60000},
    {OFF,0},
  },
  //place
  {0},
  //loopstart
  {0,0},
  //vol
  {0.2f},
  //a
  {2},
  //r
  {10000},
  //wave
  {NOISE},
  };

// SIN basic
static Channel ch4 = 
{
  //track
  {
    //    {LOOPSTART,4700},
    {LOOPSTART,0},
    {700,746},
    {1200,880},
    // {900,662},
    //   {1150,662},
    //    {1300,659},
    {LOOPUNTIL,60000},
    {OFF,0},
  },
  //place
  {0},
  //loopstart
  {0,0},
  //vol
  {0.0f},
  //a
  {4000},
  //r
  {6000},
  //wave
  {SAW},
  };


/*
static int notei[]={
  // c,    C,    d,    D,    e,    f,    F,    g,    G,    a,    A,    b

  // 33,   35,   37,   39,   41,   44,   46,   49,   52,   55,   58,   62, // 2nd octave
  // 65,   69,   73,   78,   82,   87,   92,   98,   104,  110,  117,  123, // 3rd octave
  // 131,  139,  147,  156,  165,  175,  185,  196,  208,  220,  233,  247, // 4th
  // 262,  277,  294,  311,  330,  350,  370,  392,  415,  440,  466,  494, // 5th
  // 523,  554,  587,  622,  659,  698,  740,  784,  831,  880,  932,  988, // 6th
  // 1047, 1109, 1175, 1245, 1319, 1397, 1480, 1568, 1661, 1760, 1864, 1976 // 7th
};
*/

/** waveforms: f==frequency,n==place in wave */
#define noise() (rand()%2000-1000/1000.0f)
#define sine(f,n) (sin(n*2.0f*M_PI/(float)f))
#define square(f,n) ((n<f/2)?-1.0f:1.0f)
#define saw(f,n) (-1.0+fmod(1.0f+n*2.0f/(float)f,2.0f))
//#define triangle(f,n) (1.0-fabs(fmod(1.0+n*4.0/f,4.0)-2.0)

/** ADSR envelope: a=attack duration in bits, r=release duration in bits,n=placement*/
#define adsr(a,r,n)((n<=a)?(sin(n*0.5f*M_PI/(float)a)):(sin((r-n+a-1)*0.5f*M_PI/(float)r)))

/**
   Sound initialization.
   Prepares all the sound components that needs such an attention.
*/
void init(int f)
{
  counter = 0;

  //test channel & track
  channels[5].wave=END;
  channels[0]=ch;
  channels[1]=ch1;
  channels[2]=ch2;
  channels[3]=ch3;
  channels[4]=ch4;
}

/**
   Play function.
   This function fills in the sample data to both left and right channels.
   @param void* udata, user data. Not in use atm.
   @param unsigned char* stream, data stream to be filled
   @param int len, length of the needed data
*/
void play(void* udata,unsigned char* dest,int length)
{
  int n,k,event[2],time;
  float sample,csample;
  short* stream = (short*)dest;
  length=length/4;
 
  /* Loop through all bits in current sample */
  for(n=0;n<length;n++,counter++)
    {
      sample=0.0f;

      /* Loop through channels */
      for(k=0;channels[k].wave!=END;k++)

	{
	  // get current track 
	  event[0]=channels[k].track[channels[k].place][0];
	  event[1]=channels[k].track[channels[k].place][1];
	  
	  /* Handle current event */
	  /* Handle OFF */
	  if(event[0]==OFF)
	    channels[k].vol=0.0f;
	  /* Handle LoopStart*/
	  else if(event[0]==LOOPSTART)
	    {
	      if (event[1]*BIT_MULTI <=counter)
		{
		  channels[k].loopstart[0] = event[1];
		  channels[k].place++;
		  channels[k].loopstart[1] = channels[k].place;
		}
	    }
	  /* Handle LoopUntil */
	  else if(event[0]==LOOPUNTIL)
	    {
	      if(counter>=(event[1]*BIT_MULTI))
		{
		  channels[k].place++;
		}
	      else
		{
		  channels[k].place = channels[k].loopstart[1];
		  channels[k].loopstart[0] = counter;
		}
	    }
	  /* Change Volume */
	  else if(event[0]==VOL)
	    {
	      channels[k].vol = (float)event[1]/1000.0f;
	      channels[k].place++;
	    }
	  /* Play Note */
	  else
	    {
	      /* Handle the music playing */
	      time = (int)(event[0] * BIT_MULTI + channels[k].loopstart[0]);
	      
	      //note has fully played itself continue to next one
	      if((time + channels[k].a + channels[k].r) <= counter)
		{
		  channels[k].place++;
		}
	      // play some note
	      else if(time<=counter)
		{
		  // get the waveform
		  if(channels[k].wave == SIN)
		    csample = sine((SAMPLE_RATE/event[1]),(counter-event[0]));
		  if(channels[k].wave == SQUARE)
		    csample = square((SAMPLE_RATE/event[1]),(counter-event[0]));		  
		  if(channels[k].wave == SAW)
		    csample = saw((SAMPLE_RATE/event[1]),(counter-event[0]));
		  if(channels[k].wave == NOISE)
		    csample = noise();
		  
		  // get adsr envelope
		  csample *= adsr(channels[k].a,channels[k].r,(counter-event[0]));

		  // add to rest of the channels
		  sample += csample*channels[k].vol;
		}
	    }
	}
      
      // stereo division
      sample *= 0.5f;

      // high pass filter
      if(sample > 3.0f)
	sample = 3.0f;
      if(sample < -3.0f)
	sample = -3.0f;

      // write the sample to buffer
      stream[n*2+1] = (short)(sample*FLOAT_MULTI);
      stream[n*2] = (short)(sample*FLOAT_MULTI);
    }
}

#endif
