<?php
 
 /**
  * La classe CircuitBroker gre les informations relatives aux diffrents circuits.
  * Elle permet notamment d'obtenir la description complte d'un circuit, ou des donnes
  * plus spcifiques telles que le temps de qualification, la longueur ou les records 
  * (via la BDD pour ces derniers).
  *
  * La classe est utilise par l'appel Ajax circuit.php pour renvoyer les infos du circuit
  * au client, aprs vrification qu'il a bien les droits pour l'obtenir.  
  *
  * TODO : dfinir si les oprations de contrle des requtes utilisateur ont encore leur place ici
  * ou si elles sont  refactorer.
  */
 
 class CircuitBroker {

	private $validatedLevel = -1;
	private $validatedCar = -1;
	private $validatedMode = 2; // Arcade par dfaut
	private $validatedGhostName = '';
	private $validatedUserId = -1;
	private $validatedUserName = '';
	
	/**
	 * Rcupre les champs passs en paramtre de l'appel
	 * Elimine tous les caractres interdits, convertit dans le format attendu
	 * (numrique ..) pour canaliser les tentatives de triche (approche whitelist)
	 * La validation se fait au niveau du format (numrique, jeu de caractres, ..)
	 * mais non au niveau du contenu (intervalle de valeurs, smantique).
	 */
	public function cleanInputs() {
		$levelKey = 'level';
		if (array_key_exists($levelKey, $_POST)) {
			$uncheckedLevel = $_POST[$levelKey];
			$this->validatedLevel = (int)($uncheckedLevel);
		}
		
		$carKey = 'car';
		if (array_key_exists($carKey, $_POST)) {
			$uncheckedCar = $_POST[$carKey];
			$this->validatedCar = (int)($uncheckedCar);
		}
		
		$modeKey = 'mode';
		if (array_key_exists($modeKey, $_POST)) {
			$uncheckedMode = $_POST[$modeKey];
			$this->validatedMode = (int)($uncheckedMode);
		}
		
		$ghostKey = 'ghost';
		if (array_key_exists($ghostKey, $_POST)) {
			$uncheckedGhostName = $_POST[$ghostKey];
			$this->validatedGhostName = preg_replace('/[^- _\'\w]/', '_', $uncheckedGhostName);
		}
		
		$idKey = 'userId';
		if (isset($_SESSION[$idKey])) {
			$this->validatedUserId = intval($_SESSION[$idKey]);
		}
		
		$nameKey = 'userName';
		if (isset($_SESSION[$nameKey])) {
			$this->validatedUserName = preg_replace('/[^- _\'\w]/', '_', $_SESSION[$nameKey]);
		}
	}

	/**
	 * Vrifie que le joueur a bien accs aux infos attendues.
	 *  - le circuit correspondant est dbloqu
	 *  - le ghost demand existe
	 *  - le joueur dispose d'un jeton pour un ghost autre que le sien (pas encore actif)
	 *
	 * Si oui, enregistre les donnes (voiture, mode, circuit) comme tant la course courante, 
	 * pour revalidation en fin de course, et renvoie SUCCESS.
	 * Si non, renvoie un code d'erreur appropri.
	 */
	public function checkAccess($dbHelper) {
		
		// Vrification que le joueur est logu
		if ($this->validatedUserId < 1 || $this->validatedUserName == '') {
			// Cas particulier : joueur anonyme, le premier circuit est accessible en arcade, avec les deux premires voitures
			if ($this->validatedLevel==1 && ($this->validatedCar==1 || $this->validatedCar==2) && $this->validatedMode==2) {
				return SUCCESS;
			}
			return FATAL_NOT_LOGGED;
		}
	
		// Vrification que le niveau existe
		if ($this->validatedLevel<0 ||$this->validatedLevel>16) {
			ErrorHandler::getInstance()->setParameters($this->validatedLevel);
			return ERROR_INVALID_LEVEL;
		}
		
		if ($this->validatedCar<1 ||$this->validatedCar>16) {
			ErrorHandler::getInstance()->setParameters($this->validatedCar);
			return ERROR_INVALID_CAR;
		}
		
		// TODO : si je m'appelle admin, je peux avoir tous les circuits et tous les ghosts pour les vrifier !
		
		// Vrification que le joueur a bien dbloqu le niveau, et enregistrement dans la BDD de la course,
		// du mode et de la voiture demands (pour rutilisation en fin de course)
		return $dbHelper->authorizeRace($this->validatedUserId, $this->validatedLevel, $this->validatedMode, $this->validatedCar);
	
	}

	/**
	 * Renvoie la description du circuit demand,
	 *  partir des identifiants fournis et vrifis prcdemment
	 */
	public function getDescriptionUnchecked($dbHelper) {
		return $this->getDescriptionInternal($dbHelper, $this->validatedUserId, $this->validatedLevel, $this->validatedMode);
	}
	
	/**
	 * Renvoie la description du circuit demand,
	 * sans faire de vrification sur les accs (vrifis pralablement, ou appels par la console admin)
	 */
	public function getDescriptionInternal($dbHelper, $userId, $level, $mode) {
		$roadAndScenery = array ( // niveau 0 (construction kit) en premier
				'<road>"9s9O304011W05Z9639O011W05Z9630A011Ry7W0131f011Sa8W0120F4W1O4W111AQM006221A2o2a4o11JAcg006121Q8_1K8_"</road><ground count="1"><section offset="0" leftWidth="1" rightWidth="1" gripInside="1" gripOutside="0.6" patternDistance="1000" patternSize="5" groundColor="#804000 #805000 #904000 #804010 #804020" groundLine="0 1 2 3 4" /></ground><scenery><count>7</count><item><w>6</w><h>48</h><src>australian_roadmark.png</src><cl>0</cl><cw>6</cw></item><item><w>30</w><h>70</h><src>left_turn_sign.png</src><cl>13</cl><cw>4</cw></item><item><w>30</w><h>70</h><src>right_turn_sign.png</src><cl>13</cl><cw>4</cw></item><item><w>464</w><h>180</h><src>portique.png</src><cl>0</cl><cw>12</cw></item><item><w>12</w><h>180</h><src>portique_d.png</src><cl>0</cl><cw>12</cw></item><item><w>128</w><h>140</h><src>unfenced_road_sign.png</src><cl>99</cl><cw>4</cw></item><item><w>34</w><h>100</h><src>gravel_road_sign.png</src><cl>15</cl><cw>4</cw></item><horizon><w>1280</w><h>520</h><src>bg_cevennes.png</src></horizon></scenery>',
		/*  1 */'<road>"FyFe30B9iyT29YWA30IAKlTWAZS930F8SZSkCZmC30M8mgT-CXa030Z9O-US0YMG30O4TJTCCZ8030Q4YHUI0ZSC30o4YMRKAaiA30p4dVSkBZSA30s4nPT2AaEB30P4ngSu0Zm930L50bSa9Z8030e50eSuCY2I30q4OyTgIXk032W1G4TgMY2H32X1H4T-HYMM32Y1I4U8MYqH32Z1J4TWHYCH32Y1H5TCHXkM32X1O9UIMYWH30e0e1UI2Xk234t0e8T2AXa034s0yATg7Xa034u0u7SuJZ8G34w0oFRK7Zw734y1GGTC9Xa034-0yDSGJXa034a1UVPm9ai934i1QEUS0ZS934t0y9US0ZwG34v0y7US0aYJ35s49JTCCZ8035u44HUS0ZSC35y3mMRKAaiA35z3rVSkBZSA3603_PT2AaEB35j4TgSu0Zm935p38bSa9Z803664OeSuCY2I36I5AyTgIXa038N0yCSu0Yq7389168T2AXa03880yATg7Xa038A117SuJZ8G38C0oFRK7Zw738E1GGTC9Xa038G0yDSGJXa037s1UVPm9ai937-1QEUS0ZS93890y9US0ZwG38B0y7US0aYJ3942MFTq9YqC3993wITWCZS939644gTqBZ8039744aUS0YWB39i4sSPmCb0C3AK5KQT2FXa03Ae50WUS0YqF3BQ4EeY2EYgL3Bu3mZSQETWL3C-2gHSuEXa03DS2CJT-EXa03Cq2qOUS0YCE3DI2MFUS0ZSE3DX2CSRyDaiC3Dc22WRK0a4D3Cv2lGTCFZ8F3Dc22CTqFZ803DS2CBUS0YWF3E91f9US0YCF3EO1LDTWFXa010yKGO028411aKai02C312MUOS025412-KmU0283144Aqi0253149AST025324s4_4O4_16GeeQ02022724_3b2_27W3_1c4_27v4W3f4W2884W0O4W28c4_1Q4_29E4_1g4_19nPgc02A31AC8ST02441AK5aZ02431AeKqX02832Ao4_0Q4_1BBPMO02A42Bk4_0e4_1BpPMV02A41CCFkb02F31CRFKQ02F42CW4_4J4_1Cl5QT02541CqAui02A31C-5QT02541DNAQT02A42DS2_0Y2_1DXAuY02A31Dh5QT02A42Dh3_0Z2_1DrAQT02A41E2CQT02A42E43_0b3_1EOFMR02A42EY3_0c3_1Ed5gb02A31Ei5MR02A41En5gb02A31EsAMR02A42F03_0e3_1FAFci02A31FP5QT02A4304011W05Xi63EJ011SuKXa03Fe011W05Xi6"</road><ground count="2"><section offset="0" leftWidth="1" rightWidth="1" gripInside="1" gripOutside="0.5" patternDistance="1000" patternSize="5" groundColor="#658A70 #658A70 #658A70 #3B6150 #3B6150" groundLine="0 0 0 1 1" /><section offset="1200" leftWidth="1" rightWidth="1" gripInside="1" gripOutside="0.5" patternDistance="1000" patternSize="5" groundColor="#658A70 #658A70 #658A70 #3B6150 #3B6150" groundLine="0 0 0 1 2" /></ground><scenery><count>21</count><item><w>8</w><h>40</h><src>object_test_roadsign.png</src><cl>0</cl><cw>8</cw></item><item><w>78</w><h>40</h><src>panneau_virage_gauche.gif</src><cl>0</cl><cw>78</cw></item><item><w>78</w><h>40</h><src>panneau_virage_droite.gif</src><cl>0</cl><cw>78</cw></item><item><w>504</w><h>216</h><src>portique.png</src><cl>0</cl><cw>28</cw></item><item><w>28</w><h>216</h><src>portique_d.png</src><cl>0</cl><cw>28</cw></item><item><w>140</w><h>139</h><src>acacia_small.png</src><cl>71</cl><cw>5</cw></item><item><w>191</w><h>220</h><src>chataignier.png</src><cl>83</cl><cw>15</cw></item><item><w>97</w><h>96</h><src>Arbutus_unedo_g1_16.png</src><cl>51</cl><cw>5</cw></item><item><w>54</w><h>40</h><src>calluna_vulgaris_02.png</src><cl>0</cl><cw>54</cw></item><item><w>67</w><h>72</h><src>Erica_arborea.png</src><cl>0</cl><cw>67</cw></item><item><w>53</w><h>56</h><src>Cistus_monspeliensis_Magliano.png</src><cl>4</cl><cw>39</cw></item><item><w>176</w><h>64</h><src>genets.png</src><cl>9</cl><cw>161</cw></item><item><w>128</w><h>45</h><src>caillou_1.png</src><cl>7</cl><cw>115</cw></item><item><w>53</w><h>30</h><src>buisson_1.png</src><cl>4</cl><cw>44</cw></item><item><w>81</w><h>110</h><src>conifere_ile.png</src><cl>41</cl><cw>7</cw></item><item><w>103</w><h>88</h><src>joncs3.png</src><cl>4</cl><cw>98</cw></item><item><w>83</w><h>48</h><src>tas_pierres.png</src><cl>3</cl><cw>80</cw></item><item><w>100</w><h>120</h><src>conifere_gorges_tarn.png</src><cl>43</cl><cw>10</cw></item><item><w>181</w><h>160</h><src>rochers_en_equilibre.png</src><cl>0</cl><cw>181</cw></item><item><w>57</w><h>27</h><src>caillou_2.png</src><cl>1</cl><cw>55</cw></item><item><w>70</w><h>64</h><src>joncs4.png</src><cl>6</cl><cw>63</cw></item><horizon><w>1280</w><h>520</h><src>bg_cevennes.png</src></horizon></scenery>',
		/*  2 */'<road>"9s9O3030K6TqJXpJ3040C4W00XuI3050K6TgJZ8J3060E4W00Y2I3070K6TCJXkJ30C0SFSaHXuI3060K7W00XkH3080NBW00Y2I3050K6U8IW003051aARADcGD3021e8QYDbyD3063I7OCDemD3060eAW00Y2H3060jAW00Y2I3080n7W00XkJ30D0yAUIDW0031G7qCO-EaOE31G6G9RyEYgE31G4i7TgEfGE31v387a6EsME31w385R0EkTE31x384F8ESEE31y2q6J-EZhE31z507R5EekE31-3S7MyElyE31_4E5GnCZDE3203S7DGEbyE3214i9OlEmaE3223m8S1EoSE30U3wAKhCd-F30V3wAGiCg0E30W3w9CZFbtC30X3wAOgEaJF30Y3wB8kFM0C30Z3wAJMChuE30a3w8PSCtmF30b3wAReFewE30c3w9SfCdCF30d3wAECEkOF31u7qKSuFW0031_7WHTWFYCG3227MLTqFXuF31w7gMT-GY-F31x7gOTMFY2F31z7gITMCY2G3207gwT2CYqG3217g_T2GY-C3247WITgGZIC31h0tIUN2Xf233c0LIUN2Xf234E1HIUN2Xf236D0fIUN2Xf23740tIUN2Xf23880tIUN2Xf234O5UANaCfOM34Q5SAKSQd2P34S5QAP8PgAN34U5OCMKMk4M34W5M9MoNecO34Y5KBJMMdWC34a5IDd2MhGN34c5GEHAQR0P34e5EFLsNjmQ34g5CBE2PdMO34i5ACIkOR0M38S1QFTCMW0035y3SZW00YMM36l38eW00YgM36a3IjT-MW0036n38HSGLW0036k3wANaLfOM36m1GATqQckP38K2KATqQY2P36o1GAU8PY-N38M2IAU8PYMN36q3qCSaMZAM36s3o9T2NZwO36u3mBRAOYgL36v3mCRyPaOO36x3mGSQNf4P36z3mDT2Qb0N36_3m9ReLZmL3713mAQ4QZmO34E011W00YCK365011TgKW00370011W00YCK385011T-KW0038B011W00Y2K30C011T-7W00306011UIBW0031i011UI8W00304011W05Xi639O011W05Xi620F4W1O4W111AQM026421A2o2a4o11JAcg026321Q8_1K8_12gAgg026322l2_1U2_133AMR02642384_1NA_13Q7OQ026424OA_1N4_1445ea19011495OS110115UFge026325o8_1N6_15y5MR1A011615gb11012667_2Z4_26G3_3S8_16u5ke190116z5IO110127CC_2EC_17-5AJ1A011835sj110118_Kid0263"</road><ground count="4"><section offset="0" leftWidth="1" rightWidth="1" gripInside="1" gripOutside="0.6" patternDistance="1000" patternSize="5" groundColor="#B58A51 #B58A51 #B58A51 #B58A51 #B58A51" groundLine="0 1 2 3 4" /><section offset="108" leftWidth="1" rightWidth="1" gripInside="1" gripOutside="0.6" patternDistance="2000" patternSize="5" groundColor="#C2945C #CE9F64 #D3A26C #D7A672 #DCAA78" groundLine="5 6 7 8 9" /><section offset="112" leftWidth="1" rightWidth="1" gripInside="1" gripOutside="0.6" patternDistance="1000" patternSize="5" groundColor="#E2B07B #E2B07B #E2B07B #E2B07B #E2B07B" groundLine="10 11 12 13 14" /><section offset="1260" leftWidth="1" rightWidth="1" gripInside="1" gripOutside="0.6" patternDistance="1000" patternSize="5" groundColor="#804000 #805000 #904000 #804010 #804020" groundLine="0 1 2 3 4" /></ground><scenery><count>25</count><item><w>6</w><h>48</h><src>australian_roadmark.png</src><cl>0</cl><cw>6</cw></item><item><w>30</w><h>70</h><src>left_turn_sign.png</src><cl>13</cl><cw>4</cw></item><item><w>30</w><h>70</h><src>right_turn_sign.png</src><cl>13</cl><cw>4</cw></item><item><w>504</w><h>216</h><src>portique.png</src><cl>0</cl><cw>28</cw></item><item><w>28</w><h>216</h><src>portique_d.png</src><cl>0</cl><cw>28</cw></item><item><w>128</w><h>140</h><src>unfenced_road_sign.png</src><cl>99</cl><cw>4</cw></item><item><w>34</w><h>100</h><src>gravel_road_sign.png</src><cl>15</cl><cw>4</cw></item><item><w>51</w><h>100</h><src>left_reverse_curve_sign.png</src><cl>21</cl><cw>3</cw></item><item><w>51</w><h>100</h><src>right_reverse_curve_sign.png</src><cl>21</cl><cw>3</cw></item><item><w>40</w><h>122</h><src>drive_left_sign.png</src><cl>18</cl><cw>4</cw></item><item><w>80</w><h>32</h><src>herbes1.png</src><cl>3</cl><cw>34</cw></item><item><w>107</w><h>58</h><src>buisson1.png</src><cl>0</cl><cw>107</cw></item><item><w>47</w><h>60</h><src>spinafex1.png</src><cl>18</cl><cw>20</cw></item><item><w>75</w><h>32</h><src>herbes2.png</src><cl>34</cl><cw>36</cw></item><item><w>53</w><h>22</h><src>touffe_herbes.png</src><cl>2</cl><cw>50</cw></item><item><w>88</w><h>45</h><src>rocher1.png</src><cl>0</cl><cw>88</cw></item><item><w>84</w><h>45</h><src>rocher2.png</src><cl>4</cl><cw>80</cw></item><item><w>38</w><h>26</h><src>rocher3.png</src><cl>2</cl><cw>30</cw></item><item><w>478</w><h>328</h><src>applebox.png</src><cl>208</cl><cw>34</cw></item><item><w>107</w><h>140</h><src>arbre2.png</src><cl>39</cl><cw>14</cw></item><item><w>146</w><h>170</h><src>arbre3.png</src><cl>68</cl><cw>18</cw></item><item><w>212</w><h>155</h><src>arbre4.png</src><cl>84</cl><cw>57</cw></item><item><w>114</w><h>129</h><src>arbre5.png</src><cl>32</cl><cw>15</cw></item><item><w>97</w><h>160</h><src>arbre6.png</src><cl>46</cl><cw>12</cw></item><item><w>142</w><h>184</h><src>arbre7.png</src><cl>66</cl><cw>6</cw></item><horizon><w>1720</w><h>520</h><src>bg_australie.png</src></horizon></scenery>',
		/*	3 */'<road>"JiIm30A0K4RKESuC30C0U4R0CSuC30B0K4RoETgL30D0U4RoCTgE30I0P7W01TWD30W011W00iWY30i011L4YW0030o0K1Z8CZwE30o0K1Z8CZwC31B0C4W00TWB31E0C3W00ZmB31T011W00fOX31aFe_HyAW0031fE4yfOAW0031V0U2TgAW0031W0U2W00YMA322B6_W00YgB3297-pW00boB32B011W00fOX32C2MKW00YMB32G2MMSaBW0032D44qMeCNuL32K38ecaCdgE32O6khQEBW0032S042RKLSuD32W0U8RKESuC32a0U8RACTCE32Y0K4R0CSuC32X0K4RoETgC32Z0U4RoCTgE32-011L4YW0033I0K4JWEL4C33K0K4JCCL4C33J0K4J-ELsC33L0K4J-CLsE33i011OqYW0033k011PmXW0033q042dWDfYD33w0K4dqEfOC33y0K4e8CfOD33x0K4ewEgAL33z0K4ewCgAE34E0A2W00YCM34O0e9OCOW0034Q0y6RKNW0034S0y7JWOW003480yDPmMW0034811DW00XuN34D118W00YgN34H14AW00fiM34B1uJW00hkO34F0eAR0EW0034I0yCKmEW0034Q1aPW00caF34i0aIPmIW0034s0OCW00aiI35A011P8XW0035O042byDd-D35U0K4cGEdqC35W0K4caLdqD35V0O4dMEecC35X0K4cuCecE35o084PmDRK035q084PcDRUD35y0U4PmERAC35-0K4PSCReC35z0F4QOES6L35_0F4QECSGE36C011W00caX36G084JWDL4036I084JMDL4D36Q0U4JWEL4C36S0K4JCLL4C36R0U4J-ELsC36T0K4J-CLsE36m011L4YW00377042aiDdCD378042RKDSuD37C0U8RKESuC37G0U8RACTCE37E0K4R0CSuC37D0K4aEEd2C37F0U4aOCcaE37w042YqDaiD37-0U8YqEaiC3820U8YWCc6E3800K4Y-LcGC37_0K4YqEbyC3810U4YgCcQE38K011L4YW0038c011W00ZSJ38s011L4YW0039E0A2W00YCM3940e9OCOW003960y6RKNW003981GBOCOW0038q1QDR0MW0038q1zDW00Y2N38r1aPL4EW0038v118W00Y-M38z1iCW00e8M3961aPW00dqF38t1uJW00hkO39F011L4YW0039i7qfW00b0G39r6GcPSHW0039y7qWW00XuH39_7q_JgGW003A588kS6HW003AE50LW00e8F3AZ011W00fOX3BS011YMKW003Bb011iWXW003Bf0aIPmIW003Bh0OCW00aiI3By1uLKmFW003CE011aiHW003Cu1ueSaFW003D8011TMJYqG3EE011YMKW003Es084JWDL403Eu084JMDL4D3F00U4JWEL4C3F20o4JCLL4C3F10U4J-ELsC3F30K4J-CLsE3F51G9JgEMUH3Gk084PmDRK03Gm084PcDRUD3Gu0U4PmERAC3Gw0K4PSCReC3Gv0F4QOES6L3Gx0F4QECSGE3Hu042dWDfYD3H-0K4dqEfOC3I00K4e8CfOD3H_0K4ewEgAC3I10K4ewLgAE3I30K4dCHhGD3IS084PmDRK03IU084PcLRUD3Ic0U4PmERAC3Ie0K4PSCReC3Id0P4QOES6L3If0F4QECSGE105FaY020220A4i0V2o10P3ST020210S2cT020210Z5gd020220d2o0Y2o10g3SU020210oKSR020220t4i2U3o11B2aY020211LKYc020241Q2_b11kAUU020221p4o1Y3v11z5OS02021275cZ020222C5v2N3v12MAQP020212b8cb020222l2o2Q4v12q5aZ020212v5ST02022332o2S4v138ASS020213IASS020213cKQM020223c2o2a4v14EFkg02A344T3_N44d4_U14iFIM02A444s4vJ4556_U15AFch02A315R5kf020245U6vE15y5OQ020245_3vb1615aY020216E7Ya020246G3_Z16O7UT020226a3o1a5v16kASQ020216uAYd02022723o1b5v17C2aY020217K2SU020217R3SU020217W5cb020227g3o1T5v17i8ea020217vCSR020227-4o0f5v1882SU020218D5aY020218IFUT020228S4o0R4v18XAYY020218m2aY020218r2SU020248r5iU18wKQL02A449E4_G49W2_V19Y5sk023349d1_U49g3_V19i5CK023449s5_K4AC4_U1AKCmg02024AU4_G4Ac3_V1AoAGN02024Ay8_81BB3id02021BE3KP02024BY5_T1BaKYl02834Ba8_64CA2_U1CC5EL02022CJ4v6f5v1Cq3kd02021Ct6IP02022Cv5v2R5v1Cz3kd02022DI4v3b2v1DSAgd02021DmAON02022DhAv3K4v1EY5UR02021Ed5Yb02022Ei6v2G8v1FAAEL02542FU2v0a4v1GBAed02022GQ3v0Y1v1Ga5ce02021Gf5QO02022GiAv2P3v1Gk5kh02022H2AJ2Z3_1HMKQM02022Hq4_2Z4_1H-5kh02024I88_P1IS5IU02021IX2aY02021IZ3IP0202344011W00Xk734Y011W00Xk839O011W00Xk93BQ011W00Xk73FA051V44W0030o0U2UI2Xk233m011W00Y2P33_011aOQbUS34T011W00ZIT34u011W00XuV35P011TMUW0035y011W00Y-W36L011UcTW0036m011W00XuV37H011UmTW0037l011W00ZIW38D011TqUW0038j011W00XkV399011TCUW0039d011W00XuW39x011T-TW003AP011W00Z8W3At011TqUW003BL011W00YgV3C2011UIUW003CR011Z8RcGS304011W05Xi63Im011W05Xi6"</road><ground count="12"><section offset="0" leftWidth="1" rightWidth="1" gripInside="0.7" gripOutside="0.6" patternDistance="1000" patternSize="5" groundColor="#BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0" groundLine="0 1 2 3 4" /><section offset="260" leftWidth="1" rightWidth="1" gripInside="0.8" gripOutside="0.6" patternDistance="2500" patternSize="5" groundColor="#BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0" groundLine="5 6 7 8 9" /><section offset="265" leftWidth="1" rightWidth="1" gripInside="0.9" gripOutside="0.6" patternDistance="1000" patternSize="5" groundColor="#BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0" groundLine="10 11 12 13 14" /><section offset="388" leftWidth="1" rightWidth="1" gripInside="0.9" gripOutside="0.6" patternDistance="2000" patternSize="10" groundColor="#BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0" groundLine="15 16 17 18 19 20 21 22 23 24" /><section offset="392" leftWidth="0.5" rightWidth="1" gripInside="0.9" gripOutside="0.4" patternDistance="1000" patternSize="5" groundColor="#BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0" groundLine="25 26 27 28 29" /><section offset="560" leftWidth="1" rightWidth="1" gripInside="0.9" gripOutside="0.6" patternDistance="2000" patternSize="10" groundColor="#BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0" groundLine="24 23 22 21 20 19 18 17 16 15" /><section offset="564" leftWidth="1" rightWidth="1" gripInside="0.9" gripOutside="0.6" patternDistance="1000" patternSize="5" groundColor="#BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0" groundLine="10 11 12 13 14" /><section offset="796" leftWidth="1" rightWidth="1" gripInside="0.9" gripOutside="0.4" patternDistance="2000" patternSize="10" groundColor="#BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0" groundLine="15 16 17 18 19 20 21 22 23 24" /><section offset="800" leftWidth="0.5" rightWidth="1" gripInside="0.9" gripOutside="0.4" patternDistance="1000" patternSize="5" groundColor="#BBC0D0 #CED3E5 #CED3E5 #BBC0D0 #BBC0D0" groundLine="25 26 27 28 29" /><section offset="1060" leftWidth="1" rightWidth="1" gripInside="0.9" gripOutside="0.4" patternDistance="2000" patternSize="10" groundColor="#BBC0D0 #BBC0D0 #CED3E5 #CED3E5 #CED3E5 #CED3E5 #BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0" groundLine="24 23 22 21 20 19 18 17 16 15" /><section offset="1064" leftWidth="1" rightWidth="1" gripInside="0.9" gripOutside="0.4" patternDistance="1000" patternSize="5" groundColor="#BBC0D0 #CED3E5 #CED3E5 #BBC0D0 #BBC0D0" groundLine="10 11 12 13 14" /><section offset="2520" leftWidth="1" rightWidth="1" gripInside="0.9" gripOutside="0.4" patternDistance="1000" patternSize="5" groundColor="#BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0 #BBC0D0" groundLine="10 11 12 13 14" /></ground><weather count="7"><area offset="0" image="" color="#BBC0D0" opacity="0" /><area offset="1000" image="" color="#BBC0D0" opacity="0.2" /><area offset="1050" image="" color="#BBC0D0" opacity="0.4" /><area offset="1080" image="" color="#BBC0D0" opacity="0.15" /><area offset="1100" image="" color="#BBC0D0" opacity="0.5" /><area offset="1120" image="" color="#BBC0D0" opacity="0.3" /><area offset="1140" image="" color="#BBC0D0" opacity="0.4" /></weather><scenery><count>33</count><item><w>14</w><h>48</h><src>picket.png</src><cl>1</cl><cw>9</cw></item><item><w>30</w><h>72</h><src>left_turn_sign.png</src><cl>14</cl><cw>2</cw></item><item><w>30</w><h>72</h><src>right_turn_sign.png</src><cl>14</cl><cw>2</cw></item><item><w>504</w><h>216</h><src>portique.png</src><cl>0</cl><cw>28</cw></item><item><w>28</w><h>216</h><src>portique_d.png</src><cl>0</cl><cw>28</cw></item><item><w>43</w><h>80</h><src>left_warning_sign.png</src><cl>20</cl><cw>3</cw></item><item><w>43</w><h>80</h><src>right_warning_sign.png</src><cl>20</cl><cw>3</cw></item><item><w>43</w><h>80</h><src>left_s_curve_sign.png</src><cl>20</cl><cw>3</cw></item><item><w>249</w><h>320</h><src>snow_tree_1.png</src><cl>118</cl><cw>12</cw></item><item><w>95</w><h>96</h><src>snow_tree_2.png</src><cl>19</cl><cw>72</cw></item><item><w>132</w><h>240</h><src>snow_tree_3.png</src><cl>59</cl><cw>9</cw></item><item><w>62</w><h>70</h><src>snow_tree_4.png</src><cl>8</cl><cw>50</cw></item><item><w>112</w><h>240</h><src>snow_tree_5.png</src><cl>16</cl><cw>95</cw></item><item><w>70</w><h>120</h><src>tree_1.png</src><cl>4</cl><cw>58</cw></item><item><w>97</w><h>128</h><src>tree_2.png</src><cl>28</cl><cw>68</cw></item><item><w>89</w><h>80</h><src>tree_3.png</src><cl>19</cl><cw>55</cw></item><item><w>129</w><h>40</h><src>ice_block.png</src><cl>0</cl><cw>129</cw></item><item><w>486</w><h>50</h><src>barriere_transversale.png</src><cl>0</cl><cw>486</cw></item><item><w>71</w><h>45</h><src>cailloux_1.png</src><cl>0</cl><cw>71</cw></item><item><w>96</w><h>139</h><src>snow_tree_6.png</src><cl>0</cl><cw>87</cw></item><item><w>66</w><h>44</h><src>caillou_1.png</src><cl>0</cl><cw>63</cw></item><item><w>69</w><h>30</h><src>caillou_2.png</src><cl>0</cl><cw>69</cw></item><item><w>74</w><h>64</h><src>caillou_3.png</src><cl>0</cl><cw>74</cw></item><item><w>95</w><h>59</h><src>superg_sign.png</src><cl>1</cl><cw>92</cw></item><item><w>320</w><h>131</h><src>superg_start.png</src><cl>43</cl><cw>62</cw><door>1</door><line>1</line></item><item><w>224</w><h>131</h><src>superg_finish.png</src><cl>0</cl><cw>8</cw><door>1</door><line>-1</line></item><item><w>8</w><h>1</h><src>superg_d.png</src><cl>0</cl><cw>8</cw><door>-1</door></item><item><w>40</w><h>99</h><src>red_door_1.png</src><cl>0</cl><cw>40</cw><door>-1</door></item><item><w>39</w><h>100</h><src>red_door_2.png</src><cl>0</cl><cw>39</cw><door>-1</door></item><item><w>43</w><h>100</h><src>blue_door_1.png</src><cl>0</cl><cw>43</cw><door>1</door></item><item><w>42</w><h>96</h><src>blue_door_2.png</src><cl>0</cl><cw>42</cw><door>1</door></item><item><w>1200</w><h>360</h><src>hill1.png</src><cl>0</cl><cw>1195</cw></item><item><w>890</w><h>273</h><src>hill2.png</src><cl>0</cl><cw>890</cw></item><horizon><w>1720</w><h>520</h><src>bg_alpes_16_centered.png</src></horizon></scenery>',
		/*  4 */'<road>"HqHC1063aa020210A4YY020210E2UV020210H3aY020210L5Sb020210U4cb020210b4aZ020210f2SU020210j3SU020210o2aY020210t8KN184420y3_1b2_2164_KZ8_21L4_0Q4_21k2_2a8_21u8_2R4_22W3_6a2_22bA_Ab3_22g4_4X2_22-6_0d8_2334_DU4_23S6_4Z5_2442_1Y3_24E3_0Y3_11BAab020211OCaR020211VFWY020211uFWT0202122FWa020212CFWR020212MFWc020212qAac0202133KYa020213DFUS020213NAYa020213c5ic020213rAYY0202144KeX020214Y5MP024414sUXX020215PAYY020215b8YS020215j8Ya020215y5cb02021658Wc174326G8_2dC_26QA_KiC_26u4_2T6_27C3_2V3_27l6_8e8_27q8o4e6o2886_8c8_28m8_4R6_29O2_0b2_29s8_4Y3_29x4_0b5_2AK8_0V6_2Ae4_2Y6_2B62_2Y2_16ECaa020216S8Wb020216W8WR020216a8Wb020216pUWZ020217HKWS020217bAWb020217o6ac020217-UeS020218cAad020218jAUZ0202193PUe020219NAIP020219dCQN020219qAca02021A0KaQ02021AKCiQ02021AeUaX02021Am6aa02021Ax6SS02021BGPad02021BjFaa02021Bu8WQ02021C2KWd02021CRAWT02021Cd5Zb02021CqATS02021CyEWc02021DIKWM02441DhASS02022DmC_Ki8_2E44_8S6_2Ei4_4c4_2En2_2V2_2FK3_1a3_2Fo3_0b3_2G62_0X2_2Ga3_0b3_2Gu1_2X2_1Dr6SQ02021Dz6ad02021E56SR02021ED6aa02021EL6SS02021EZCWb02021EiKWT02021EqKWZ02021F0Cgc02021FUPkY02021FeKOZ02021G1FOa02021G6UWf02531GVKWU02021GuUaa020239G011UI9W003Ci011UI9W00304011W05Xi63HC011W05Xi631a011TgWYMW338011TgXYMX34i011TgYYMY36G011TgZYMZ37q011TgaYMa39O011TgbYMb3Ay011TgcYMc3CW011TgdYMd3E4011TgeYMe3Fe011TgfYMf"</road><ground count="7"><section offset="0" leftWidth="1" rightWidth="1" gripInside="0.7" gripOutside="0.6" patternDistance="1000" patternSize="10" groundColor="#80C020 #80C020 #80C020 #80C020 #80C020 #80C020 #80C020 #80C020 #80C020 #80C020" groundLine="0 1 2 3 4 5 6 7 8 9" groundRight="9 3 2.5 1.5 1.6 1.9 4 9" /><section offset="60" leftWidth="1" rightWidth="1" gripInside="0.7" gripOutside="0.6" patternDistance="1000" patternSize="10" groundColor="#80C020 #80C020 #80C020 #80C020 #80C020 #80C020 #80C020 #80C020 #80C020 #80C020" groundLine="0 1 2 3 4 5 6 7 8 9" /><section offset="300" leftWidth="1" rightWidth="1" gripInside="0.7" gripOutside="0.6" patternDistance="1000" patternSize="10" groundColor="#80C020 #80C020 #80C020 #80C020 #80C020 #80C020 #80C020 #80C020 #80C020 #80C020" groundLine="0 1 2 3 4 5 6 7 8 9" groundLeft="-9 -3 -2 -1.1 -1.2 -1.9 -4 -9" /><section offset="400" leftWidth="1" rightWidth="1" gripInside="0.7" gripOutside="0.6" patternDistance="1000" patternSize="10" groundColor="#80C020 #80C020 #80C020 #80C020 #80C020 #80C020 #80C020 #80C020 #80C020 #80C020" groundLine="0 1 2 3 4 5 6 7 8 9" /><section offset="740" leftWidth="1" rightWidth="1" gripInside="0.7" gripOutside="0.6" patternDistance="1000" patternSize="10" groundColor="#80C020 #80C020 #80C020 #80C020 #80C020 #80C020 #80C020 #80C020 #80C020 #80C020" groundLine="0 1 2 3 4 5 6 7 8 9" groundLeft="-9 -3 -2.5 -1.5 -1.6 -1.9 -4 -9 -40 -40 -40 -40 -40" groundRight="40 40 40 40 40 9 3 2.5 1.5 1.6 1.9 4 9" /><section offset="880" leftWidth="1" rightWidth="1" gripInside="0.7" gripOutside="0.6" patternDistance="1000" patternSize="10" groundColor="#80C020 #80C020 #80C020 #80C020 #80C020 #80C020 #80C020 #80C020 #80C020 #80C020" groundLine="0 1 2 3 4 5 6 7 8 9" /><section offset="2280" leftWidth="1" rightWidth="1" gripInside="0.9" gripOutside="0.4" patternDistance="1000" patternSize="10" groundColor="#80C020 #80C020 #80C020 #80C020 #80C020 #80C020 #80C020 #80C020 #80C020 #80C020" groundLine="0 1 2 3 4 5 6 7 8 9" /></ground><weather count="1"><area offset="0" image="" color="#BBC0D0" opacity="0" /></weather><scenery><count>41</count><item><w>14</w><h>48</h><src>picket.png</src><cl>1</cl><cw>9</cw></item><item><w>40</w><h>80</h><src>left_turn_sign.png</src><cl>18</cl><cw>4</cw></item><item><w>40</w><h>80</h><src>right_turn_sign.png</src><cl>19</cl><cw>4</cw></item><item><w>504</w><h>216</h><src>portique.png</src><cl>0</cl><cw>28</cw></item><item><w>28</w><h>216</h><src>portique_d.png</src><cl>0</cl><cw>28</cw></item><item><w>40</w><h>82</h><src>left_warning_sign.png</src><cl>18</cl><cw>4</cw></item><item><w>40</w><h>82</h><src>right_warning_sign.png</src><cl>18</cl><cw>4</cw></item><item><w>40</w><h>105</h><src>s_curve_sign.png</src><cl>18</cl><cw>4</cw></item><item><w>100</w><h>50</h><src>100.png</src><cl>0</cl><cw>0</cw></item><item><w>100</w><h>50</h><src>100.png</src><cl>0</cl><cw>0</cw></item><item><w>100</w><h>50</h><src>100.png</src><cl>0</cl><cw>0</cw></item><item><w>100</w><h>50</h><src>100.png</src><cl>0</cl><cw>0</cw></item><item><w>100</w><h>50</h><src>100.png</src><cl>0</cl><cw>0</cw></item><item><w>100</w><h>50</h><src>100.png</src><cl>0</cl><cw>0</cw></item><item><w>100</w><h>50</h><src>100.png</src><cl>0</cl><cw>0</cw></item><item><w>100</w><h>50</h><src>100.png</src><cl>0</cl><cw>0</cw></item><item><w>100</w><h>50</h><src>100.png</src><cl>0</cl><cw>0</cw></item><item><w>100</w><h>50</h><src>100.png</src><cl>0</cl><cw>0</cw></item><item><w>100</w><h>50</h><src>100.png</src><cl>0</cl><cw>0</cw></item><item><w>100</w><h>50</h><src>100.png</src><cl>0</cl><cw>0</cw></item><item><w>100</w><h>50</h><src>100.png</src><cl>0</cl><cw>0</cw></item><item><w>100</w><h>50</h><src>100.png</src><cl>0</cl><cw>0</cw></item><item><w>100</w><h>50</h><src>100.png</src><cl>0</cl><cw>0</cw></item><item><w>100</w><h>50</h><src>100.png</src><cl>0</cl><cw>0</cw></item><item><w>100</w><h>50</h><src>100.png</src><cl>0</cl><cw>0</cw></item><item><w>100</w><h>50</h><src>100.png</src><cl>0</cl><cw>0</cw></item><item><w>100</w><h>50</h><src>100.png</src><cl>0</cl><cw>0</cw></item><item><w>100</w><h>50</h><src>100.png</src><cl>0</cl><cw>0</cw></item><item><w>100</w><h>50</h><src>100.png</src><cl>0</cl><cw>0</cw></item><item><w>100</w><h>50</h><src>100.png</src><cl>0</cl><cw>0</cw></item><item><w>100</w><h>50</h><src>100.png</src><cl>0</cl><cw>0</cw></item><item><w>100</w><h>50</h><src>200.png</src><cl>0</cl><cw>0</cw></item><item><w>100</w><h>50</h><src>300.png</src><cl>0</cl><cw>0</cw></item><item><w>100</w><h>50</h><src>400.png</src><cl>0</cl><cw>0</cw></item><item><w>100</w><h>50</h><src>500.png</src><cl>0</cl><cw>0</cw></item><item><w>100</w><h>50</h><src>600.png</src><cl>0</cl><cw>0</cw></item><item><w>100</w><h>50</h><src>700.png</src><cl>0</cl><cw>0</cw></item><item><w>100</w><h>50</h><src>800.png</src><cl>0</cl><cw>0</cw></item><item><w>100</w><h>50</h><src>900.png</src><cl>0</cl><cw>0</cw></item><item><w>100</w><h>50</h><src>1000.png</src><cl>0</cl><cw>0</cw></item><item><w>100</w><h>50</h><src>1100.png</src><cl>0</cl><cw>0</cw></item><horizon><w>1720</w><h>520</h><src>bg_boussole.png</src></horizon></scenery>',
				'<road>"Ff0K10yKGO008211aKai00C112MUOS005212-KmU0081144Aqi0051149AST005124s4_4O4_16GeeQ00002724_3b2_27W3_1c4_27v4V3f4V2884V0O4V28c4_1Q4_29E4_1g4_19nPgc00A11AC8ST00421AK5aZ00411AeKqX00812Ao4_0Q4_1BBPMO00A22Bk4_0e4_1BpPMV00A21CCFkb00F11CRFKQ00F22CW4_4J4_1Cl5QT00521CqAui00A11C-5QT00521DNAQT00A22DS2_0Y2_1DXAuY00A11Dh5QT00A22Dh3_0Z2_1DrAQT00A21E2CQT00A22E43_0b3_1EOFMR00A22EY3_0c3_1Ed5gb00A11Ei5MR00A21En5gb00A11EsAMR00A22F03_0e3_1FAFci00A11FP5QT00A230e0y1Sa2ZS2304011W06Z9730O011W00aO830m011W00aO9318011ReAaOA3Fe011W06Z97"</road><ground count="2"><section offset="0" leftWidth="1" rightWidth="1" gripInside="1" gripOutside="0.5" patternDistance="1000" patternSize="5" groundColor="#45708A #45708A #45708A #3B5061 #3B5061" groundLine="0 0 0 1 1" /><section offset="1200" leftWidth="1" rightWidth="1" gripInside="1" gripOutside="0.5" patternDistance="1000" patternSize="5" groundColor="#658A70 #658A70 #658A70 #3B6150 #3B6150" groundLine="0 0 0 1 2" /></ground><scenery><count>11</count><item><w>8</w><h>40</h><src>object_test_roadsign.png</src><cl>0</cl><cw>8</cw></item><item><w>78</w><h>40</h><src>lotus_panneau_virage_gauche.gif</src><cl>0</cl><cw>78</cw></item><item><w>78</w><h>40</h><src>lotus_panneau_virage_droite.gif</src><cl>0</cl><cw>78</cw></item><item><w>32</w><h>64</h><src>speed_limit_110.gif</src><cl>12</cl><cw>8</cw></item><item><w>464</w><h>180</h><src>portique.png</src><cl>0</cl><cw>12</cw></item><item><w>12</w><h>180</h><src>portique_d.png</src><cl>0</cl><cw>12</cw></item><item><w>140</w><h>139</h><src>acacia_small.png</src><cl>71</cl><cw>5</cw></item><item><w>213</w><h>180</h><src>arbre_printemps_small.png</src><cl>62</cl><cw>16</cw></item><item><w>198</w><h>160</h><src>arbre_vert_small.png</src><cl>83</cl><cw>15</cw></item><item><w>191</w><h>220</h><src>chataignier.gif</src><cl>83</cl><cw>15</cw></item><item><w>215</w><h>180</h><src>noisetiers.gif</src><cl>74</cl><cw>87</cw></item><horizon><w>1280</w><h>470</h><src>bg_cevennes.png</src></horizon></scenery>',
				'<road>"FfFe24s4_4O4_2724_3b2_27W3_1c4_27v4V3f4V2884V0O4V28c4_1Q4_29E4_1g4_2Ao4_0Q4_2Bk4_0e4_2CW4_4J4_2DS2_0Y2_2Dh3_0Z2_2E43_0b3_2EY3_0c3_2F03_0e3_30e0y1Sa2ZS2304011W06Z9730O011W00aO830m011W00aO9318011ReAaOA3Fe011W06Z97"</road><ground count="1"><section offset="0" leftWidth="1" rightWidth="1" gripInside="1" gripOutside="0.5" patternDistance="1000" patternSize="5" groundColor="#658A70 #658A70 #658A70 #3B6150 #3B6150" groundLine="0 0 0 1 1" /></ground><scenery><count>21</count><item><w>8</w><h>40</h><src>object_test_roadsign.png</src><cl>0</cl><cw>8</cw></item><item><w>78</w><h>40</h><src>panneau_virage_gauche.gif</src><cl>0</cl><cw>78</cw></item><item><w>78</w><h>40</h><src>panneau_virage_droite.gif</src><cl>0</cl><cw>78</cw></item><item><w>464</w><h>180</h><src>portique.png</src><cl>0</cl><cw>12</cw></item><item><w>12</w><h>180</h><src>portique_d.png</src><cl>0</cl><cw>12</cw></item><item><w>140</w><h>139</h><src>acacia_small.png</src><cl>71</cl><cw>5</cw></item><item><w>191</w><h>220</h><src>chataignier.png</src><cl>83</cl><cw>15</cw></item><item><w>97</w><h>96</h><src>Arbutus_unedo_g1_16.png</src><cl>51</cl><cw>5</cw></item><item><w>54</w><h>40</h><src>calluna_vulgaris_02.png</src><cl>0</cl><cw>54</cw></item><item><w>67</w><h>72</h><src>Erica_arborea.png</src><cl>0</cl><cw>67</cw></item><item><w>53</w><h>56</h><src>Cistus_monspeliensis_Magliano.png</src><cl>4</cl><cw>39</cw></item><item><w>176</w><h>64</h><src>genets.png</src><cl>9</cl><cw>161</cw></item><item><w>128</w><h>45</h><src>caillou_1.png</src><cl>7</cl><cw>115</cw></item><item><w>53</w><h>30</h><src>buisson_1.png</src><cl>4</cl><cw>44</cw></item><item><w>81</w><h>110</h><src>conifere_ile.png</src><cl>41</cl><cw>7</cw></item><item><w>103</w><h>88</h><src>joncs3.png</src><cl>4</cl><cw>98</cw></item><item><w>83</w><h>48</h><src>tas_pierres.png</src><cl>3</cl><cw>80</cw></item><item><w>100</w><h>120</h><src>conifere_gorges_tarn.png</src><cl>43</cl><cw>10</cw></item><item><w>181</w><h>160</h><src>rochers_en_equilibre.png</src><cl>0</cl><cw>181</cw></item><item><w>57</w><h>27</h><src>caillou_2.png</src><cl>1</cl><cw>55</cw></item><item><w>70</w><h>64</h><src>joncs4.png</src><cl>6</cl><cw>63</cw></item><horizon><w>1280</w><h>520</h><src>bg_cevennes.png</src></horizon></scenery>'
		);
		
		$lapRecordsXML = $this->getLapRecordsAsXML($dbHelper, $level, $mode);
		
		$playerLapRecordXML = $this->getPlayerLapRecordAsXML($dbHelper, $userId, $level, $mode);

		$qualifyingTime = $this->getQualificationTime($level);
		$qualifyingTimeXML = '<qualifyingTime>'.$qualifyingTime.'</qualifyingTime>';
		
		$description = '<track>';
		
		if ($level >= 0 && $level < count($roadAndScenery)) {
			$description .= $roadAndScenery[$level];
		}
		$description .= '<lapRecords>'.$lapRecordsXML.$playerLapRecordXML.$qualifyingTimeXML.'</lapRecords></track>';
		return $description;
	}
 
	/**
	 * Renvoie la description XML des 5 meilleurs temps du couple circuit/mode de jeu demands.
	 * Cette information est visible de tous les joueurs et n'est pas soumise  contrle d'accs
	 *
	 * Le modle d'un record est <lapTime><name>STRING</name><car>INT</car><time>INT</time></lapTime>
	 */
	public function getLapRecordsAsXML($dbHelper, $raceIndex, $modeIndex)
	{
		$lapRecords = $dbHelper->getBestLapTimes($raceIndex, $modeIndex, 5);
		$lapRecordsXML = '';
		foreach ($lapRecords as $singleRecord) {
			$lapRecordsXML.= '<lapTime><name>'.$singleRecord['NAME'].'</name><car>'.$singleRecord['CAR'].'</car><time>'.$singleRecord['TIME'].'</time></lapTime>';
		}
		return $lapRecordsXML;
	}
 
	/**
	 * Renvoie la description XML des du meilleur temps du joueur
	 * Cette information est visible de tous les joueurs et n'est pas soumise  contrle d'accs
	 *
	 * Le modle d'un record est <playerTime><name>STRING</name><car>INT</car><time>INT</time></playerTime>
	 */
	public function getPlayerLapRecordAsXML($dbHelper, $userId, $raceIndex, $modeIndex)
	{
		$dbRecord = $dbHelper->getPlayerLapRecord($userId, $raceIndex, $modeIndex);
		$playerLapRecordXML = '<playerTime><name>'.$dbRecord['NAME'].'</name><car>'.$dbRecord['CAR'].'</car><time>'.$dbRecord['TIME'].'</time></playerTime>';
		return $playerLapRecordXML;
	}
 
	/**
	 * Renvoie le temps de qualification pour un circuit ($roadIndex dbutant  1)
	 * en 1/100e de seconde
	 */
	public function getQualificationTime($roadIndex)
	{
		$qualificationTime = array ( 0, // dfaut pour index inconnu
			10000,	// 1 mn 40 s
			6500,	// 1 mn 5 s
			11500,  // 1 mn 55 s
			10200,	// 1 mn 42 s
			1000,	// 10 s (zone de test)
			5000	// 50 s
		);
		if ($roadIndex<0 || $roadIndex >= count($qualificationTime)) {
			return 0;
		}
		return $qualificationTime[$roadIndex];
	}
 
	/**
	 * Renvoie la longueur de la course en cm (unit utilise pour la simulation).
	 * Celle-ci est mesure depuis le point zro (pas la ligne de dpart) jusqu'
	 * la ligne d'arrive au dernier tour.
	 *
	 * Ainsi, un circuit de 3 km dont on fait 4 tours renverra 1 200 000.
	 *
	 * Le paramtre $roadIndex dbut  1 (1-based) pour le premier circuit
	 */
	public function getRaceLength($roadIndex)
	{
		$raceLength = array ( 4700, // dfaut pour index inconnu
			4700,	// 5 km
			300000, // 3 km
			600000, // 6 km
			550000, // 5.5 km
			10000,	// 100 m (zone de test)
			500000 // 5 km
		);
		if ($roadIndex<0 || $roadIndex >= count($raceLength)) {
			return 0;
		}
		return $raceLength[$roadIndex];
	}
 
	/**
	 * Renvoie le nom du circuit correspondant, ou une chane vide
	 * si l'index demand n'existe pas.
	 * Le paramtre $roadIndex dbut  1 (1-based) pour le premier circuit
	 */
	public function getRaceName($roadIndex)
	{
		$raceName = array ( '',
			'Grands Causses',
			'Territoire du Nord',
			'Alpes Suisses',
			'Connemara',
			'Test Zone',
			'Ligne droite',
			'Track 7',
			'Track 8',
			'Track 9',
			'Track 10',
			'Track 11',
			'Track 12',
			'Track 13',
			'Track 14',
			'Track 15',
			'Track 16'
		 );
		if ($roadIndex<0 || $roadIndex >= count($raceName)) {
			return '';
		}
		return $raceName[$roadIndex];
	}
	 
 	/**
	 * Enregistre un incident dans la table approprie de la BDD.
	 * Cette fonction a t factorise dans CircuitBroker car l'enregistrement
	 * inclut l'ensemble des variables membres de la classe.
	 */
	public function recordIncident($dbHelper, $status, $errorMessage)
	{
		$dbHelper->recordIncident($this->validatedUserId, $this->validatedUserName, $status, $errorMessage, $this->validatedLevel, $this->validatedMode, $this->validatedCar, 0, $this->validatedGhostName);
	}

 
}

?>