#include "generator.h"
#include <cmath>
#include <ncurses.h>


void DisplayFrame(WINDOW* window, const Frame& frame)
{
  for (size_t i = 0, offset = 0; i < frame.height; ++i) {
    for (size_t j = 0; j < frame.width; ++j, ++offset) {
      const auto& cell = frame.data[offset];
      if (cell.covergenceStep > 100) {
        mvwaddch(window, i, j, '#');
      } else if (cell.covergenceStep > 80) {
        mvwaddch(window, i, j, '*');
      } else if (cell.covergenceStep > 40) {
        mvwaddch(window, i, j, '+');
      } else {
        mvwaddch(window, i, j, ' ');
      }
    }
  }
}

void ClearWindow(WINDOW* window)
{
  for (size_t i = 0; i < LINES; ++i) {
    for (size_t j = 0; j < COLS; ++j) {
      mvwaddch(window, i, j, ' ');
    }
  }
}


double factor(size_t begin, size_t end, size_t index)
{
  return static_cast<double>(end - begin) / (index - begin);
}


struct View {
  double viewX;
  double viewY;
  double viewWidth;
  double viewHeight;
  double alpha;
  double beta;
};


bool Stage(View* view, size_t move_length, size_t stage_length, double newViewX, double newViewY, double dAlpha, double dBeta)
{
  const double oldViewX = view->viewX;
  const double oldViewY = view->viewY;
  
  const double dViewX = (newViewX - oldViewX) / move_length;
  const double dViewY = (newViewY - oldViewY) / move_length;
  
  for (size_t step = 0; step < move_length; ++step) {
    if (getch() == KEY_F(1)) {
      return false;
    }
    const auto cos_beta = std::cos(view->beta);
    DisplayFrame(stdscr, GenerateFrame(COLS, LINES, 
                                       view->viewX, view->viewY, 
                                       cos_beta * view->viewWidth, cos_beta * view->viewHeight, 
                                       view->alpha));
    view->viewX += dViewX;
    view->viewY += dViewY;
    view->alpha += dAlpha;
    view->beta += dBeta;
    mvprintw(LINES-2, 0,"viewX: %.2f  viewY: %.2f", view->viewX, view->viewY);
  }

  for (size_t step = 0; step < stage_length; ++step) {
    if (getch() == KEY_F(1)) {
      return false;
    }
    const auto cos_beta = std::cos(view->beta);
    DisplayFrame(stdscr, GenerateFrame(COLS, LINES, 
                                       view->viewX, view->viewY, 
                                       cos_beta * view->viewWidth, cos_beta * view->viewHeight, 
                                       view->alpha));
    view->alpha += dAlpha;
    view->beta += dBeta;
    mvprintw(LINES-2, 0,"viewX: %.2f  viewY: %.2f", view->viewX, view->viewY);
  }
  return true;
}


int main()
{
  initscr();
  if(!has_colors()) {
    endwin();
    printf("Your terminal does not support color\n");
    return -1;
  }

  cbreak();
  timeout(20);
  keypad(stdscr, TRUE);

  /* Stage 1 */
  View view;
  view.viewX = -0.5;
  view.viewY = 0.0;
  view.viewWidth = 3.0;
  view.viewHeight = 2.0;
  view.alpha = 0.0;
  view.beta = 0.0;

  bool ok =
    Stage(&view, 300, 200, -0.75, 0.1, 0.02, 0.005) &&
    Stage(&view, 140,   0,  -0.1, 0.8, 0.01, 0.004) &&
    Stage(&view, 0,  600,  -0.1, 0.8, 0.01, 0.005) &&
    Stage(&view, 200, 200, -0.75, 0.1, 0.002, 0.005) &&
    Stage(&view, 0,  200,  -0.0, 0.9, 0.0025, 0.03) &&
//    Stage(&view, 40,  60,  -0.5, 0.0, 0.02, 0.04);
    true;


  ClearWindow(stdscr);
  
  mvprintw(LINES/2, COLS/2 - 5, "THE END.");

  if (ok) {  
    while (getch() != KEY_F(1));
  }


  endwin();/* End curses mode  */

  return 0;
}
