#pragma once

#include <cstdint>
#include <cstddef>
#include <vector>
#include <complex>

struct Cell {
  int32_t covergenceStep;
};

struct Frame {
  size_t width;
  size_t height;

  double firstX;
  double lastX;
  double firstY;
  double lastY;
  
  std::vector<Cell> data;
};


Frame GenerateFrame(size_t width, size_t height, double firstX, double lastX, double firstY, double lastY) {
  Frame result;
  result.width = width;
  result.height = height;
  result.firstX = firstX;
  result.lastX = lastX;
  result.firstY = firstY;
  result.lastY = lastY;
  result.data.resize(height * width);

  const double dx = (lastX - firstX) / width;
  const double dy = (lastY - firstY) / height;

  for (size_t i = 0, offset = 0; i < height; ++i) {
    const double y0 = firstY + i*dy;
    for (size_t j = 0; j < width; ++j, ++offset) {
      const double x0 = firstX + j*dx;

      int32_t iteration = 0;
      double x = x0;
      double y = y0;

      for (; x*x + y*y < 4.0 && iteration < 200; ++iteration) {
        const auto t = x*x - y*y + x0;
        y = 2*x*y + y0;
        x = t;
      }

      result.data[offset].covergenceStep = iteration;
    }
  }

  return result;
}


Cell GenerateCell(double x0, double y0)
{
  int32_t iteration = 0;
  double x = x0;
  double y = y0;
  
  for (; x*x + y*y < 4.0 && iteration < 105; ++iteration) {
    const auto t = x*x - y*y + x0;
    y = 2*x*y + y0;
    x = t;
  }

  Cell result;
  result.covergenceStep = iteration;
  return result;
}


Frame GenerateFrame(size_t width, size_t height, double viewX, double viewY, double viewWidth, double viewHeight, double alpha)
{
  Frame result;
  result.width = width;
  result.height = height;
  result.data.reserve(height * width);

  const auto cos_alpha = std::cos(alpha);
  const auto sin_alpha = std::sin(alpha);

  const double dx = viewWidth / width;
  const double dy = viewHeight / height;

  const int lastI = height / 2;
  const int firstI = lastI - height;

  const int lastJ = width / 2;
  const int firstJ = lastJ - width;

  for (int i = firstI; i < lastI; ++i) {
    for (int j = firstJ; j < lastJ; ++j) {
      const double x0 = viewX + cos_alpha * dx * j + sin_alpha * dy * i;
      const double y0 = viewY - sin_alpha * dx * j + cos_alpha * dy * i;
      result.data.push_back( GenerateCell(x0, y0) );
    }
  }

  return result;
}
