#ifndef _GAMESTATE_H_
#define _GAMESTATE_H_

//#include <GL/glew.h>
#include "common.h"
#include "util.h"
#include "vecmath.h"
//#include "vec2.h"
#include <iostream>
#include <vector>


#include "RenderWindow.h"
#include "Graphics.h"

#include <fstream>


bool getLowestRoot(float a, float b, float c, float maxR, float* root);
void drawLine(float px, float py, float w, float h);

using namespace std;

class Cloud {
public:
    vec2 pos;
    vec2 vel;
    //float px, py;
    //float vx, vy;
    float vapor;
    string text;
    vec2 new_vel;

    Cloud(float px, float py, float vx, float vy, float vapor)
    : pos(px, py), vel(vx, vy), vapor(vapor) { }

    Cloud(vec2 p, vec2 v, float vapor)
    : pos(p), vel(v), vapor(vapor) { }

    Cloud() {}

    float getRadius() const {
        return sqrt(vapor);
    }

    void addVelocity(float vx, float vy) {
        this->vel.x += vx;
        this->vel.y += vy;
    }
};

class ThunderStorm : public Cloud {
public:
    ThunderStorm(float px, float py, float vx, float vy, float vapor)
    : Cloud(px, py, vx, vy, vapor) {
    }
    ThunderStorm(vec2 p, vec2 v, float vapor)
    : Cloud(p, v, vapor) { }
    ThunderStorm() {}
};

class RainCloud : public Cloud {
public:
    RainCloud(float px, float py, float vx, float vy, float vapor)
    : Cloud(px, py, vx, vy, vapor) {
    }
    RainCloud(vec2 p, vec2 v, float vapor)
    : Cloud(p, v, vapor) { }
    RainCloud() {}
};


class GameState {
public:
    vector<Cloud*> clouds;
    //vector<RainCloud*> rainclouds;
    size_t myindex;
    int iteration;

    GameState() {
    }
    ~GameState() {
        for (size_t i = 0; i < clouds.size(); i++) {
            delete clouds[i];
        }
    }

    GameState(const GameState &L) {
        myindex = L.myindex;
        iteration = L.iteration;
        for (size_t i = 0; i < L.clouds.size(); i++) {
            Cloud *c = new Cloud(*L.clouds[i]);
            clouds.push_back(c);
        }
    }


    void addThunderStorm(vec2 pos, vec2 vel, float vapor) {
        clouds.push_back(new ThunderStorm(pos, vel, vapor));
    }
	void addThunderStorm(ThunderStorm *c) {
        clouds.push_back(c);
    }

    void addRainCloud(vec2 pos, vec2 vel, float vapor) {
        clouds.push_back(new RainCloud(pos, vel, vapor));
    }
	void addRainCloud(RainCloud *c) {
        clouds.push_back(c);
    }

    void myIndex(size_t index) {
        myindex = index;
    }

    Cloud* getMyStorm() {
        return clouds[myindex];
    }

    void MakeWind() {

    }

    bool intersects(Cloud *a, Cloud *b) {
        return ((b->pos - a->pos).length() < (a->getRadius() + b->getRadius()));
    }

    bool goingToCollide(Cloud &a, Cloud &b, float &collision_time) {

        float r0 = a.getRadius();
        float r1 = b.getRadius();

        vec2 C0 = vec2(a.pos.x, a.pos.y);
        vec2 V0 = vec2(a.vel.x, a.vel.y);

        vec2 C1 = vec2(b.pos.x, b.pos.y);
        vec2 V1 = vec2(b.vel.x, b.vel.y);

        vec2 dv = V1 - V0;
        vec2 dc = C1 - C0;

        float ga = dv.dot(dv);
        float gb = 2.0 * dv.dot(dc);
        float gc = dc.dot(dc) - (r0+r1)*(r0+r1);

        float root = 0.0;
        bool res = getLowestRoot(ga, gb, gc, 1000000, &root);
        if (res) {
            collision_time = root;
            return true;
        }
    }

    float next_collision(Cloud *&first, Cloud *&second) {

        float lowest_time = 9999999;
        //Cloud *first;
        //Cloud *second;
        first = NULL;
        second = NULL;

        for (size_t i = 0; i < clouds.size(); i++) {
            Cloud *a = clouds[i];

            // Check collide with walls
            vec2 new_vel;
            float wall_time = checkBorderCollision(a->pos, a->vel, a->getRadius(), new_vel);
            if (wall_time < lowest_time) {
                lowest_time = wall_time;
                first = a;
                a->new_vel = new_vel;
            }

            for (size_t i = 0; i < clouds.size(); i++) {
                Cloud *b = clouds[i];

                Cloud *smallest = a->getRadius() < b->getRadius() ? b : a;
                Cloud *biggest = a->getRadius() > b->getRadius() ? b : a;

                //while (intersects(*a, *b)) {
                    //if (smallest->vapor < 1.0) break;
                    //biggest->vapor += 1.0;
                    //smallest->vapor -= 1.0;
                //}

                // Check if they're going to collide
                float col_time;

                bool gtc = goingToCollide(*a, *b, col_time);
                if (gtc) {
                    if (col_time < wall_time) {
                        // Collide!
                        if (col_time < lowest_time) {
                            lowest_time = col_time;
                            first = a;
                            second = b;
                        }
                    }
                }
            }
        }
#ifdef RENDER
        glLoadIdentity();
        glColor3f(1, 1, 0);
        glTranslatef(first->pos.x + first->vel.x*lowest_time, 
                     first->pos.y + first->vel.y*lowest_time, 0);
        glutSolidSphere(first->getRadius(), 100, 100);
        glLoadIdentity();
#endif

        //logg << "lowest_time: ";
		//logg << lowest_time << endl;
        return lowest_time;
    }

    void next_move(MyWindow *window, ofstream &logg) {
        double time_start = getTime();
        int simulations = 0;
        Cloud *me = getMyStorm();

        vec2 d;
        GameState *bestcopy = NULL;
        float kk = -9999999;

        vec2 bestmove;

        // null move
        {
            GameState *copy = new GameState(*this);
            float gg = -99999;
            for (int i = 0; i < 50; i++) {
                simulations++;
                float g = copy->simulate(logg);
                if (g > gg) {
                    gg = g;
                }
            }
            if (gg > kk) {
                bestmove = vec2(0,0);
                bestcopy = copy;
                kk = gg;
            }
			free(copy);
        }
		
        for (size_t i = 0; i < clouds.size(); i++) {
            float t = 0.1;
            for (int lol = 0; lol < 2; lol++, t *= 2) {
                for (; t < 20.0; t += 0.1) {
                    if (i == myindex) continue;
                    Cloud *c = clouds[i];
                    if (c->vapor <= 0) continue;
                    if (c->vapor > me->vapor) continue; // prune

                    vec2 wm = (c->pos + (c->vel * t) - me->pos)/t;
                    vec2 hm = wm - me->vel;
                    d = (hm * me->getRadius() / 5.0);
                    float strength = d.length();

                    if (strength > me->vapor/2.0) continue;
                    if (strength < 1.0) continue;
                    if (c->vapor - strength < strength*0.1) continue;
                    logg << "t " << t << endl;
                    break;
                }
                //cout << d << endl;

                GameState *copy = new GameState(*this);
                copy->MakeMove(d);
                float gg = -99999;
                for (int i = 0; i < 50; i++) {
                    simulations++;
                    float g = copy->simulate(logg)/((0.1*i+1));
                    if (g > gg) {
                        gg = g;
                    }
                }
				logg << "gg " << gg << endl;
				logg << "kk " << kk << endl;
                if (gg > kk) {
                    bestmove = d;
                    bestcopy = copy;
                    kk = gg;
                }
				free(copy);
				copy = NULL;
#ifdef RENDER
                glLoadIdentity();
                glColor3f(0, gg/1000.0, gg/1000.0);
                glColor3f(0, 1.0, 1.0);
                drawLine(me->pos.x, me->pos.y, 
                        5*(me->vel.x + 5.0 * (d.x / me->getRadius())),
                        5*(me->vel.y + 5.0 * (d.y / me->getRadius())));
#endif
            }
        }
        logg << "time " << (getTime() - time_start)*1000 << "ms" <<  endl;
        logg << "sims " << simulations << endl;
        //logg << "best " << bestmove << endl;
        logg << "cost " << bestmove.length() <<  endl;
        logg << "lol " << me->vapor <<  endl;
        logg << "gain " << kk <<  endl;
        if (bestcopy)
            window->Render(bestcopy);
    }

    void MakeMove(vec2 d) {
        float strength = sqrt(d.x*d.x + d.y*d.y);
        Cloud *me = getMyStorm();

        // Make move
        float storm_radius = me->getRadius();
        float raincloud_radius = sqrt(strength);
        vec2 w = d / strength;

        me->vapor -= strength;
        // if vapor < 1.0; kill

        me->addVelocity(
                (d.x / me->getRadius())*5.0f,
                (d.y / me->getRadius())*5.0f);

        float distance = (storm_radius + raincloud_radius) * 1.1;
        vec2 rainPosition = me->pos - (w * distance);
        vec2 rainVelocity = (d / strength) * -20.0f + me->vel;
        addRainCloud(rainPosition, rainVelocity, strength);
    }

    float simulate(ofstream &logg) {
        for (size_t i = 0; i < clouds.size(); i++) {
            // Move stuff
            Cloud *a = clouds[i];

            a->pos += a->vel*0.1f;
            a->vel *= 0.999f;

            // Resolve collisions
            for (size_t j = 0; j < clouds.size(); j++) {
                if (i == j) continue;
                Cloud *b = clouds[j];

                Cloud *smallest = a->getRadius() < b->getRadius() ? a : b;
                Cloud *biggest = a->getRadius() < b->getRadius() ? b : a;

                while (intersects(smallest, biggest)) {
                    if (smallest->vapor < 1.0f) {
                        break;
                    }
                    biggest->vapor += 1.0f;
                    smallest->vapor -= 1.0f;
                }
            }
            // Bounce
            float radius = a->getRadius();
            if (a->pos.x < radius)
            {
                a->pos.x = radius;
                a->vel.x = fabs(a->vel.x) * 0.6;
            }
            if (a->pos.y < radius)
            {
                a->pos.y = radius;
                a->vel.y = fabs(a->vel.y) * 0.6;
            }
            if (a->pos.x + radius > WIDTH)
            {
                a->pos.x = WIDTH - radius;
                a->vel.x = -fabs(a->vel.x) * 0.6;
            }
            if (a->pos.y + radius > HEIGHT)
            {
                a->pos.y = HEIGHT - radius;
                a->vel.y = -fabs(a->vel.y) * 0.6;
            }
        }
        return getMyStorm()->vapor; // todo
    }

    void simple() {
        for (int iter = 0; iter < 100; ++iter) {
            Cloud *first;
            Cloud *second;
            float next_time = next_collision(first, second);

            // Move stuff
            for (size_t i = 0; i < clouds.size(); i++) {
                Cloud *c = clouds[i];
                c->pos += c->vel*0.1f*next_time;
                c->vel *= powf(0.999f,next_time);
            }

            // Resolve collisions
            if (first && second) {
                //if (intersects(*first, *second))
                    //cout << "resolve" << endl;
                // resolve
            } else if (first) {
                first->vel = first->new_vel;
            }
        }
    }

    float checkBorderCollision(vec2 pos, vec2 velocity, float r, vec2 &new_vel) {
        vec2 C0 = pos;
        vec2 V0 = velocity;

        new_vel = velocity;

        float min_root = 1e09;
        for (int i = 0; i < 4; i++) {
            vec2 E;
            vec2 D;
            float x1 = C0.x;
            float y1 = C0.y;
            switch (i) {
                case 0: E = vec2(0,0); D = vec2(0, 720); x1 = C0.x-r; y1 = C0.y; break; // left
                case 1: E = vec2(0,0); D = vec2(1280, 0); x1 = C0.x; y1 = C0.y-r; break; // top
                case 2: E = vec2(1280,0); D = vec2(0, 720); x1 = C0.x+r; y1 = C0.y; break; // right
                case 3: E = vec2(0,720); D = vec2(1280, 0); x1 = C0.x; y1 = C0.y+r; break; // bottom
            }
            float x2x1 = velocity.x;
            float y2y1 = velocity.y;
            float x2 = C0.x + V0.x;
            float y2 = C0.y + V0.y;
            float x3 = E.x;
            float y3 = E.y;
            float x4 = E.x + D.x;
            float y4 = E.y + D.y;

            float det = (y4-y3)*(x2x1) - (x4-x3)*(y2y1);
            float ua = ((x4-x3)*(y1-y3) - (y4-y3)*(x1-x3))/det;
            //float ub = ((x2-x1)*(y1-y3) - (y2-y1)*(x1-x3))/det;

            if (ua < 0) continue;

            vec2 d = vec2(C0.x + ua*velocity.x, C0.y + ua*velocity.y);
            //cout << d << endl;

            if (ua < min_root) {
                min_root = ua;

                switch (i) {
                    case 0: new_vel = vec2(abs(velocity.x)*0.6, velocity.y); break;
                    case 1: new_vel = vec2(velocity.x, abs(velocity.y)*0.6); break;
                    case 2: new_vel = vec2(-abs(velocity.x)*0.6, velocity.y); break;
                    case 3: new_vel = vec2(velocity.x, -abs(velocity.y)*0.6); break;
                }
#ifdef RENDER
                glLoadIdentity();
                glColor3f(0,1,1);
                glBegin(GL_LINES);
                glVertex3f(C0.x, C0.y, 0);
                glVertex3f(d.x, d.y, 0);
                glVertex3f(d.x, d.y, 0);
                glVertex3f(d.x+new_vel.x, d.y+new_vel.y, 0);
                glEnd();
#endif
            }
        }

        return min_root;
    }
};

#endif
