-------------------------------------------------------------------
-- Small program for testing motion blur as post processing effect
-------------------------------------------------------------------

import Graphics.GPipe
import Data.Vec hiding (map, sum, foldl)
import Data.Vec.LinAlg.Transform3D
import Data.IORef
import Data.Monoid
import Graphics.UI.GLUT (Window, mainLoop, postRedisplay, idleCallback, getArgsAndInitialize, ($=))

-- Prolog

main :: IO ()
main = do
    getArgsAndInitialize
    timeRef <- newIORef 0.0
    newWindow "Motion blur" (100:.100:.()) (800:.600:.()) (renderFrame timeRef) initWindow
    mainLoop
    where
        initWindow win = idleCallback $= Just (postRedisplay (Just win))

        renderFrame timeRef size = do
            time <- readIORef timeRef
            writeIORef timeRef (time + (1/60))
            return $ outFrameBuffer time size
 
-- Interesting parts

sphericalCoordCube :: Int -> Int -> PrimitiveStream Triangle (Vec3 (Vertex Float))          
sphericalCoordCube w h = stream
    where strip p = [ (i,j) | i <- coords 0 w w, j <- coords p (p+1) h ]
          plane = [ strip p | p <- [0..(h-1)] ]
          coords a b w = [ (fromIntegral x * 2) / fromIntegral w - 1 | x <- [a..b] ]
          x1 (i,j) = 1:.i:.j:.()
          x2 (i,j) = (-1):.i:.j:.()
          y1 (i,j) = i:.1:.j:.()
          y2 (i,j) = i:.(-1):.j:.()
          z1 (i,j) = i:.j:.1:.()
          z2 (i,j) = i:.j:.(-1):.()
          side f = map (\s -> map (spherical . f) s) plane
          sides = mconcat $ map side [x1,x2,y1,y2,z1,z2]
          stream = mconcat $ map (toGPUStream TriangleStrip) sides

fooCube = sphericalCoordCube 4 4 --12 12
fooSphere time = fmap spherify fooCube
    where spherify (r:.i:.a:.()) = cartesian $ (mix 1 r ratio):.i:.a:.()
          ratio = (toGPU 1.5) --(sin $ toGPU time) * 1.5

sphericalNormalize ratio point = cartesian ((1.0 + ratio * (r - 1.0)):.i:.a:.())
    where (r:.i:.a:.()) = spherical point

spherical (x:.y:.z:.()) = r:.i:.a:.()
    where r = norm (x:.y:.z:.())
          i = acos (z / r)
          a = atan2 x y

cartesian (r:.i:.a:.()) = x:.y:.z:.()
    where x = r * sin i * cos a
          y = r * sin i * sin a
          z = r * cos i     

quadAnim :: Float -> Mat44 Float
quadAnim time = rotationY $ time

fooAnim :: Float -> Float -> Mat44 Float
fooAnim off time = rot `multmm` move
    where move = translation $ off:.0:.0:.()
          rot = rotationY $ -time 

turnTableCamera :: Float -> Mat44 Float
turnTableCamera time = proj `multmm` pos `multmm` rot
    where proj = perspective 0.1 100 (pi / 3) 1
          pos = translation $ 0:.0:.(-10):.()
          rot = rotationY $ time * 0.5 

transform :: Float -> (Float -> Mat44 Float) -> (Float -> Mat44 Float) -> (Vec3 (Vertex Float) -> Vec4 (Vertex Float))
transform time view model = op 
    where mat = toGPU $ (view time) `multmm` (model time)
          op v = mat `multmv` (homPoint v :: Vec4 (Vertex Float))

draw :: Float
     -> [((Float -> Mat44 Float), PrimitiveStream Triangle (Vec3 (Vertex Float)))] 
     -> FrameBuffer RGBFormat DepthFormat ()
draw time objects = outColor
    where 
          paint = paintColorRastDepth Lequal True NoBlending (RGB $ vec True)
          paintAll f bg = foldl (\acc x -> paint (f x) acc) bg objects

          outColor = paintAll (colorPipe) $ newFrameBufferColorDepth (RGB 0) 1

          --rasterize both sides and strip front facing tag
          rasterize x = fmap (\(_, values) -> values) $ rasterizeFrontAndBack x            
          
          colorPipe (anim, geom) = fmap colorize $ rasterize $ fmap (\(v) -> (pos1 anim v, (world anim v, v))) $ geom
          colorize (world, object) = RGB $ vec $ (normal world) `dot` light
          tu (x:.y:.z:.()) = (dFdx x):.(dFdx y):.(dFdx z):.()
          tv (x:.y:.z:.()) = (dFdy x):.(dFdy y):.(dFdy z):.()
          normal p = normalize $ cross (tu p) (tv p)
          world anim (x:.y:.z:.()) = hdivide $ (toGPU $ anim time) `multmv` (x:.y:.z:.1:.())
          light = toGPU $ normalize $ 1:.1:.1:.()

          pos1 anim v = transform time (turnTableCamera) (anim) v
          hdivide (x:.y:.z:.w:.()) = (x/w):.(y/w):.(z/w):.()

outFrameBuffer :: Float -> Vec2 Int -> (FrameBuffer RGBFormat DepthFormat ())
outFrameBuffer time size = out
    where out = draw time objs
          objs = map (\m -> (place m, fooSphere time)) mms
          place m t = m
          mms = Prelude.take 20 $ iterate step (identity :: Mat44 Float) 
          step acc = acc `multmm` translation (0:.3:.0:.()) `multmm` rotationX 0.4 

