module Util where

import Data.Vec 
import Graphics.GPipe
import Graphics.GPipe.Texture.Load
import Codec.Image.STB
import Data.Bitmap.IO (withBitmap)
import Data.Either
import Types
import TextGen
import Foreign
import Foreign.C
import Codec.Binary.UTF8.String (encodeString)

data FlashTimes = FlashTimes { begin::Float,
                               stable::Float,
                               fadeout::Float,
                               end::Float }

data RawSceneItem = RawFlashText String Float (Vec2 Float) FlashTimes
     		  | RawFlashImage String Float (Vec2 Float) FlashTimes    

data SceneItem = FlashText { texture::Texture2D RGBAFormat,
     	       	             scale::Float,
                             size::Vec2 Int,
                             position::Vec2 Float,
                             times::FlashTimes }
               | FlashImage { texture::Texture2D RGBAFormat,
                              scale::Float,
                              size::Vec2 Int,
			      position::Vec2 Float,
			      times::FlashTimes }

data SceneValues = SceneValues { haskellTexture :: Texture2D RGBAFormat,
                                 letterTexture :: Texture2D RGBAFormat,
                                 cube :: Geometry,
                                 hresCube :: Geometry,
                                 generalScenes :: [SceneItem] }

load (RawFlashText text scale position times) = do
  (texture, dims) <- toTexture text $ "Monospace " ++ show (round (scale*7))
  return $ FlashText texture scale dims position times
load (RawFlashImage tfile scale position times) = do
  texture <- loadTexture' tfile
  dim <- getTexture2DSize tfile
  return $ FlashImage texture scale dim position times

-- Type declared loading for satisfying type function TextureFormat
loadTexture' :: String -> IO (Texture2D RGBAFormat)
loadTexture' path = loadTexture RGBA8 path


-- Utilities to get image dimensions from a file
texture2DSizeFromImage path (w,h) comp 0 ptr = return (w:.h:.())
texture2DSizeFromImage path _ _ _ _ = ioError $ userError ("loadTexture: Row padding is not supported, in " ++ show path)

getTexture2DSize' io path = do image <- loadImage path
                               either
                                  (ioError . userError)
                                  (flip withBitmap io)
                                  image

-- Returns dimensions of a image file
getTexture2DSize :: FilePath -> IO (Vec2 Int)
getTexture2DSize path = getTexture2DSize' (texture2DSizeFromImage path) path

defMarqueeTime = 300 -- ms

calcBegin :: (Float, Float) -> Float
calcBegin (begin, end) = begin - defMarqueeTime

calcStable :: (Float, Float) -> Float
calcStable (begin, end) = begin

calcFadeout :: (Float, Float) -> Float
calcFadeout (begin, end) = end - defMarqueeTime

calcEnd :: (Float, Float) -> Float
calcEnd (begin, end) = end

generateText rawTxt font =
    withCString txt $ \txt' ->
        withCString font $ \font' -> do
          img <- createTextImage txt' font'
          return img
    where txt = encodeString rawTxt -- Unicode to UTF-8

toTexture :: String -> String -> IO (Texture2D RGBAFormat, Vec2 Int)
toTexture txt font = do
    imagePtr <- generateText txt font
    image <- peek imagePtr 
    let size = ((mkInt (width image)):.(mkInt (height image):.()))
    tex <- newTexture (PerComp4 UnsignedByteFormat) RGBA8 size [idata image]
    return (tex, size)

mkInt :: CInt -> Int
mkInt n = fromIntegral n
