-----------------------------------------------------------------------------
--
-- Module:      Scheduler
-- Author:      Hot Lambda Lovin' (2011)
--
-- Description: In short: Time -> FrameBuffer.
--              Schedules elements in place.
--
-----------------------------------------------------------------------------

module Scheduler where

import Graphics.GPipe
import Data.Vec hiding (map, sum, foldl)
import Data.Monoid
import Data.Maybe
import Control.Monad
import Env
import MathUtil
import Types
import qualified Util as U
import DisplayUtils as DU

import qualified LayerEffects as S
import qualified MetaCircleScene as S
import qualified SimpleScene as S
import qualified CubeScene as S
import qualified CubeScene2 as S
import qualified HypnoScene as S
import qualified MatrixScene as S

renderFrame :: Vec2 Int -> Time -> U.SceneValues -> Taulu
renderFrame size time sceneVals = foldl (\bg scene -> S.drawSceneItem bg scene t) bg items
  where 
    items     = U.generalScenes sceneVals
    bg        = sailTheSeas (t / 1000) galleons
    t         = time -- -300 -- compensate audio delay (temp)
    galleons  = [
            matrixScreenSaver (U.letterTexture sceneVals),
            introScreenSaver sceneVals,
            topicScreenSaver size sceneVals,
            furryScreenSaver sceneVals,
            tentacleScreenSaver (U.cube sceneVals),
            endScreenSaver
          ]

bgblack      = (0.0:.0.0:.0.0:.())
bgpurple     = (0.7:.0.0:.0.81:.())
sunglasses c = 0.5*c -- eye protection for jypa's #be00cf

pirunSynkkää     (_,_,_) = colorPlease bgblack 1
violettiaKaipaan (r,_,_) = colorPlease (lerp bgblack (bgpurple) (vec r)) r


dimmerBar (y1,y2) intensity (r,_,_) = fmap (\(u:.v:.()) -> RGBA (vec 0) (toGPU r * intensity)) $ screenQuad (0:.(y1):.()) (2:.y2:.())
movingBar (y,h) intensity (r,_,_) = fmap (\(u:.v:.()) -> RGBA (vec 0) (toGPU intensity)) $ screenQuad (0:.(pos):.()) (2:.h:.())
    where off = (1+(h/2)) * (y / abs y)
          pos = lerp off y r
                                                                                                                           
matrixScreenSaver :: Texture2D RGBAFormat -> SpanishGalleon
matrixScreenSaver tex = SpanishGalleon scene [] [] []
    where scene = adapt [
            (51, 57, cosEW 2, S.matrixEffect tex),
            -- Hickup hack, preload 
            (0,0.1, rectW, S.matrixEffect tex),
            (0,0.1, rectW, pirunSynkkää)
           ]

introScreenSaver :: U.SceneValues -> SpanishGalleon
introScreenSaver vals = SpanishGalleon scene [] [] []
    where scene = adapt [
              (0.5, 12, cosEW 1, S.showSomeHaskell vals),
              (0, 14, cosEW 0.5, S.showSomeBalls bgblack)
            ]

topicScreenSaver :: Vec2 Int -> U.SceneValues -> SpanishGalleon
topicScreenSaver koko vals = SpanishGalleon bg mg [] fg
    where bg = adapt [(14.5, 20, softEW 0.1 1.0, violettiaKaipaan)]
          mg = adapt [ (13.5, 21.5, rectW, S.addShadow koko . S.hypnoShutter)]
          fg = adapt [(15, 20.5, cosEW 0.5, dimmerBar (-0.05,0.4) 0.4 )]


furryScreenSaver :: U.SceneValues -> SpanishGalleon
furryScreenSaver vals = SpanishGalleon scene hugecube [] []
    where scene = adapt [
              (22, 50, cosEW 1, S.showSomeHaskell vals),
              (20.5, 50, cosEW 1, S.showSomeBalls (bgpurple)),
              (14.5, 50, softEW 0.1 2.0, S.dimRim (**1.2))
            ]
          hugecube = adapt [
              (21, 49, cosEW 0.5, S.cubeFrame2 (U.hresCube vals))
            ]

tentacleScreenSaver :: Geometry -> SpanishGalleon
tentacleScreenSaver geom = SpanishGalleon [] [] tentacles []
    where tentacles = adapt [(56, 200, rectW, S.genLonkero geom)]
          rings = adapt [(49.5+x, 50+x, rectW, S.ringEffect (0:.0.2:.()) bgpurple) | x <- [0,0.15..0.5] ]

endScreenSaver :: SpanishGalleon
endScreenSaver = SpanishGalleon bg [] [] fg
    where 
          bg    = adapt [
                (60.0, 100, cosEW 3.0, violettiaKaipaan),
                (58, 64, softEW 0.2 1, S.dimRim (**1.7))
            ]
          fg    = adapt [
                (65, 80, softEW 0.2 1, movingBar (-0.75,0.7) 0.7 ),
                (65, 80, softEW 0.2 1, movingBar (0.75,0.7) 0.7 )
            ]


notTentacles :: SpanishGalleon
notTentacles = SpanishGalleon [] scene [] []
    where scene = adapt  [(50, 108, rectW, S.überHypnoChaosEffect), (16,22,rectW, S.singleSpiral)]

-- Piirrä kerrostumia 
sailTheSeas :: Aika -> [SpanishGalleon] -> Taulu
sailTheSeas aika galleons = all
    where 
        kerrokset = [(pohja, DU.blend), (tausta, DU.blend), (geometria, DU.ztest), (edusta, DU.blend)]
        allHandsHoay bg (kerros,raapustin) = piirustele aika raapustin (kuori kerros) bg
        all = foldl (allHandsHoay) DU.tyhjyys kerrokset
        kuori k = concatMap k galleons
            
-- Tutkaile listaa vitkuttimia ja piirrä ajanhetkellä näkyvissä olevat            
piirustele :: Aika -> Raapustin -> [Vitkutin] -> Taulu -> Taulu
piirustele aika raapustin vitkuttimet pohja = taulu
    where 
        taulu = foldl (\tausta (kurkistus, maisema) -> raapustin (maisema (kurkistus aika)) tausta) pohja kelvot
        kelvot = mapMaybe valikoija vitkuttimet
        valikoija (kurkistus, maisema) = do
            let tarkistin (vahvuus, _, _) = vahvuus > 0
            guard $ tarkistin (kurkistus aika)
            return (kurkistus, maisema)

-- Apufunktio ikkunafunktioiden käsittelyyn
kurkistelija :: Aika -> Aika -> Ikkuna -> Kurkistus
kurkistelija alku loppu ikkuna = sikiö
    where leveys = distance alku loppu
          sikiö aika = (ikkuna alku loppu aika, aika', aika' / leveys)
            where aika' = aika - alku          

adapt raw = [ (kurkistelija a b w, f) | (a, b, w, f) <- raw ]
