module MathUtil where

import Data.Boolean
import Data.Vec
import Types
import Graphics.GPipe


spiral n c (r:.a:.()) = sin (a * n + r * c)

funstep w s i x = x + blockcos s (x * w) / w * i
blockcos s x = cos x / (s + abs (cos x)) * (1 + s)    


-- Rescale wavelength pi*2 to 1    
rad01 :: (Floating a) => (a -> a) -> a -> a
rad01 f x = f (pi*2*x)

--Scale range from [-1,1] to [0,1]
range01 :: (OrdB bool a, IfB bool a, Fractional a) => a -> a
range01 = (+ 0.5) . (* 0.5)

range11 x = x * 2 - 1    

-- Mask f x to 0 with x outside [0,1]
mask01 :: (OrdB bool a, IfB bool a, Floating a) => (a -> a) -> a -> a
mask01 f x = ifB (x <* 0) 0 $ ifB (x >* 1) 0 (f x)

-- Move 
moveF :: (OrdB bool a, IfB bool a, Floating a) => a -> a -> (a -> a) -> a -> a
moveF a b f x = f $ (x - a) / (b - a)

moveEdgedF :: (OrdB bool a, IfB bool a, Floating a) => a -> a -> a -> (a -> a) -> a -> a
moveEdgedF e a b f x = moveF (a - e) (b + e) (f) x

-- 
taper :: (OrdB bool a, IfB bool a, Floating a) => a -> a -> a
taper ratio x = ifB (x <* r') (x / ratio) $ ifB (x >* (1-r')) ((x-1) / ratio + 1) 0.5
    where r' = ratio / 2

-- Raw, unclamped versions for working on
rectangle' _ = 1
sawtooth' x = x
triangle' x = 1 - abs (x * 2 - 1)
cosine' x = (range01 . rad01 cos . (+ 0.5)) x
cosineB' softness x = (range01 . rad01 (blockcos softness) . (+ 0.5)) x

-- Utilites to mask and move window functions 
buildW f a b x = moveF a b (mask01 f) x
buildEdgedW f e a b x = moveEdgedF e a b (mask01 f . taper r) x
    where r = e / (hw + e) -- absolute edge with to taper ratio
          hw = (distance a b) / 2

-- Window functions
rectW a b x = buildW rectangle' a b x
triW a b x = buildW triangle' a b x
sawW a b x = buildW sawtooth' a b x
cosW a b x = buildW cosine' a b x
softW softness a b x = buildW (cosineB' softness) a b x

-- Window functions with controllable edge width
rectEW e a b x = buildEdgedW rectangle' e a b x
triEW e a b x = buildEdgedW triangle' e a b x 
sawEW e a b x = buildEdgedW sawtooth' e a b x
cosEW e a b x = buildEdgedW cosine' e a b x 
softEW softness e a b x = buildEdgedW (cosineB' softness) e a b x


distance :: (OrdB bool a, Fractional a) => a -> a -> a
distance a b = abs (a - b)

--GPipe doesn't define GLSL atan2
atan2' (x:.y:.()) = ifB (x >* 0) a $ ifB ((x <* 0) &&* (y >=* 0)) (a + pi) (a - pi)
    where a = atan (y / x)

polar xy = (norm xy):.(atan2' xy):.()
lerp a b r = a + r * (b - a) 

spherical (x:.y:.z:.()) = r:.i:.a:.()
    where r = norm (x:.y:.z:.())
          i = acos (z / r)
          a = atan2' (x:.y:.())

cartesian (r:.i:.a:.()) = x:.y:.z:.()
    where x = r * sin i * cos a
          y = r * sin i * sin a
          z = r * cos i    
