module DisplayUtils where

import Graphics.GPipe
import qualified Data.Vec as Vec
import Data.Vec (vec, Mat33, n3)
import Data.Vec.LinAlg.Transform3D
import Types
import MathUtil
import Data.List as List (transpose)
import Data.Maybe (fromJust)

emptyFB :: Float -> Float -> Float -> Float -> Taulu
emptyFB r g b a = newFrameBufferColorDepth (RGBA (r:.g:.b:.()) a) 1

tyhjyys :: Taulu
tyhjyys = newFrameBufferColorDepth (RGBA (vec 0) 0) 1

-- Blending for transparent layers
blend :: Raapustin
blend = paintColor niceBlending (RGBA (True:.True:.True:.()) True)

-- Depth test for 3D objects
ztest :: Raapustin
ztest = paintColorRastDepth Lequal True niceBlending (RGBA (vec True) True)

niceBlending = Blend (FuncAdd, FuncAdd) (factor, factor) (RGBA (vec 1) 1)
    where factor = (SrcAlpha, OneMinusSrcAlpha)


-- Screen sized quad
quad :: PrimitiveStream Triangle (Vec4 (Vertex Float), Vec2 (Vertex Float))
quad = toGPUStream TriangleStrip $ zip vertices uvCoords
    where vertices = [ i:.j:.0:.1:.() :: Vec4 Float | i <- [-1,1], j <- [-1,1] ]
          uvCoords = [ i:.j:.() :: Vec2 Float | i <- [0,1], j <- [0,1] ]

-- Raw screen space UV values
uvPlane :: FragmentStream (Vec2 (Fragment Float))
uvPlane = rasterizeBack $ quad

-- Paint rgba values to the whole screen
colorPlease :: Vec3 Float -> Float -> Kuva
colorPlease rgb alpha = fmap (colorize) uvPlane
    where (r:.g:.b:.()) = (toGPU rgb) :: Vec3 (Fragment Float)
          a = toGPU alpha :: Fragment Float
          --Got segfaults with _, using uv seems to avoid it
          colorize (u:.v:.()) = RGBA (r:.g:.(b + 0.001 * u):.()) a

-- Paint a texture screen sized Y-inverted for text/images                                                                
texturePleaseF :: Texture2D RGBAFormat -> Float -> Kuva
texturePleaseF tex alpha = screenTextureFlip true tex alpha (vec 0) (vec 2)

-- Paint a texture screen sized    
texturePlease :: Texture2D RGBAFormat -> Float -> Kuva
texturePlease tex alpha = screenTextureFlip false tex alpha (vec 0) (vec 2)

-- Positioned and screen aligned quad with parametric coordinates
screenQuad :: Vec2 Float -> Vec2 Float -> FragmentStream (Vec2 (Fragment Float))
screenQuad (x:.y:.()) (w:.h:.()) = rasterizeBack $ scaledQuad
    where scaledQuad = fmap (\(v, a) -> (matrix `multmv` v, a)) quad
          scale = scaling $ (w/2):.(h/2):.1:.()
          translate = translation $ (x:.y:.0:.())
          matrix = toGPU $ translate `multmm` scale

-- Texture rendering with position, alpha control and reversed Y axis for text or images
-- Our top is (1.0) and bottom (-1.0) whilst forementioned have top of (0.0) and bottom of (1.0)
screenTexture :: Texture2D RGBAFormat -> Float -> Vec2 Float -> Vec2 Float -> Kuva
screenTexture tex alpha pos dim = screenTextureFlip true tex alpha pos dim
    
screenTextureFlip :: Bool -> Texture2D RGBAFormat -> Float -> Vec2 Float -> Vec2 Float -> Kuva
screenTextureFlip flip tex alpha pos dim = (fmap colorize) $ screenQuad pos dim
    where a = toGPU alpha
          colorize (u:.v:.()) = RGBA c (a'*a)
            where RGBA c a' = sample (Sampler Linear Wrap) tex (u:.v':.())
                  v' = ifB (toGPU flip) (1-v) v

-- Uses only alpha channel of texture with given color
screenText :: Texture2D RGBAFormat -> Vec3 Float -> Float -> Vec2 Float -> Vec2 Float -> Kuva
screenText tex color alpha pos dim = (fmap colorize) $ screenQuad pos dim
    where a = toGPU alpha
          c = toGPU color
          colorize (u:.v:.()) = RGBA c (a'*a)
            where RGBA _ a' = sample (Sampler Linear Wrap) tex (u:.(1-v):.())
                  


-- Generate a cube with desired resolution
-- Should not be used directly, instead generate once (like with texture loading)
generateCube :: Int -> Int -> Geometry
generateCube width height = toIndexedGPUStream TriangleList planes indices
    where
        n = (width + 1) * (height + 1)
        w = fromIntegral width
        h = fromIntegral height
        
        -- Generate grid of parametric coordinates and vertices
        coords = [ x:.y:.() | x <- [0,(1/w)..1], y <- [0,(1/h)..1] ]
        vertices = map (range11) coords
        
        -- Utils to move vertices to 3D domain
        fx d (x:.y:.()) = d:.x:.y:.()
        fy d (x:.y:.()) = x:.d:.y:.()
        fz d (x:.y:.()) = x:.y:.d:.()

        -- Move vertices and zip with normals and UVs
        planeFuncs = [ (f d) | d <- [-1,1], f <- [fx,fy,fz] ]
        planes = concat $ map (side) planeFuncs
        side f = zip3 (map (f) vertices) (repeat $ f (0:.0:.())) (coords)
        
        -- Get indices starting at offset
        g i j = [ i'*(height+1)+j' | i' <- [i..width+i-1], j' <- [j..height+j-1] ]
        
        -- Build indices for pairs of triangles at offsets
        face = concat $ List.transpose ([ g 0 0, g 0 1, g 1 0, g 1 1, g 0 1, g 1 0])
        
        -- Patch together for all six faces
        indices = concat $ map (\m -> map (+ (n*m)) face) [0..5]


-- Create fragments from geometry        
toFrags :: GeometryTransformed -> Fragments          
toFrags geom = toFrags' geom
    where toFrags' = fmap (\(_, a) -> a) . rasterizeFrontAndBack
          
-- Transform Geometry and Normals    
transformPlease :: Transform -> Geometry -> GeometryTransformed
transformPlease (proj,view,model) object = applyTr object
    where 
        applyTr = fmap (\(p,n,uv) -> (trScreen p, (p, trWorld p, trNormal n, uv)))
        
        trNormal n = (toGPU normMat :: Mat33 (Vertex Float)) `multmv` (normalize n)
        trScreen v = (toGPU modelViewProj) `multmv` (homPoint v :: Vec4 (Vertex Float))
        trWorld  v = homDivide $ (toGPU model) `multmv` (homPoint v :: Vec4 (Vertex Float))

        viewProj = proj `multmm` view 
        modelView = view `multmm` model 
        modelViewProj = viewProj `multmm` model 
        normMat = Vec.transpose $ fromJust $ invert $ Vec.map (Vec.take n3) $ Vec.take n3 model 
        homDivide (x:.y:.z:.w:.()) = (x/w):.(y/w):.(z/w):.()
