/*

 Name      :  VGA
 Hardware  :  4D Systems Micro VGA
 Notes     :  Video Module using serial protocol to make graphics
 
 I encountered two major bottlenecks which makes the demo slow:
 1 - The serial protocol.  A 310 x 210 image @ 256 indexed color 
 occupies 520800 bytes. Multiply this by 15 for average FPS.
 yikes.  Then squeeze down a serial protocol connection.
 2 - waiting for response from the card halts all program execution.
 
 Three reasons why I kept it:
 1 - The demo would have a greater impact on a big projected screen.
 2 - This was the easiest option to get double buffered 256 color graphics
 3 - working with restrictions is FUN!
 
 workarounds:
 increase hardware serial to 1 Million Baud giving 125000 bytes per second.
 reduce frame size - put effects inside mini monitor 
 leverage built-in graphic functions of uVGA module
 make everything else fast
 
 references:
 http://mekonik.wordpress.com/2009/03/02/modified-arduino-library-serial/
 http://www.arduino.cc/cgi-bin/yabb2/YaBB.pl?num=1242466935
 http://www.4dsystems.com.au/downloads//micro-VGA/uVGA-PICASO-MD1/Docs/Pdf/uVGA-PICASO-MD1_Serial_Users_Manual_Rev1.2.pdf  
 
 */

byte IMGbuffer[IMGBUFFER_LENGTH];
byte IMGbufferCounter=0;

void init_vga(){
  Serial.begin(1000000); // this is the fastest we can go without data corruption
  Serial.flush();        // get rid of any data in hardware serial buffer
                         // todo: we should trigger the VGA reset pin here
  delayMicro(1200);      // wait for uVGA to power up
  VGA_sendCommand(0x55); // autobaud - sync the communication speed
  VGA_GetResponse();
  clearScreen();         // clear the screen
  VGA_sendCommand(0x42); // set background color
  VGA_sendCommand(0x00); // indexed color value for black
  VGA_GetResponse();
  VGA_sendCommand(0x4F); // opaque text
  VGA_sendCommand(0x01); 
  VGA_GetResponse();
}

// send instruction to uVGA module
void VGA_sendCommand(byte command){
  Serial.write(command);
}

// prepare pixel array 
void setupIMG(int x, int y, int w, int h){
  VGA_sendCommand(0x49);
  VGA_sendCommand(x>>8);
  VGA_sendCommand(x & 0xFF);
  VGA_sendCommand(y>>8);
  VGA_sendCommand(y & 0xFF);
  VGA_sendCommand(w>>8);
  VGA_sendCommand(w & 0xFF);
  VGA_sendCommand(h>>8);
  VGA_sendCommand(h & 0xFF);
  VGA_sendCommand(8);
}

// uVGA palette RGB color conversion 
byte color(byte r, byte g, byte b){
  return ((b&255 & 0xC0)+((g&255 & 0xE0) >> 2)+((r & 0xE0) >> 5))&0xFF;
}

// place characters on screen using built-in font
void text(int x, int y, int w, int h, int font, byte color, char * str){
  int c=0;
  VGA_sendCommand(0x53);
  VGA_sendCommand(x>>8);
  VGA_sendCommand(x & 0xFF);
  VGA_sendCommand(y>>8);
  VGA_sendCommand(y & 0xFF);
  VGA_sendCommand(font);
  VGA_sendCommand(color);
  VGA_sendCommand(w);
  VGA_sendCommand(h);
  while(str[c])	
  {
    VGA_sendCommand(str[c++]);
    vgm();
  }
  VGA_sendCommand(0);
  vgm();
  VGA_GetResponse();
  vgm();
}

// get confirmation byte from video module saying it completed drawing
void VGA_GetResponse(){
  while (!Serial.available()) {}    // Wait for data avaliable
  Serial.read();                    // burn read incoming byte
}

// draw a rectangle
void rect(int x, int y, int w, int h, byte c){
  VGA_sendCommand(0x72); 
  VGA_sendCommand(x>>8);
  VGA_sendCommand(x & 0xFF);
  VGA_sendCommand(y>>8);
  VGA_sendCommand(y & 0xFF);
  VGA_sendCommand(w>>8);
  VGA_sendCommand(w & 0xFF);
  VGA_sendCommand(h>>8);
  VGA_sendCommand(h & 0xFF);
  VGA_sendCommand(c);          // indexed color value
  VGA_GetResponse();
}

// get pre-encoded indexed color image from SD card
void loadImage(int x, int y, int w, int h, char * graphicFile){
  GHI_PutS("C 2\r");     // close file handle #2  
  GHI_GetResponse(4);    // handle !00 response
  GHI_PutS(graphicFile); // send open file command
  GHI_GetResponse(4);    // handle !00 response
  IMGbufferCounter=0;
  IMGfillBuffer();

  // slam those pixels on the screen
  setupIMG(x,y,w,h);     
  unsigned int pixels = w*h;
  do
  {
    VGA_sendCommand(IMGgetByte());
  }
  while(--pixels);
  VGA_GetResponse();
}

// get a single byte from buffer
byte IMGgetByte(){
  if (IMGbufferCounter==IMGBUFFER_LENGTH)
  {
    IMGfillBuffer();
    IMGbufferCounter=0;
  }
  return IMGbuffer[IMGbufferCounter++];
}

// fill the image buffer
void IMGfillBuffer(){
  GHI_PutS("R 20>0000001E\r"); // read 30 bytes (0x1E) from file handle #2
  GHI_GetResponse(4);          // handle !00 response

  for (byte i = 0; i <IMGBUFFER_LENGTH; i++)
  {
    IMGbuffer[i]=GHI_GetC();
  }   
  GHI_GetResponse(14);        // handle $00000000<CR>!00<CR> response
}

// draw a line
void line(int x1, int y1, int x2, int y2, byte color){
  VGA_sendCommand(0x4C);
  VGA_sendCommand(x1>>8);
  VGA_sendCommand(x1 & 0xFF);
  VGA_sendCommand(y1>>8);
  VGA_sendCommand(y1 & 0xFF);
  VGA_sendCommand(x2>>8);
  VGA_sendCommand(x2 & 0xFF);
  VGA_sendCommand(y2>>8);
  VGA_sendCommand(y2 & 0xFF);
  VGA_sendCommand(color);  
  VGA_GetResponse();
}

// set all write operations to page n
void switchWritePage(byte page){
  VGA_sendCommand(0x59); 
  VGA_sendCommand(0x02); 
  VGA_sendCommand(page); 
  VGA_GetResponse();
}

// select page n to be displayed
void switchViewPage(byte page){
  VGA_sendCommand(0x59); 
  VGA_sendCommand(0x00); 
  VGA_sendCommand(page);
  VGA_GetResponse();  
}

// snag pixels from specified video page and copy them to destination page
void copyPage(int xs,int ys,int xd,int yd, int w, int h, int src, int dest) {
  VGA_sendCommand(0x63); 
  VGA_sendCommand(xs>>8);
  VGA_sendCommand(xs & 0xFF);
  VGA_sendCommand(ys>>8);
  VGA_sendCommand(ys & 0xFF);
  VGA_sendCommand(xd>8);
  VGA_sendCommand(xd & 0xFF);
  VGA_sendCommand(yd>>8);
  VGA_sendCommand(yd & 0xFF);
  VGA_sendCommand(w>8);
  VGA_sendCommand(w & 0xFF);
  VGA_sendCommand(h>>8);
  VGA_sendCommand(h & 0xFF);
  VGA_sendCommand(src);
  VGA_sendCommand(dest);  
  VGA_GetResponse();
}

// doin it round
void drawBall(int x,int y,int r, byte color) {
  VGA_sendCommand(0x43); 
  VGA_sendCommand(x>>8);
  VGA_sendCommand(x & 0xFF);
  VGA_sendCommand(y>>8);
  VGA_sendCommand(y & 0xFF);
  VGA_sendCommand(r>>8);
  VGA_sendCommand(r & 0xFF);
  VGA_sendCommand(color);
  VGA_GetResponse();
}

// clear screen
void clearScreen(){
  VGA_sendCommand(0x45); 
  VGA_GetResponse();
}



