#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <limits.h>
#include <vector>
#include <map>
#include <algorithm>
#include <string>

#define NUM_DISTRO 2
#define SWITCHES_PER_ROW 6
#define TRUNCATE_METRIC 0
#define TENPLUSTEN 250
#define THIRTYPLUSTEN 450
#define FIFTYPLUSTEN 650

struct sw {
	unsigned char row, num;

	sw(unsigned char row, unsigned char num) : row(row), num(num) {}
};
int distro_placements[NUM_DISTRO] = {
	10, 26
};
unsigned horiz_cost[SWITCHES_PER_ROW] = {
	// one seat is 80cm wide. first switch is after 6 seats, then 18, then
	// 30. gap is 5m extra (the distribution switches are on the eastern
	// side).
	290, 194, 98, 48, 144, 240
};

// memoization table
struct key {
	unsigned char sw_ind, distro_bound, distro_bound_hard;
	unsigned char ports_left[NUM_DISTRO];

	bool operator< (const key &other) const
	{
		if (sw_ind != other.sw_ind)
			return (sw_ind < other.sw_ind);
		if (distro_bound != other.distro_bound)
			return (distro_bound < other.distro_bound);
		if (distro_bound_hard != other.distro_bound_hard)
			return (distro_bound_hard < other.distro_bound_hard);
		for (unsigned i = distro_bound_hard; i < NUM_DISTRO - 1; ++i)
			if (ports_left[i] != other.ports_left[i])
				return (ports_left[i] < other.ports_left[i]);
		return (ports_left[NUM_DISTRO - 1] < other.ports_left[NUM_DISTRO - 1]);
	}
};
struct value {
	unsigned char distro;
	unsigned cost, this_cost, this_distance;
};
std::map<key, value> cache;
unsigned cache_hits = 0;
std::vector<sw> switches;
std::map<unsigned, unsigned> num_ports_used;

unsigned find_distance(sw from_where, unsigned distro)
{
	// 3.7m from row to row (2.5m gap + 1.2m boards)
	unsigned base_cost = 37 * abs(from_where.row - distro_placements[distro]) + horiz_cost[from_where.num];
	
	// 4m, 5m, 4m gaps (1.5m, 2.5m, 1.5m extra)
	if ((from_where.row <= 9) == (distro_placements[distro] >= 10))
		base_cost += 15;
	if ((from_where.row <= 17) == (distro_placements[distro] >= 18))
		base_cost += 25;
	if ((from_where.row <= 25) == (distro_placements[distro] >= 26))
		base_cost += 15;

	// add 5.6m slack.
	return base_cost + 56;
}

unsigned cable_select(unsigned distance)
{

	if (distance > 600)
		return 1000;
	if (distance > 500)
		return FIFTYPLUSTEN;
	if (distance > 400)
		return 500;
	if (distance > 300)
		return THIRTYPLUSTEN;
	if (distance > 200)
		return 300;
	if (distance > 100)
		return TENPLUSTEN;
	return 100;
}

unsigned find_slack(unsigned distance)
{
	switch (cable_select(distance)) {
	case 1000:
		return 1000 - distance;
	case FIFTYPLUSTEN:
		return 600 - distance;
	case 500:
		return 500 - distance;
	case THIRTYPLUSTEN:
		return 400 - distance;
	case 300:
		return 300 - distance;
	case TENPLUSTEN:
		return 200 - distance;
	case 100:
		return 100 - distance;
	}
}

unsigned find_cost(unsigned distance)
{
	//return find_slack(distance);	

#if TRUNCATE_METRIC
	return cable_select(distance);
#else
	return distance;
//	return ((distance + 90) / 100) * 100;
#endif
}

unsigned find_optimal_cost(key &k)
{
	unsigned best_cost = 999999990, best_this_cost = 0, best_this_distance = 0;
	int best = -1;
	unsigned char db = k.distro_bound;
	unsigned char dbh = k.distro_bound_hard;
	
	if (k.sw_ind == switches.size())
		return 0;
	if (cache.count(k)) {
		++cache_hits;
		return cache[k].cost;
	}

	bool next_changes_row = (k.sw_ind + 1U < switches.size() && switches[k.sw_ind].row != switches[k.sw_ind + 1].row);
	
	for (unsigned char i = dbh; i < dbh + 2 && i < NUM_DISTRO; ++i) {
		if (k.ports_left[i] == 0)
			continue;
		
		unsigned distance = find_distance(switches[k.sw_ind], i);
		unsigned cost = find_cost(distance);
		
		--(k.ports_left[i]);
		++(k.sw_ind);
		
		k.distro_bound = std::max(i, db);
		if (next_changes_row) {
			k.distro_bound_hard = k.distro_bound;
		}
		
		cost += find_optimal_cost(k);
		--(k.sw_ind);
		++(k.ports_left[i]);

		if (best == -1 || cost < best_cost) {
			best = i;
			best_cost = cost;
			best_this_distance = distance;
			best_this_cost = find_cost(distance);
		}
	}
	k.distro_bound = db;
	k.distro_bound_hard = dbh;

	value v = { best, best_cost, best_this_cost, best_this_distance };
	cache[k] = v;

	return best_cost;
}
			
std::string port_name(unsigned distro, unsigned portnum)
{
	char buf[16];

	if (distro == 0) {
		int ports_per_card = 29;
		if (portnum <= ports_per_card) {
			sprintf(buf, "core1 g6/%u", portnum);
		} else if (portnum <= ports_per_card*2) {
			sprintf(buf, "core1 g7/%u", portnum - ports_per_card);
		} else {
			sprintf(buf, "core1 g8/%u", portnum - ports_per_card*2);
		}
	} else if (distro == 1) {
		int ports_per_card = 39;
		if (portnum <= ports_per_card) {
			sprintf(buf, "core2 g6/%u", portnum);
		} else if (portnum <= ports_per_card*2) {
			sprintf(buf, "core2 g7/%u", portnum - ports_per_card);
		} else {
			sprintf(buf, "core2 g8/%u", portnum - ports_per_card*2);
		}
	}

	return buf;
}

int main(int argc, char **argv)
{
	struct key start;
	FILE *patchlist = fopen("patchlist.txt", "w");
	FILE *switchlist = fopen("switches.txt", "w");
#if TRUNCATE_METRIC
	std::map<unsigned, unsigned> cable_count;
#endif
	unsigned total_slack = 0;

#if 0
	// used for placement optimization
	distro_placements[0] = atoi(argv[1]);
	distro_placements[1] = atoi(argv[2]);
	distro_placements[2] = atoi(argv[3]);
	distro_placements[3] = atoi(argv[4]);
	distro_placements[4] = atoi(argv[5]);
	distro_placements[5] = atoi(argv[6]);
	distro_placements[6] = atoi(argv[7]);
#endif

	switches.push_back(sw(1, 3));
	switches.push_back(sw(1, 4));
	switches.push_back(sw(1, 5));
	switches.push_back(sw(2, 3));
	switches.push_back(sw(2, 4));
	switches.push_back(sw(2, 5));

	for (unsigned i = 3; i <= 15; ++i)
		for (unsigned j = 0; j < SWITCHES_PER_ROW; ++j)
			switches.push_back(sw(i, j)); 

	switches.push_back(sw(16, 0));
	switches.push_back(sw(16, 1));
	switches.push_back(sw(16, 2));
	switches.push_back(sw(16, 4));
	switches.push_back(sw(16, 5));
	
	switches.push_back(sw(17, 0));
	switches.push_back(sw(17, 1));
	switches.push_back(sw(17, 2));
	switches.push_back(sw(17, 4));
	switches.push_back(sw(17, 5));
	
	for (unsigned i = 18; i <= 34; ++i)
		for (unsigned j = 0; j < SWITCHES_PER_ROW; ++j)
			switches.push_back(sw(i, j)); 
	
	switches.push_back(sw(35, 3));
	switches.push_back(sw(35, 4));
	switches.push_back(sw(35, 5));
	
	switches.push_back(sw(36, 3));
	switches.push_back(sw(36, 4));
	switches.push_back(sw(36, 5));

	// Ekstra sponsor-rad
	switches.push_back(sw(37, 3));
	//switches.push_back(sw(37, 4));

	start.sw_ind = 0;
	start.distro_bound = 0;
	start.distro_bound_hard = 0;
	start.ports_left[0] = 96 - 9;
	start.ports_left[1] = 48*3 - 5;

#if 0
	// split distro
	start.ports_left[0] = 22;
	start.ports_left[1] = 22;
	start.ports_left[2] = 42;
	start.ports_left[3] = 37;
	start.ports_left[4] = 42;
	start.ports_left[5] = 22;
	start.ports_left[6] = 22; 
#endif
	
	printf("Finding optimal layout for %u switches\n", switches.size());
	find_optimal_cost(start);
	printf("%u cache nodes, %u cache hits.\n", cache.size(), cache_hits);

	key k = start;
	int last_row = 0, last_num = -1;
	int management_address = 2;
	for (unsigned i = 0; i < switches.size(); ++i) {
		value v = cache[k];
		bool next_changes_row = (k.sw_ind + 1U < switches.size() && switches[k.sw_ind].row != switches[k.sw_ind + 1].row);

		if (cache.count(k) == 0) {
			fprintf(stderr, "FATAL: no solutions!\n");
			exit(1);
		}
		
		k.distro_bound = std::max(k.distro_bound, v.distro);
		if (next_changes_row)
			k.distro_bound_hard = k.distro_bound;
		
		if (last_row != switches[k.sw_ind].row) {
			printf("\n%2u-%2u    ", switches[k.sw_ind].row * 2 - 1, switches[k.sw_ind].row * 2 + 0 );
			last_num = -1;
		}
		for (int j = last_num; j + 1 < switches[k.sw_ind].num; ++j) {
			printf("%11s", "");
		}

		
		printf("[%um%u ", v.distro + 33, v.distro);
#if TRUNCATE_METRIC
		if (cable_select(v.this_distance) == FIFTYPLUSTEN)
			printf("(50+10)  ");
		else if (cable_select(v.this_distance) == THIRTYPLUSTEN)
			printf("(30+10)  ");
		else if (cable_select(v.this_distance) == TENPLUSTEN)
			printf("(10+10)  ");
		else
			printf("(%-3u  )  ", cable_select(v.this_distance) / 10);
		total_slack += find_slack(v.this_distance);
		cable_count[cable_select(v.this_distance)]++;
#else
		total_slack += find_slack(v.this_distance);
		printf("(%3.1f)   ", v.this_cost / 10.0);
#endif
				
		last_row = switches[k.sw_ind].row;
		last_num = switches[k.sw_ind].num;
			
		++(k.sw_ind);
		--k.ports_left[v.distro];

		fprintf(patchlist, "e%u-%u %s\n", last_row * 2 - 1, last_num + 1,
			port_name(v.distro, ++num_ports_used[v.distro]).c_str());

		switch (last_num + 1) {
		case 1:
			fprintf(switchlist, "87.76.%u.0 26 e%u-%u 87.76.242.%u\n",
	 			last_row * 2 - 1, last_row * 2 - 1, last_num + 1, management_address++);
			break;
		case 2:
			fprintf(switchlist, "87.76.%u.64 26 e%u-%u 87.76.242.%u\n",
	 			last_row * 2 - 1, last_row * 2 - 1, last_num + 1, management_address++);
			break;
		case 3:
			fprintf(switchlist, "87.76.%u.128 26 e%u-%u 87.76.242.%u\n",
	 			last_row * 2 - 1, last_row * 2 - 1, last_num + 1, management_address++);
			break;
		case 4:
			fprintf(switchlist, "87.76.%u.0 26 e%u-%u 87.76.242.%u\n",
	 			last_row * 2, last_row * 2 - 1, last_num + 1, management_address++);
			break;
		case 5:
			fprintf(switchlist, "87.76.%u.64 26 e%u-%u 87.76.242.%u\n",
	 			last_row * 2, last_row * 2 - 1, last_num + 1, management_address++);
			break;
		case 6:
			fprintf(switchlist, "87.76.%u.128 26 e%u-%u 87.76.242.%u\n",
	 			last_row * 2, last_row * 2 - 1, last_num + 1, management_address++);
			break;
		}
	}
	printf("\n");
	printf("[%um\n", 37);

#if TRUNCATE_METRIC
	cable_count[100] += cable_count[TENPLUSTEN];
	cable_count[100] += cable_count[TENPLUSTEN];

	cable_count[100] += cable_count[THIRTYPLUSTEN];
	cable_count[300] += cable_count[THIRTYPLUSTEN];

	cable_count[100] += cable_count[FIFTYPLUSTEN];
	cable_count[500] += cable_count[FIFTYPLUSTEN];

	printf("\n");
	printf("10m: %3u\n", cable_count[100]);
	printf("30m: %3u\n", cable_count[300]);
	printf("50m: %3u\n", cable_count[500]);
	printf("Extensions: %u\n", cable_count[TENPLUSTEN] + cable_count[THIRTYPLUSTEN] + cable_count[FIFTYPLUSTEN]);
	printf("\n");

	unsigned total_cable = 100 * cable_count[100] + 300 * cable_count[300] + 500 * cable_count[500];
	printf("Total cable: %.1fm\n", total_cable / 10.0);
	printf("Total slack: %.1fm (%.2f%%)\n", total_slack / 10.0, 100.0 * double(total_slack) / double(total_cable));
#endif
}
