/*
Copyright 2007 John Tsiombikas <nuclear@siggraph.org>

This file is part of the pixelshow 2007 invitation demo.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
You should have received a copy of the GNU General Public License
along with the program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <ctype.h>
#include <GL/glew.h>
#include <SDL.h>

#include "invtext.h"
#include "text.h"
#include "text_ml.h"
#include "sdr.h"

void startup_text(const char*);
void text_key_handler(int key);

static char *inv_text;
static float text_pos = 0.65, text_size;
static unsigned int fnt_reg, fnt_ital, fnt_head;
static unsigned int text_prog;
static float rect[] = {0.01, 0.25, 0.32, 1.0};

static int auto_scroll = 1;


int init_invtext(void)
{
	FILE *fp;
	long sz, tmp = 0;
	size_t bytes;
	int ati = 0;

	if(strstr((char*)glGetString(GL_VENDOR), "ATI")) {
		ati = 1;
	}

	startup_text("loading text shaders...");
	if(!(text_prog = create_program_load("sdr/text.v.glsl", ati ? "sdr/ati_text.p.glsl" : "sdr/text.p.glsl"))) {
		startup_text("... failed");
		return -1;
	}

	startup_text("loading fonts...");
	fnt_reg = create_font("data/kerkis.pfb", 20);
	fnt_ital = create_font("data/kerkis_italic.pfb", 20);
	fnt_head = create_font("data/kerkis_sans_bold.pfb", 24);
	if(!fnt_reg || !fnt_ital || !fnt_head) {
		startup_text("... failed");
		return -1;
	}
	tml_set_font(TML_FONT_REGULAR, fnt_reg);
	tml_set_font(TML_FONT_ITALIC, fnt_ital);
	tml_set_font(TML_FONT_HEADING, fnt_head);

	if(!(fp = fopen("data/inv_text", "rb"))) {
		startup_text("... failed, no invitation text available!");
		return -1;
	}
	fseek(fp, 0, SEEK_END);
	sz = ftell(fp);
	fseek(fp, 0, SEEK_SET);

	if(!(inv_text = malloc(sz + 1))) {
		startup_text("memory allocation failure");
		return -1;
	}
	while((bytes = fread(inv_text + tmp, 1, sz, fp)) > 0) {
		tmp += bytes;
	}
	fclose(fp);

	inv_text[sz] = 0;

	/* calculate text size */
	glColorMask(0, 0, 0, 0);

	tml_set_text_rect(rect[0], rect[1], rect[2], rect[3]);
	text_size = tml_print_string(inv_text);

	glColorMask(1, 1, 1, 1);

	return 0;
}

void draw_invtext(unsigned int msec)
{
	float sdoffs = 0.003;
	extern int xsz, ysz;

	if(auto_scroll) {
		text_pos = -(fmod(0.01 * (float)msec / 1000.0, 1.0) * (text_size + 1.0)) + 0.7;
	}

	glScissor(0, 0, xsz, ysz * 0.69);
	glEnable(GL_SCISSOR_TEST);

	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

	bind_program(text_prog);
	set_uniform_float(text_prog, "screen_ysz", ysz);

	set_text_color(0.43, 0.008, 0.06, 1.0);
	tml_set_text_rect(rect[0] + sdoffs, rect[1] + sdoffs + text_pos, rect[2] + sdoffs, rect[3] + sdoffs);
	tml_print_string(inv_text);

	set_text_color(0.99, 0.94, 0.0, 1.0);
	tml_set_text_rect(rect[0], rect[1] + text_pos, rect[2], rect[3]);
	tml_print_string(inv_text);

	bind_program(0);

	glDisable(GL_BLEND);

	glDisable(GL_SCISSOR_TEST);
}

void text_key_handler(int key)
{
	float txsize = text_size - 0.6;

	switch(key) {
	case SDLK_UP:
		auto_scroll = 0;
		text_pos += 0.01;
		if(text_pos > 0.1) {
			text_pos = 0.1;
		}
		break;

	case SDLK_DOWN:
		auto_scroll = 0;
		text_pos -= 0.01;
		if(text_pos < -txsize) {
			text_pos = -txsize;
		}
		break;
	}
}
