/*
This file is part of a small image library.
Copyright (C) 2004 - 2006 John Tsiombikas <nuclear@siggraph.org>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include <stdio.h>
#include <stdlib.h>
#include "image.h"

#ifdef IMGLIB_USE_PNG
int check_png(FILE *fp);
void *load_png(FILE *fp, int *xsz, int *ysz);
int save_png(FILE *fp, void *pixels, int xsz, int ysz);
#endif	/* IMGLIB_USE_PNG */

#ifdef IMGLIB_USE_JPEG
int check_jpeg(FILE *fp);
void *load_jpeg(FILE *fp, int *xsz, int *ysz);
int save_jpeg(FILE *fp, void *pixels, int xsz, int ysz);
#endif	/* IMGLIB_USE_JPEG */

#ifdef IMGLIB_USE_TGA
int check_tga(FILE *fp);
void *load_tga(FILE *fp, int *xsz, int *ysz);
int save_tga(FILE *fp, void *pixels, int xsz, int ysz);
#endif	/* IMGLIB_USE_TGA */

#ifdef IMGLIB_USE_PPM
int check_ppm(FILE *fp);
void *load_ppm(FILE *fp, int *xsz, int *ysz);
int save_ppm(FILE *fp, void *pixels, int xsz, int ysz);
#endif	/* IMGLIB_USE_PPM */

static unsigned int save_flags;


void *load_image(const char *fname, int *xsz, int *ysz) {
	FILE *file;
	void *pix = 0;

	if(!(file = fopen(fname, "rb"))) {
		fprintf(stderr, "Image loading error: could not open file %s\n", fname);
		return 0;
	}
	
#ifdef IMGLIB_USE_PNG
	if(check_png(file)) {
		pix = load_png(file, xsz, ysz);
	}
#endif	/* IMGLIB_USE_PNG */
	
#ifdef IMGLIB_USE_JPEG
	if(check_jpeg(file)) {
		pix = load_jpeg(file, xsz, ysz);
	}
#endif	/* IMGLIB_USE_JPEG */
	
#ifdef IMGLIB_USE_TGA
	if(check_tga(file)) {
		pix = load_tga(file, xsz, ysz);
	}
#endif	/* IMGLIB_USE_TGA */

#ifdef IMGLIB_USE_PPM
	if(check_ppm(file)) {
		pix = load_ppm(file, xsz, ysz);
	}
#endif	/* IMGLIB_USE_PPM */
	
	fclose(file);
	return pix;
}

void free_image(void *img) {
	free(img);
}

int save_image(const char *fname, void *pixels, int xsz, int ysz, enum image_file_format fmt) {
	FILE *fp;
	int res;

	if(!(fp = fopen(fname, "wb"))) {
		fprintf(stderr, "Image saving error: could not open file %s for writing\n", fname);
		return -1;
	}

	switch(fmt) {
	case IMG_FMT_PNG:
#ifdef IMGLIB_USE_PNG
		res = save_png(fp, pixels, xsz, ysz);
#else
		fprintf(stderr, "tried saving a png file, but png support is not compiled in");
#endif	/* IMGLIB_USE_PNG */
		break;

	case IMG_FMT_JPEG:
#ifdef IMGLIB_USE_JPEG
		res = save_jpeg(fp, pixels, xsz, ysz);
#else
		fprintf(stderr, "tried saving a jpeg file, but jpeg support is not compiled in");
#endif	/* IMGLIB_USE_JPEG */
		break;
		
	case IMG_FMT_TGA:
#ifdef IMGLIB_USE_TGA
		res = save_tga(fp, pixels, xsz, ysz);
#else
		fprintf(stderr, "tried saving a targa file, but targa support is not compiled in");
#endif	/* IMGLIB_USE_TGA */
		break;

	case IMG_FMT_PPM:
#ifdef IMGLIB_USE_PPM
		res = save_ppm(fp, pixels, xsz, ysz);
#else
		fprintf(stderr, "tried saving a ppm file, but ppm support is not compiled in");
#endif	/* IMGLIB_USE_PPM */
		break;
		
	default:
		fprintf(stderr, "Image saving error: error saving %s, invalid format specification", fname);
		res = -1;
	}

	fclose(fp);
	return res;
}


void set_image_save_flags(unsigned int flags) {
	save_flags = flags;
}

unsigned int get_image_save_flags(void) {
	return save_flags;
}
