///////////////////////////////////////////////////////////////////////
// Moira library
// Copyright (c) 2005 Camilla Berglund <camilla.berglund@gmail.com>
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any
// damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any
// purpose, including commercial applications, and to alter it and
// redistribute it freely, subject to the following restrictions:
//
//  1. The origin of this software must not be misrepresented; you
//     must not claim that you wrote the original software. If you use
//     this software in a product, an acknowledgment in the product
//     documentation would be appreciated but is not required.
//
//  2. Altered source versions must be plainly marked as such, and
//     must not be misrepresented as being the original software.
//
//  3. This notice may not be removed or altered from any source
//     distribution.
//
///////////////////////////////////////////////////////////////////////

#include <moira/Config.h>
#include <moira/Portability.h>
#include <moira/Core.h>
#include <moira/Stream.h>
#include <moira/Socket.h>

#if MOIRA_HAVE_ERRNO_H
#include <errno.h>
#endif

#if MOIRA_HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif

#if MOIRA_HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif

#if MOIRA_HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif

#if MOIRA_HAVE_NETDB_H
#include <netdb.h>
#endif

#if MOIRA_HAVE_FCNTL_H
#include <fcntl.h>
#endif

#if MOIRA_HAVE_CTYPE_H
#include <ctype.h>
#endif

#if MOIRA_HAVE_MEMORY_H
#include <memory.h>
#endif

#if MOIRA_HAVE_UNISTD_H
#include <unistd.h>
#endif

#if _WIN32

#if MOIRA_HAVE_IO_H
#include <io.h>
#endif

#if MOIRA_HAVE_WINSOCK2_H
#include <winsock2.h>
#endif

#endif /*_WIN32*/

///////////////////////////////////////////////////////////////////////

namespace moira
{
  
///////////////////////////////////////////////////////////////////////

unsigned short Connection::getClientPort(void) const
{
  return clientPort;
}

unsigned short Connection::getServerPort(void) const
{
  return serverPort;
}

const String& Connection::getClientHostName(void) const
{
  return clientHostName;
}

const String& Connection::getServerHostName(void) const
{
  return serverHostName;
}

Stream& Connection::getStream(void)
{
  return *stream;
}

Connection::Connection(void):
  clientPort(0),
  serverPort(0)
{
}

///////////////////////////////////////////////////////////////////////
    
ClientSocket::ClientSocket(void):
  blocking(true)
{
}

ClientSocket::~ClientSocket(void)
{
}

Connection* ClientSocket::connect(const String& hostName, unsigned short port)
{
  int clientSocket = ::socket(PF_INET, SOCK_STREAM, 0);
  if (clientSocket == -1)
  {
    Log::writeError("Failed to create TCP socket: %s", strerror(errno));
    return NULL;
  }
  
  sockaddr_in socketAddress;
  memset(&socketAddress, 0, sizeof(socketAddress));
  
  hostent* he = gethostbyname(hostName.c_str());
  if (he == NULL)
  {
    Log::writeError("Failed to map host %s: %s", hstrerror(h_errno));
    return NULL;
  }
  
#if HAVE_SIN_LEN
  socketAddress.sin_len = sizeof(socketAddress);
#endif
  socketAddress.sin_port = htons(port);
  socketAddress.sin_family = he->h_addrtype;
  memcpy(&socketAddress.sin_addr.s_addr, he->h_addr_list[0], he->h_length);
  
  if (::connect(clientSocket, (sockaddr*) &socketAddress, sizeof(socketAddress)) != 0)
  {
    Log::writeError("Failed to connect to %s port %u: %s",
		    hostName.c_str(), port, strerror(errno));
    return NULL;
  }
  
  if (!blocking)
  {
    int flags = fcntl(clientSocket, F_GETFL, 0);
    if (flags == -1)
    {
      Log::writeError("Failed to retrieve flags for socket %u: %s",
		      clientSocket, strerror(errno));

      shutdown(clientSocket, 2);
      close(clientSocket);
      return NULL;
    }

    if (fcntl(clientSocket, F_SETFL, flags | O_NONBLOCK) == -1)
    {
      Log::writeError("Failed to set flags for socket %u: %s",
		      clientSocket, strerror(errno));

      shutdown(clientSocket, 2);
      close(clientSocket);
      return NULL;
    }
  }

  Ptr<Connection> connection = new Connection();
  connection->clientHostName = "localhost";
  connection->clientPort = 0/* TODO: Fix this */;
  connection->serverHostName = he->h_name;
  connection->serverPort = port/* TODO: Fix this */;
  
  connection->stream = SocketStream::createInstance(clientSocket,
                                                    Stream::READABLE | Stream::WRITABLE);
  if (!connection->stream)
  {
    shutdown(clientSocket, 2);
    close(clientSocket);
    return NULL;
  }
  
  return connection.detachObject();
}
    
bool ClientSocket::isBlocking(void) const
{
  return blocking;
}

void ClientSocket::setBlocking(bool enabled)
{
  blocking = enabled;
}

///////////////////////////////////////////////////////////////////////

ServerSocket::ServerSocket(void):
  socket(-1),
  blocking(false)
{
}

Connection* ServerSocket::accept(void)
{
  sockaddr_in socketAddress;
  socklen_t addressLength = sizeof(socketAddress);
  
  int clientSocket = ::accept(socket, (struct sockaddr*) &socketAddress, &addressLength);
  if (clientSocket == -1)
  {
    Log::writeError("Failed to accept connection: %s", strerror(errno));
    return NULL;
  }
  
  Ptr<Connection> connection = new Connection();
  connection->serverHostName = "localhost";
  connection->serverPort = port/* TODO: Fix this */;

  connection->clientPort = ntohs(socketAddress.sin_port);

  hostent* host = gethostbyaddr((const char*) &(socketAddress.sin_addr), addressLength, AF_INET);
  if (host)
    connection->clientHostName = host->h_name;
  else
    connection->clientHostName = inet_ntoa(socketAddress.sin_addr);

  connection->stream = SocketStream::createInstance(clientSocket,
                                                    Stream::READABLE | Stream::WRITABLE);
  if (!connection->stream)
  {
    shutdown(clientSocket, 2);
    close(clientSocket);
    return NULL;
  }
  
  return connection.detachObject();
}

bool ServerSocket::isBlocking(void) const
{
  return blocking;
}

void ServerSocket::setBlocking(bool enabled)
{
  if (enabled == blocking)
    return;
    
  int flags = fcntl(socket, F_GETFL, 0);
  if (flags != -1)
  {
    if (enabled)
      flags &= ~O_NONBLOCK;
    else
      flags |= O_NONBLOCK;

    if (::fcntl(socket, F_SETFL, flags) != -1)
      blocking = enabled;
    else
      Log::writeError("Failed to set flags for socket %u: %s",
		      socket, strerror(errno));
  }
  else
    Log::writeError("Failed to retrieve flags for socket %u: %s",
		    socket, strerror(errno));
}

unsigned short ServerSocket::getPort(void) const
{
  return port;
}

ServerSocket* ServerSocket::createInstance(unsigned short port)
{
  Ptr<ServerSocket> socket = new ServerSocket();
  if (!socket->init(port))
    return NULL;

  return socket.detachObject();
}

ServerSocket::~ServerSocket(void)
{
  if (socket != -1)
  {
    shutdown(socket, 2);
    close(socket);
  }
}

bool ServerSocket::init(unsigned short initPort)
{
  port = initPort;

  int serverSocket = ::socket(PF_INET, SOCK_STREAM, 0);
  if (serverSocket == -1)
  {
    Log::writeError("Failed to create TCP socket: %s", strerror(errno));
    return false;
  }

  sockaddr_in socketAddress;
  memset(&socketAddress, 0, sizeof(socketAddress));
  
#if HAVE_SIN_LEN
  socketAddress.sin_len = sizeof(socketAddress);
#endif
  socketAddress.sin_family = AF_INET;
  socketAddress.sin_port = htons(port);
  socketAddress.sin_addr.s_addr = htonl(INADDR_ANY);

  if (::bind(serverSocket, (struct sockaddr*) &socketAddress, sizeof(socketAddress)) != 0)
  {
    Log::writeError("Failed to bind socket %u to port %u: %s",
		    serverSocket, port, strerror(errno));

    close(serverSocket);
    return false;
  }

  if (listen(serverSocket, 1) == -1)
  {
    Log::writeError("Failed to listen on socket %u: %s",
		    strerror(errno));
    
    close(serverSocket);
    return false;
  }
  
  int flags = fcntl(serverSocket, F_GETFL, 0);
  if (flags == -1)
  {
    Log::writeError("Failed to retrieve flags for socket %u: %s:",
		    serverSocket, strerror(errno));

    close(serverSocket);
    return false;
  }

  blocking = (flags & O_NONBLOCK) == 0;

  socket = serverSocket;
  return true;
}

///////////////////////////////////////////////////////////////////////
    
SocketStream::~SocketStream(void)
{
  fflush(file);
  shutdown(fileno(file), 2);
}

bool SocketStream::isSeekable(void) const
{
  return false;
}

SocketStream* SocketStream::createInstance(int fd, unsigned int flags)
{
  Ptr<SocketStream> stream = new SocketStream();
  if (!stream->init(fd, flags))
    return NULL;

  return stream.detachObject();
}

SocketStream::SocketStream(void)
{
}

bool SocketStream::init(int fd, unsigned int initFlags)
{
  flags = initFlags;

  String mode;
  if (!convertFlags(flags, mode))
    return false;

  file = fdopen(fd, mode.c_str());
  if (!file)
    return false;

  return true;
}

///////////////////////////////////////////////////////////////////////

} /*namespace moira*/

///////////////////////////////////////////////////////////////////////
