///////////////////////////////////////////////////////////////////////
// Moira library
// Copyright (c) 2006 Camilla Berglund <elmindreda@elmindreda.org>
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any
// damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any
// purpose, including commercial applications, and to alter it and
// redistribute it freely, subject to the following restrictions:
//
//  1. The origin of this software must not be misrepresented; you
//     must not claim that you wrote the original software. If you use
//     this software in a product, an acknowledgment in the product
//     documentation would be appreciated but is not required.
//
//  2. Altered source versions must be plainly marked as such, and
//     must not be misrepresented as being the original software.
//
//  3. This notice may not be removed or altered from any source
//     distribution.
//
///////////////////////////////////////////////////////////////////////

#include <moira/Config.h>
#include <moira/Core.h>
#include <moira/Vector.h>
#include <moira/Rectangle.h>

///////////////////////////////////////////////////////////////////////

namespace moira
{
  
///////////////////////////////////////////////////////////////////////

Rectangle::Rectangle(void)
{
}

Rectangle::Rectangle(const Vector2& initPosition, const Vector2& initSize):
  position(initPosition),
  size(initSize)
{
}

Rectangle::Rectangle(float x, float y, float width, float height):
  position(x, y),
  size(width, height)
{
}

bool Rectangle::contains(const Vector2& point) const
{
  float minX, minY, maxX, maxY;
  getBounds(minX, minY, maxX, maxY);

  if (point.x < minX || point.x > maxX)
    return false;

  if (point.y < minY || point.y > maxY)
    return false;

  return true;
}

bool Rectangle::contains(const Rectangle& other) const
{
  float minX, minY, maxX, maxY;
  getBounds(minX, minY, maxX, maxY);

  float otherMinX, otherMinY, otherMaxX, otherMaxY;
  other.getBounds(otherMinX, otherMinY, otherMaxX, otherMaxY);

  if (minX > otherMinX || maxX < otherMaxX)
    return false;
    
  if (minY > otherMinY || maxY < otherMaxY)
    return false;
    
  return true;  
}

bool Rectangle::intersects(const Rectangle& other) const
{
  float minX, minY, maxX, maxY;
  getBounds(minX, minY, maxX, maxY);

  float otherMinX, otherMinY, otherMaxX, otherMaxY;
  other.getBounds(otherMinX, otherMinY, otherMaxX, otherMaxY);

  if (minX > otherMaxX || maxX < otherMinX)
    return false;
    
  if (minY > otherMaxY || maxY < otherMinY)
    return false;
    
  return true;  
}

bool Rectangle::clipBy(const Rectangle& other)
{
  float minX, minY, maxX, maxY;
  getBounds(minX, minY, maxX, maxY);

  float otherMinX, otherMinY, otherMaxX, otherMaxY;
  other.getBounds(otherMinX, otherMinY, otherMaxX, otherMaxY);

  if (minX > otherMaxX || maxX < otherMinX)
    return false;
    
  if (minY > otherMaxY || maxY < otherMinY)
    return false;

  if (minX < otherMinX)
    minX = otherMinX;
  if (minY < otherMinY)
    minY = otherMinY;
  if (maxX > otherMaxX)
    maxX = otherMaxX;
  if (maxY > otherMaxY)
    maxY = otherMaxY;

  setBounds(minX, minY, maxX, maxY);
  return true;  
}

void Rectangle::envelop(const Rectangle& other)
{
  float minX, minY, maxX, maxY;
  getBounds(minX, minY, maxX, maxY);

  float otherMinX, otherMinY, otherMaxX, otherMaxY;
  other.getBounds(otherMinX, otherMinY, otherMaxX, otherMaxY);

  if (minX > otherMinX)
    minX = otherMinX;
  if (minY > otherMinY)
    minY = otherMinY;
  if (maxX < otherMaxX)
    maxX = otherMaxX;
  if (maxY < otherMaxY)
    maxY = otherMaxY;

  setBounds(minX, minY, maxX, maxY);
}

void Rectangle::normalize(void)
{
  if (size.x < 0.f)
  {
    position.x += size.x;
    size.x = -size.x;
  }

  if (size.y < 0.f)
  {
    position.y += size.y;
    size.y = -size.y;
  }
}

bool Rectangle::operator == (const Rectangle& other) const
{
  return position == other.position && size == other.size;
}

bool Rectangle::operator != (const Rectangle& other) const
{
  return position != other.position || size != other.size;
}

Rectangle Rectangle::operator + (const Vector2& offset) const
{
  return Rectangle(position + offset, size);
}

Rectangle Rectangle::operator - (const Vector2& offset) const
{
  return Rectangle(position - offset, size);
}

Rectangle Rectangle::operator * (const Vector2& scale) const
{
  return Rectangle(position * scale, size * scale);
}

Rectangle& Rectangle::operator += (const Vector2& offset)
{
  position += offset;
  return *this;
}

Rectangle& Rectangle::operator -= (const Vector2& offset)
{
  position -= offset;
  return *this;
}

Rectangle& Rectangle::operator *= (const Vector2& scale)
{
  position *= scale;
  size *= scale;
  return *this;
}

Vector2 Rectangle::getCenter(void) const
{
  return position + size / 2.f;
}

void Rectangle::getBounds(float& minX, float& minY, float& maxX, float& maxY) const
{
  minX = position.x;
  minY = position.y;

  maxX = position.x + size.x;
  maxY = position.y + size.y;

  if (minX > maxX)
  {
    float quux = minX;
    minX = maxX;
    maxX = quux;
  }

  if (minY > maxY)
  {
    float quux = minY;
    minY = maxY;
    maxY = quux;
  }
}

void Rectangle::setBounds(float minX, float minY, float maxX, float maxY)
{
  position.set(minX, minY);
  size.set(maxX - minX, maxY - minY);
}

void Rectangle::set(const Vector2& newPosition, const Vector2& newSize)
{
  position = newPosition;
  size = newSize;
}

void Rectangle::set(float x, float y, float width, float height)
{
  position.set(x, y);
  size.set(width, height);
}

///////////////////////////////////////////////////////////////////////

RectangleClipStack::RectangleClipStack(void)
{
}

bool RectangleClipStack::push(const Rectangle& rectangle)
{
  Entry entry;
  entry.total = rectangle;
  entry.local = rectangle;

  if (!entries.empty())
  {
    if (!entry.total.clipBy(entries.top().total))
      return false;
  }

  entries.push(entry);
  return true;
}

void RectangleClipStack::pop(void)
{
  if (entries.empty())
    throw Exception("Cannot pop empty scissor area clip stack");

  entries.pop();
}

bool RectangleClipStack::isEmpty(void) const
{
  return entries.empty();
}

unsigned int RectangleClipStack::getCount(void) const
{
  return (unsigned int) entries.size();
}

const Rectangle& RectangleClipStack::getTop(void) const
{
  if (entries.empty())
    throw Exception("Rectangle clip stack is empty");

  return entries.top().local;
}

const Rectangle& RectangleClipStack::getTotal(void) const
{
  if (entries.empty())
    throw Exception("Rectangle clip stack is empty");

  return entries.top().total;
}

///////////////////////////////////////////////////////////////////////

} /*namespace moira*/

///////////////////////////////////////////////////////////////////////
