///////////////////////////////////////////////////////////////////////
// Moira library
// Copyright (c) 2006 Camilla Berglund <elmindreda@elmindreda.org>
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any
// damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any
// purpose, including commercial applications, and to alter it and
// redistribute it freely, subject to the following restrictions:
//
//  1. The origin of this software must not be misrepresented; you
//     must not claim that you wrote the original software. If you use
//     this software in a product, an acknowledgment in the product
//     documentation would be appreciated but is not required.
//
//  2. Altered source versions must be plainly marked as such, and
//     must not be misrepresented as being the original software.
//
//  3. This notice may not be removed or altered from any source
//     distribution.
//
///////////////////////////////////////////////////////////////////////

#include <moira/Config.h>
#include <moira/Portability.h>
#include <moira/Core.h>
#include <moira/Vector.h>
#include <moira/Stream.h>
#include <moira/XML.h>
#include <moira/Resource.h>
#include <moira/Mesh.h>
#include <moira/MeshIO.h>

///////////////////////////////////////////////////////////////////////

namespace moira
{
  
///////////////////////////////////////////////////////////////////////

namespace
{

const unsigned int MESH_XML_VERSION = 1;

}

///////////////////////////////////////////////////////////////////////

MeshCodecXML::MeshCodecXML(void):
  MeshCodec("XML mesh codec")
{
  addSuffix("mesh");
}

Mesh* MeshCodecXML::read(const Path& path, const String& name)
{
  return MeshCodec::read(path, name);
}

Mesh* MeshCodecXML::read(Stream& stream, const String& name)
{
  currentGeometry = NULL;
  currentTriangle = NULL;
  currentVertex = NULL;

  meshName = name;

  if (!XML::Codec::read(stream))
    return NULL;

  return mesh.detachObject();
}

bool MeshCodecXML::write(const Path& path, const Mesh& mesh)
{
  return MeshCodec::write(path, mesh);
}

bool MeshCodecXML::write(Stream& stream, const Mesh& mesh)
{
  try
  {
    setStream(&stream);

    beginElement("mesh");
    addAttribute("version", MESH_XML_VERSION);

    for (unsigned int i = 0;  i < mesh.geometries.size();  i++)
    {
      const MeshGeometry& geometry = mesh.geometries[i];

      beginElement("geometry");
      addAttribute("shader", geometry.shaderName);

      for (unsigned int j = 0;  j < geometry.triangles.size();  j++)
      {
	const MeshTriangle& triangle = geometry.triangles[j];

        beginElement("triangle");
        addAttribute("a", (int) triangle.indices[0]); 
        addAttribute("b", (int) triangle.indices[1]); 
        addAttribute("c", (int) triangle.indices[2]); 

        beginElement("normal");
        addAttributes(triangle.normal);
        endElement();

        endElement();
      }

      endElement();
    }

    for (unsigned int i = 0;  i < mesh.vertices.size();  i++)
    {
      const MeshVertex& vertex = mesh.vertices[i];

      beginElement("vertex");
      addAttributes(vertex.position);

      beginElement("normal");
      addAttributes(vertex.normal);
      endElement();

      beginElement("texcoord");
      addAttributes(vertex.texcoord);
      endElement();

      endElement();
    }

    endElement();

    setStream(NULL);
  }
  catch (Exception& exception)
  {
    Log::writeError("Failed to write mesh %s: %s", mesh.getName().c_str(), exception.what());
    setStream(NULL);
    return false;
  }
  
  return true;
}

bool MeshCodecXML::onBeginElement(const String& name)
{
  if (name == "mesh")
  {
    mesh = new Mesh(meshName);

    const unsigned int version = readInteger("version");
    if (version != MESH_XML_VERSION)
    {
      Log::writeError("Mesh XML format version mismatch");
      return false;
    }

    return true;
  }

  if (mesh)
  {
    if (name == "vertex")
    {
      mesh->vertices.push_back(MeshVertex());
      currentVertex = &(mesh->vertices.back());
      readAttributes(currentVertex->position);
      return true;
    }

    if (name == "geometry")
    {
      mesh->geometries.push_back(MeshGeometry());
      currentGeometry = &(mesh->geometries.back());
      currentGeometry->shaderName = readString("shader");
      return true;
    }

    if (currentVertex)
    {
      if (name == "normal")
      {
        readAttributes(currentVertex->normal);
        return true;
      }

      if (name == "texcoord")
      {
	readAttributes(currentVertex->texcoord);
	return true;
      }
    }

    if (currentGeometry)
    {
      if (name == "triangle")
      {
        currentGeometry->triangles.push_back(MeshTriangle());
        currentTriangle = &(currentGeometry->triangles.back());
        currentTriangle->indices[0] = readInteger("a");
        currentTriangle->indices[1] = readInteger("b");
        currentTriangle->indices[2] = readInteger("c");
        return true;
      }

      if (currentTriangle)
      {
        if (name == "normal")
        {
          readAttributes(currentTriangle->normal);
          return true;
        }
      }
    }
  }
  
  return true;
}

bool MeshCodecXML::onEndElement(const String& name)
{
  if (name == "vertex")
    currentVertex = NULL;

  if (name == "geometry")
    currentGeometry = NULL;

  if (name == "triangle")
    currentTriangle = NULL;

  return true;
}

///////////////////////////////////////////////////////////////////////

} /*namespace moira*/

///////////////////////////////////////////////////////////////////////
