///////////////////////////////////////////////////////////////////////
// Moira library
// Copyright (c) 2005 Camilla Berglund <elmindreda@elmindreda.org>
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any
// damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any
// purpose, including commercial applications, and to alter it and
// redistribute it freely, subject to the following restrictions:
//
//  1. The origin of this software must not be misrepresented; you
//     must not claim that you wrote the original software. If you use
//     this software in a product, an acknowledgment in the product
//     documentation would be appreciated but is not required.
//
//  2. Altered source versions must be plainly marked as such, and
//     must not be misrepresented as being the original software.
//
//  3. This notice may not be removed or altered from any source
//     distribution.
//
///////////////////////////////////////////////////////////////////////

#include <moira/Config.h>
#include <moira/Portability.h>
#include <moira/Vector.h>
#include <moira/Quaternion.h>
#include <moira/Matrix.h>

///////////////////////////////////////////////////////////////////////

namespace moira
{
  
///////////////////////////////////////////////////////////////////////

Matrix2::Matrix2(void)
{
  setIdentity();
}

Matrix2::Matrix2(const Vector2& sx, const Vector2& sy):
  x(sx),
  y(sy)
{
}

void Matrix2::invert(void)
{
  // TODO: Implement.
}

void Matrix2::transpose(void)
{
  float temp;

  temp = x.y; x.y = y.x; y.x = temp;
}

void Matrix2::normalize(void)
{
  // TODO: Implement.
}

void Matrix2::concatenate(const Matrix2& matrix)
{
  Matrix2 result;

  result.x.x = x.x * matrix.x.x + y.x * matrix.x.y;
  result.y.x = x.x * matrix.y.x + y.x * matrix.y.y;

  result.x.y = x.y * matrix.x.x + y.y * matrix.x.y;
  result.y.y = x.y * matrix.y.x + y.y * matrix.y.y;

  operator = (result);
}

Matrix2::operator float* (void)
{
  return &(x.x);
}

Matrix2::operator const float* (void) const
{
  return &(x.x);
}

float Matrix2::operator () (unsigned int column, unsigned int row) const
{
  return (&(x.x))[column * 2 + row];
}

float& Matrix2::operator () (unsigned int column, unsigned int row)
{
  return (&(x.x))[column * 2 + row];
}

Matrix2 Matrix2::operator + (const Matrix2& matrix) const
{
  return Matrix2(x + matrix.x, y + matrix.y);
}

Matrix2 Matrix2::operator - (const Matrix2& matrix) const
{
  return Matrix2(x - matrix.x, y - matrix.y);
}

Matrix2 Matrix2::operator * (const Matrix2& matrix) const
{
  Matrix2 result;

  result.x.x = x.x * matrix.x.x + y.x * matrix.x.y;
  result.y.x = x.x * matrix.y.x + y.x * matrix.y.y;

  result.x.y = x.y * matrix.x.x + y.y * matrix.x.y;
  result.y.y = x.y * matrix.y.x + y.y * matrix.y.y;

  return result;
}

Matrix2& Matrix2::operator += (const Matrix2& matrix)
{
  x += matrix.x;
  y += matrix.y;

  return *this;
}

Matrix2& Matrix2::operator -= (const Matrix2& matrix)
{
  x -= matrix.x;
  y -= matrix.y;

  return *this;
}

Matrix2& Matrix2::operator *= (const Matrix2& matrix)
{
  Matrix2 result;

  result.x.x = x.x * matrix.x.x + y.x * matrix.x.y;
  result.y.x = x.x * matrix.y.x + y.x * matrix.y.y;

  result.x.y = x.y * matrix.x.x + y.y * matrix.x.y;
  result.y.y = x.y * matrix.y.x + y.y * matrix.y.y;

  return operator = (result);
}

void Matrix2::rotateVector(Vector2& vector) const
{
  Vector2 temp;

  temp.x = vector.x * x.x + vector.y * y.x;
  temp.y = vector.x * x.y + vector.y * y.y;

  vector = temp;
}

void Matrix2::setIdentity(void)
{
  x.set(1.f, 0.f);
  y.set(0.f, 1.f);
}

void Matrix2::set(const Vector2& sx, const Vector2& sy)
{
  x = sx;
  y = sy;
}

void Matrix2::setRotation(const float angle)
{
  const float sina = sinf(angle);
  const float cosa = cosf(angle);

  x.x = cosa;
  x.y = sina;
  y.x = -sina;
  y.y = cosa;
}

///////////////////////////////////////////////////////////////////////

Matrix3::Matrix3(void)
{
  setIdentity();
}

Matrix3::Matrix3(const Vector3& sx, const Vector3& sy, const Vector3& sz):
  x(sx),
  y(sy),
  z(sz)
{
}

void Matrix3::invert(void)
{
  // TODO: Implement.
}

void Matrix3::transpose(void)
{
  float temp;

  temp = x.y; x.y = y.x; y.x = temp;
  temp = x.z; x.z = z.x; z.x = temp;
  temp = y.z; y.z = z.y; z.y = temp;
}

void Matrix3::normalize(void)
{
  // TODO: Implement.
}

void Matrix3::concatenate(const Matrix3& matrix)
{
  Matrix3 result;

  result.x.x = x.x * matrix.x.x + y.x * matrix.x.y + z.x * matrix.x.z;
  result.y.x = x.x * matrix.y.x + y.x * matrix.y.y + z.x * matrix.y.z;
  result.z.x = x.x * matrix.z.x + y.x * matrix.z.y + z.x * matrix.z.z;

  result.x.y = x.y * matrix.x.x + y.y * matrix.x.y + z.y * matrix.x.z;
  result.y.y = x.y * matrix.y.x + y.y * matrix.y.y + z.y * matrix.y.z;
  result.z.y = x.y * matrix.z.x + y.y * matrix.z.y + z.y * matrix.z.z;

  result.x.z = x.z * matrix.x.x + y.z * matrix.x.y + z.z * matrix.x.z;
  result.y.z = x.z * matrix.y.x + y.z * matrix.y.y + z.z * matrix.y.z;
  result.z.z = x.z * matrix.z.x + y.z * matrix.z.y + z.z * matrix.z.z;

  operator = (result);
}

Matrix3::operator float* (void)
{
  return &(x.x);
}

Matrix3::operator const float* (void) const
{
  return &(x.x);
}

float Matrix3::operator () (unsigned int column, unsigned int row) const
{
  return (&(x.x))[column * 3 + row];
}

float& Matrix3::operator () (unsigned int column, unsigned int row)
{
  return (&(x.x))[column * 3 + row];
}

Matrix3 Matrix3::operator + (const Matrix3& matrix) const
{
  return Matrix3(x + matrix.x, y + matrix.y, z + matrix.z);
}

Matrix3 Matrix3::operator - (const Matrix3& matrix) const
{
  return Matrix3(x - matrix.x, y - matrix.y, z - matrix.z);
}

Matrix3 Matrix3::operator * (const Matrix3& matrix) const
{
  Matrix3 result;

  result.x.x = x.x * matrix.x.x + y.x * matrix.x.y + z.x * matrix.x.z;
  result.y.x = x.x * matrix.y.x + y.x * matrix.y.y + z.x * matrix.y.z;
  result.z.x = x.x * matrix.z.x + y.x * matrix.z.y + z.x * matrix.z.z;

  result.x.y = x.y * matrix.x.x + y.y * matrix.x.y + z.y * matrix.x.z;
  result.y.y = x.y * matrix.y.x + y.y * matrix.y.y + z.y * matrix.y.z;
  result.z.y = x.y * matrix.z.x + y.y * matrix.z.y + z.y * matrix.z.z;

  result.x.z = x.z * matrix.x.x + y.z * matrix.x.y + z.z * matrix.x.z;
  result.y.z = x.z * matrix.y.x + y.z * matrix.y.y + z.z * matrix.y.z;
  result.z.z = x.z * matrix.z.x + y.z * matrix.z.y + z.z * matrix.z.z;

  return result;
}

Matrix3& Matrix3::operator += (const Matrix3& matrix)
{
  x += matrix.x;
  y += matrix.y;
  z += matrix.z;

  return *this;
}

Matrix3& Matrix3::operator -= (const Matrix3& matrix)
{
  x -= matrix.x;
  y -= matrix.y;
  z -= matrix.z;

  return *this;
}

Matrix3& Matrix3::operator *= (const Matrix3& matrix)
{
  Matrix3 result;

  result.x.x = x.x * matrix.x.x + y.x * matrix.x.y + z.x * matrix.x.z;
  result.y.x = x.x * matrix.y.x + y.x * matrix.y.y + z.x * matrix.y.z;
  result.z.x = x.x * matrix.z.x + y.x * matrix.z.y + z.x * matrix.z.z;

  result.x.y = x.y * matrix.x.x + y.y * matrix.x.y + z.y * matrix.x.z;
  result.y.y = x.y * matrix.y.x + y.y * matrix.y.y + z.y * matrix.y.z;
  result.z.y = x.y * matrix.z.x + y.y * matrix.z.y + z.y * matrix.z.z;

  result.x.z = x.z * matrix.x.x + y.z * matrix.x.y + z.z * matrix.x.z;
  result.y.z = x.z * matrix.y.x + y.z * matrix.y.y + z.z * matrix.y.z;
  result.z.z = x.z * matrix.z.x + y.z * matrix.z.y + z.z * matrix.z.z;

  return operator = (result);
}

void Matrix3::rotateVector(Vector3& vector) const
{
  Vector3 temp;

  temp.x = vector.x * x.x + vector.y * y.x + vector.z * z.x;
  temp.y = vector.x * x.y + vector.y * y.y + vector.z * z.y;
  temp.z = vector.x * x.z + vector.y * y.z + vector.z * z.z;

  vector = temp;
}

void Matrix3::setIdentity(void)
{
  x.set(1.f, 0.f, 0.f);
  y.set(0.f, 1.f, 0.f);
  z.set(0.f, 0.f, 1.f);
}

void Matrix3::set(const Vector3& sx, const Vector3& sy, const Vector3& sz)
{
  x = sx;
  y = sy;
  z = sz;
}

void Matrix3::setQuatRotation(const Quaternion& quat)
{
  x.x = 1.f - 2.f * quat.y * quat.y - 2.f * quat.z * quat.z;
  x.y = 2.f * quat.x * quat.y + 2.f * quat.w * quat.z;
  x.z = 2.f * quat.x * quat.z - 2.f * quat.w * quat.y;

  y.x = 2.f * quat.x * quat.y - 2.f * quat.w * quat.z;
  y.y = 1.f - 2.f * quat.x * quat.x - 2.f * quat.z * quat.z;
  y.z = 2.f * quat.y * quat.z + 2.f * quat.w * quat.x;

  z.x = 2.f * quat.x * quat.z + 2.f * quat.w * quat.y;
  z.y = 2.f * quat.y * quat.z - 2.f * quat.w * quat.x;
  z.z = 1.f - 2.f * quat.x * quat.x - 2.f * quat.y * quat.y;
}

void Matrix3::setEulerRotation(const Vector3& angles)
{
  // TODO: Implement.
}

void Matrix3::setVectorRotation(const Vector3& vector)
{
  // NOTE: This code sucks.

  Vector3 origin(0.f, 0.f, 1.f);

  const float dot = origin.dotProduct(vector);

  if (fabsf(dot) < 0.9999f)
  {
    Vector3 axis = origin.crossProduct(vector).normalize();
    setAxisRotation(axis, acosf(vector.dotProduct(origin)));
  }
  else if (dot < 0.f)
  {
    const Vector3 axis(1.f, 0.f, 0.f);
    setAxisRotation(axis, (float) M_PI);
  }
  else
    setIdentity();
}

void Matrix3::setAxisRotation(const Vector3& axis, float angle)
{
  const float cosAngle = cosf(angle);
  const float sinAngle = sinf(angle);

  const float oneMinusCos = 1.f - cosAngle;

  const float x2 = axis.x * axis.x;
  const float y2 = axis.y * axis.y;
  const float z2 = axis.z * axis.z;
  const float xy = axis.x * axis.y * oneMinusCos;
  const float xz = axis.x * axis.z * oneMinusCos;
  const float yz = axis.y * axis.z * oneMinusCos;
  const float xs = axis.x * sinAngle;
  const float ys = axis.y * sinAngle;
  const float zs = axis.z * sinAngle;

  x.x = x2 * oneMinusCos + cosAngle;
  y.x = xy - zs;
  z.x = xz + ys;
  x.y = xy + zs;
  y.y = y2 * oneMinusCos + cosAngle;
  z.y = yz - xs;
  x.z = xz - ys;
  y.z = yz + xs;
  z.z = z2 * oneMinusCos + cosAngle;
}

///////////////////////////////////////////////////////////////////////

Matrix4::Matrix4(void)
{
  setIdentity();
}

Matrix4::Matrix4(const Vector4& sx, const Vector4& sy, const Vector4& sz, const Vector4& sw):
  x(sx),
  y(sy),
  z(sz),
  w(sw)
{
}

void Matrix4::invert(void)
{
  // TODO: implement!
}

void Matrix4::transpose(void)
{
  float temp;

  temp = x.y; x.y = y.x; y.x = temp;
  temp = x.z; x.z = z.x; z.x = temp;
  temp = x.w; x.w = w.x; w.x = temp;
  temp = y.z; y.z = z.y; z.y = temp;
  temp = y.w; y.w = w.y; w.y = temp;
  temp = z.w; z.w = w.z; w.z = temp;
}

void Matrix4::normalize(void)
{
  // TODO: implement!
}

void Matrix4::concatenate(const Matrix4& matrix)
{
  Matrix4 result;

  result.x.x = x.x * matrix.x.x + y.x * matrix.x.y + z.x * matrix.x.z + w.x * matrix.x.w;
  result.y.x = x.x * matrix.y.x + y.x * matrix.y.y + z.x * matrix.y.z + w.x * matrix.y.w;
  result.z.x = x.x * matrix.z.x + y.x * matrix.z.y + z.x * matrix.z.z + w.x * matrix.z.w;
  result.w.x = x.x * matrix.w.x + y.x * matrix.w.y + z.x * matrix.w.z + w.x * matrix.w.w;

  result.x.y = x.y * matrix.x.x + y.y * matrix.x.y + z.y * matrix.x.z + w.y * matrix.x.w;
  result.y.y = x.y * matrix.y.x + y.y * matrix.y.y + z.y * matrix.y.z + w.y * matrix.y.w;
  result.z.y = x.y * matrix.z.x + y.y * matrix.z.y + z.y * matrix.z.z + w.y * matrix.z.w;
  result.w.y = x.y * matrix.w.x + y.y * matrix.w.y + z.y * matrix.w.z + w.y * matrix.w.w;

  result.x.z = x.z * matrix.x.x + y.z * matrix.x.y + z.z * matrix.x.z + w.z * matrix.x.w;
  result.y.z = x.z * matrix.y.x + y.z * matrix.y.y + z.z * matrix.y.z + w.z * matrix.y.w;
  result.z.z = x.z * matrix.z.x + y.z * matrix.z.y + z.z * matrix.z.z + w.z * matrix.z.w;
  result.w.z = x.z * matrix.w.x + y.z * matrix.w.y + z.z * matrix.w.z + w.z * matrix.w.w;

  result.x.w = x.w * matrix.x.x + y.w * matrix.x.y + z.w * matrix.x.z + w.w * matrix.x.w;
  result.y.w = x.w * matrix.y.x + y.w * matrix.y.y + z.w * matrix.y.z + w.w * matrix.y.w;
  result.z.w = x.w * matrix.z.x + y.w * matrix.z.y + z.w * matrix.z.z + w.w * matrix.z.w;
  result.w.w = x.w * matrix.w.x + y.w * matrix.w.y + z.w * matrix.w.z + w.w * matrix.w.w;

  operator = (result);
}

void Matrix4::transformVector(Vector3& vector) const
{
  Vector3 temp;

  temp.x = vector.x * x.x + vector.y * y.x + vector.z * z.x + w.x;
  temp.y = vector.x * x.y + vector.y * y.y + vector.z * z.y + w.y;
  temp.z = vector.x * x.z + vector.y * y.z + vector.z * z.z + w.z;

  vector = temp;
}

void Matrix4::transformVector(Vector4& vector) const
{
  Vector4 temp;

  temp.x = vector.x * x.x + vector.y * y.x + vector.z * z.x + vector.w * w.x;
  temp.y = vector.x * x.y + vector.y * y.y + vector.z * z.y + vector.w * w.y;
  temp.z = vector.x * x.z + vector.y * y.z + vector.z * z.z + vector.w * w.z;
  temp.w = vector.x * x.w + vector.y * y.w + vector.z * z.w + vector.w * w.w;

  vector = temp;
}

void Matrix4::translateVector(Vector3& vector) const
{
  vector.x += w.x;
  vector.y += w.y;
  vector.z += w.z;
}

void Matrix4::rotateVector(Vector3& vector) const
{
  Vector3 temp;

  temp.x = vector.x * x.x + vector.y * y.x + vector.z * z.x;
  temp.y = vector.x * x.y + vector.y * y.y + vector.z * z.y;
  temp.z = vector.x * x.z + vector.y * y.z + vector.z * z.z;

  vector = temp;
}

Matrix4::operator float* (void)
{
  return &(x.x);
}

Matrix4::operator const float* (void) const
{
  return &(x.x);
}

float Matrix4::operator () (unsigned int column, unsigned int row) const
{
  return (&(x.x))[column * 4 + row];
}

float& Matrix4::operator () (unsigned int column, unsigned int row)
{
  return (&(x.x))[column * 4 + row];
}

Matrix4 Matrix4::operator + (const Matrix4& matrix) const
{
  return Matrix4(x + matrix.x, y + matrix.y, z + matrix.z, w + matrix.w);
}

Matrix4 Matrix4::operator - (const Matrix4& matrix) const
{
  return Matrix4(x - matrix.x, y - matrix.y, z - matrix.z, w - matrix.w);
}

Matrix4 Matrix4::operator * (const Matrix4& matrix) const
{
  Matrix4 result;

  result.x.x = x.x * matrix.x.x + y.x * matrix.x.y + z.x * matrix.x.z;
  result.y.x = x.x * matrix.y.x + y.x * matrix.y.y + z.x * matrix.y.z;
  result.z.x = x.x * matrix.z.x + y.x * matrix.z.y + z.x * matrix.z.z;
  result.w.x = x.x * matrix.w.x + y.x * matrix.w.y + z.x * matrix.w.z + w.x;

  result.x.y = x.y * matrix.x.x + y.y * matrix.x.y + z.y * matrix.x.z;
  result.y.y = x.y * matrix.y.x + y.y * matrix.y.y + z.y * matrix.y.z;
  result.z.y = x.y * matrix.z.x + y.y * matrix.z.y + z.y * matrix.z.z;
  result.w.y = x.y * matrix.w.x + y.y * matrix.w.y + z.y * matrix.w.z + w.y;

  result.x.z = x.z * matrix.x.x + y.z * matrix.x.y + z.z * matrix.x.z;
  result.y.z = x.z * matrix.y.x + y.z * matrix.y.y + z.z * matrix.y.z;
  result.z.z = x.z * matrix.z.x + y.z * matrix.z.y + z.z * matrix.z.z;
  result.w.z = x.z * matrix.w.x + y.z * matrix.w.y + z.z * matrix.w.z + w.z;

  result.x.w = 0.f;
  result.y.w = 0.f;
  result.z.w = 0.f;
  result.w.w = 1.f;

  return result;
}

Matrix4& Matrix4::operator += (const Matrix4& matrix)
{
  x += matrix.x;
  y += matrix.y;
  z += matrix.z;
  w += matrix.w;

  return *this;
}

Matrix4& Matrix4::operator -= (const Matrix4& matrix)
{
  x -= matrix.x;
  y -= matrix.y;
  z -= matrix.z;
  w -= matrix.w;

  return *this;
}

Matrix4& Matrix4::operator *= (const Matrix4& matrix)
{
  Matrix4 result;

  result.x.x = x.x * matrix.x.x + y.x * matrix.x.y + z.x * matrix.x.z;
  result.y.x = x.x * matrix.y.x + y.x * matrix.y.y + z.x * matrix.y.z;
  result.z.x = x.x * matrix.z.x + y.x * matrix.z.y + z.x * matrix.z.z;
  result.w.x = x.x * matrix.w.x + y.x * matrix.w.y + z.x * matrix.w.z + w.x;

  result.x.y = x.y * matrix.x.x + y.y * matrix.x.y + z.y * matrix.x.z;
  result.y.y = x.y * matrix.y.x + y.y * matrix.y.y + z.y * matrix.y.z;
  result.z.y = x.y * matrix.z.x + y.y * matrix.z.y + z.y * matrix.z.z;
  result.w.y = x.y * matrix.w.x + y.y * matrix.w.y + z.y * matrix.w.z + w.y;

  result.x.z = x.z * matrix.x.x + y.z * matrix.x.y + z.z * matrix.x.z;
  result.y.z = x.z * matrix.y.x + y.z * matrix.y.y + z.z * matrix.y.z;
  result.z.z = x.z * matrix.z.x + y.z * matrix.z.y + z.z * matrix.z.z;
  result.w.z = x.z * matrix.w.x + y.z * matrix.w.y + z.z * matrix.w.z + w.z;

  result.x.w = 0.f;
  result.y.w = 0.f;
  result.z.w = 0.f;
  result.w.w = 1.f;

  return operator = (result);
}

void Matrix4::setIdentity(void)
{
  x.set(1.f, 0.f, 0.f, 0.f);
  y.set(0.f, 1.f, 0.f, 0.f);
  z.set(0.f, 0.f, 1.f, 0.f);
  w.set(0.f, 0.f, 0.f, 1.f);
}

void Matrix4::set(const Vector4& sx, const Vector4& sy, const Vector4& sz, const Vector4& sw)
{
  x = sx;
  y = sy;
  z = sz;
  w = sw;
}

void Matrix4::setEulerRotation(const Vector3& angles)
{
  // TODO: implement!
  /*
  Vector3 sin, cos;

  sin.set(sinf(angles.x), sinf(angles.y), sinf(angles.z));
  cos.set(cosf(angles.x), cosf(angles.y), cosf(angles.z));

  x.x = cos.y * cos.z;
  x.y = sin.x * sin.y * cos.z - cos.x * sin.z;
  x.z = cos.x * sin.y * cos.z + sin.x * sin.z;

  y.x = cos.y * sin.z;
  y.y = cos.x * cos.z + sin.x * sin.y * sin.z;
  y.z = cos.x * sin.y * sin.z - sin.x * cos.z;

  z.x = -sin.y;
  z.y = sin.x * cos.y;
  z.z = cos.x * cos.y;
  */
}

void Matrix4::setQuatRotation(const Quaternion& quat)
{
  x.x = 1.f - 2.f * quat.y * quat.y - 2.f * quat.z * quat.z;
  x.y = 2.f * quat.x * quat.y + 2.f * quat.w * quat.z;
  x.z = 2.f * quat.x * quat.z - 2.f * quat.w * quat.y;

  y.x = 2.f * quat.x * quat.y - 2.f * quat.w * quat.z;
  y.y = 1.f - 2.f * quat.x * quat.x - 2.f * quat.z * quat.z;
  y.z = 2.f * quat.y * quat.z + 2.f * quat.w * quat.x;

  z.x = 2.f * quat.x * quat.z + 2.f * quat.w * quat.y;
  z.y = 2.f * quat.y * quat.z - 2.f * quat.w * quat.x;
  z.z = 1.f - 2.f * quat.x * quat.x - 2.f * quat.y * quat.y;
}

void Matrix4::setVectorRotation(const Vector3& vector)
{
  // NOTE: This code sucks.

  Vector3 origin(0.f, 0.f, 1.f);

  const float dot = origin.dotProduct(vector);

  if (fabsf(dot) < 0.9999f)
  {
    Vector3 axis = origin.crossProduct(vector).normalize();
    setAxisRotation(axis, acosf(vector.dotProduct(origin)));
  }
  else if (dot < 0.f)
  {
    const Vector3 axis(1.f, 0.f, 0.f);
    setAxisRotation(axis, (float) M_PI);
  }
  else
    setIdentity();
}

void Matrix4::setAxisRotation(const Vector3& axis, float angle)
{
  const float cosAngle = cosf(angle);
  const float sinAngle = sinf(angle);

  const float oneMinusCos = 1.f - cosAngle;

  const float x2 = axis.x * axis.x;
  const float y2 = axis.y * axis.y;
  const float z2 = axis.z * axis.z;
  const float xy = axis.x * axis.y * oneMinusCos;
  const float xz = axis.x * axis.z * oneMinusCos;
  const float yz = axis.y * axis.z * oneMinusCos;
  const float xs = axis.x * sinAngle;
  const float ys = axis.y * sinAngle;
  const float zs = axis.z * sinAngle;

  x.x = x2 * oneMinusCos + cosAngle;
  y.x = xy - zs;
  z.x = xz + ys;
  x.y = xy + zs;
  y.y = y2 * oneMinusCos + cosAngle;
  z.y = yz - xs;
  x.z = xz - ys;
  y.z = yz + xs;
  z.z = z2 * oneMinusCos + cosAngle;
}

void Matrix4::getMatrixRotation(Matrix3& matrix) const
{
  matrix.x.x = x.x;
  matrix.x.y = x.y;
  matrix.x.z = x.z;

  matrix.y.x = y.x;
  matrix.y.y = y.y;
  matrix.y.z = y.z;

  matrix.z.x = z.x;
  matrix.z.y = z.y;
  matrix.z.z = z.z;
}

void Matrix4::setMatrixRotation(const Matrix3& matrix)
{
  x.x = matrix.x.x;
  x.y = matrix.x.y;
  x.z = matrix.x.z;

  y.x = matrix.y.x;
  y.y = matrix.y.y;
  y.z = matrix.y.z;

  z.x = matrix.z.x;
  z.y = matrix.z.y;
  z.z = matrix.z.z;
}

void Matrix4::getScaling(Vector3& scaling) const
{
  Vector3 axis;

  axis.x = x.x;
  axis.y = y.x;
  axis.z = z.x;

  scaling.x = axis.length();

  axis.x = x.y;
  axis.y = y.y;
  axis.z = z.y;

  scaling.y = axis.length();

  axis.x = x.z;
  axis.y = y.z;
  axis.z = z.z;

  scaling.z = axis.length();
}

void Matrix4::setScaling(const Vector3& scaling)
{
  x.x = scaling.x;
  y.y = scaling.y;
  z.z = scaling.z;
}

void Matrix4::getTranslation(Vector3& vector) const
{
  vector.x = w.x;
  vector.y = w.y;
  vector.z = w.z;
}

void Matrix4::setTranslation(const Vector3& vector)
{
  w.x = vector.x;
  w.y = vector.y;
  w.z = vector.z;
}

///////////////////////////////////////////////////////////////////////

} /*namespace moira*/

///////////////////////////////////////////////////////////////////////
