///////////////////////////////////////////////////////////////////////
// Moira library
// Copyright (c) 2005 Camilla Berglund <elmindreda@elmindreda.org>
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any
// damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any
// purpose, including commercial applications, and to alter it and
// redistribute it freely, subject to the following restrictions:
//
//  1. The origin of this software must not be misrepresented; you
//     must not claim that you wrote the original software. If you use
//     this software in a product, an acknowledgment in the product
//     documentation would be appreciated but is not required.
//
//  2. Altered source versions must be plainly marked as such, and
//     must not be misrepresented as being the original software.
//
//  3. This notice may not be removed or altered from any source
//     distribution.
//
///////////////////////////////////////////////////////////////////////
#ifndef MOIRA_COLOR_H
#define MOIRA_COLOR_H
///////////////////////////////////////////////////////////////////////

#if MOIRA_HAVE_MATH_H
#include <math.h>
#endif

///////////////////////////////////////////////////////////////////////

namespace moira
{
  
///////////////////////////////////////////////////////////////////////

class ColorHSL;
class ColorHSV;
class ColorRGBA;

///////////////////////////////////////////////////////////////////////

class ColorRGB
{
public:
  /*! Default constructor.
   *  @remarks This does not initialize the object's fields.
   */
  inline ColorRGB(void);
  /*! Creates a color object with the specified values.
   */
  inline ColorRGB(float sr, float sg, float sb);
  inline ColorRGB(const ColorRGBA& rgba);
  ColorRGB(const ColorHSL& hsl);
  ColorRGB(const ColorHSV& hsv);
  ColorRGB(const String& string);
  /*! Clamps each component to the range [0, 1].
   */
  void clamp(void);
  inline ColorRGB min(const ColorRGB& color) const;
  inline ColorRGB max(const ColorRGB& color) const;
  void convertToString(String& result) const;
  inline operator float* (void);
  inline operator const float* (void) const;
  inline ColorRGB operator - (void) const;
  inline ColorRGB operator + (float value) const;
  inline ColorRGB operator - (float value) const;
  inline ColorRGB operator * (float value) const;
  inline ColorRGB operator / (float value) const;
  inline ColorRGB operator += (float value);
  inline ColorRGB& operator -= (float value);
  inline ColorRGB& operator *= (float value);
  inline ColorRGB& operator /= (float value);
  inline ColorRGB operator + (const ColorRGB& color) const;
  inline ColorRGB operator - (const ColorRGB& color) const;
  inline ColorRGB operator * (const ColorRGB& color) const;
  inline ColorRGB operator / (const ColorRGB& color) const;
  inline ColorRGB operator += (const ColorRGB& color);
  inline ColorRGB& operator -= (const ColorRGB& color);
  inline ColorRGB& operator *= (const ColorRGB& color);
  inline ColorRGB& operator /= (const ColorRGB& color);
  inline bool operator == (const ColorRGB& color) const;
  inline bool operator != (const ColorRGB& color) const;
  inline float getLength(void) const;
  inline float getSquaredLength(void) const;
  inline void setDefaults(void);
  inline void set(float sr, float sg, float sb);
  float r;
  float g;
  float b;
  static const ColorRGB WHITE;
  static const ColorRGB BLACK;
};

///////////////////////////////////////////////////////////////////////

class ColorRGBA
{
public:
  inline ColorRGBA(void);
  inline ColorRGBA(const ColorRGB& rgb, float a = 1.f);
  inline ColorRGBA(const ColorHSL& hsl, float a = 1.f);
  inline ColorRGBA(const ColorHSV& hsv, float a = 1.f);
  inline ColorRGBA(float r, float g, float b, float a);
  ColorRGBA(const String& string);
  /*! Clamps each component to the range [0, 1].
   */
  void clamp(void);
  inline ColorRGBA min(const ColorRGBA& color) const;
  inline ColorRGBA max(const ColorRGBA& color) const;
  void convertToString(String& result) const;
  inline operator float* (void);
  inline operator const float* (void) const;
  inline ColorRGBA operator - (void) const;
  inline ColorRGBA operator + (float value) const;
  inline ColorRGBA operator - (float value) const;
  inline ColorRGBA operator * (float value) const;
  inline ColorRGBA operator / (float value) const;
  inline ColorRGBA operator += (float value);
  inline ColorRGBA& operator -= (float value);
  inline ColorRGBA& operator *= (float value);
  inline ColorRGBA& operator /= (float value);
  inline ColorRGBA operator + (const ColorRGBA& color) const;
  inline ColorRGBA operator - (const ColorRGBA& color) const;
  inline ColorRGBA operator * (const ColorRGBA& color) const;
  inline ColorRGBA operator / (const ColorRGBA& color) const;
  inline ColorRGBA operator += (const ColorRGBA& color);
  inline ColorRGBA& operator -= (const ColorRGBA& color);
  inline ColorRGBA& operator *= (const ColorRGBA& color);
  inline ColorRGBA& operator /= (const ColorRGBA& color);
  inline bool operator == (const ColorRGBA& color) const;
  inline bool operator != (const ColorRGBA& color) const;
  inline float getLength(void) const;
  inline float getSquaredLength(void) const;
  inline void setDefaults(void);
  inline void set(const ColorRGB& rgb, float a);
  inline void set(float r, float g, float b, float a);
  float r;
  float g;
  float b;
  float a;
  static const ColorRGBA WHITE;
  static const ColorRGBA BLACK;
};

///////////////////////////////////////////////////////////////////////

class ColorHSL
{
public:
  /*! Default constructor.
   *  @remarks This does not initialize the object's fields.
   */
  inline ColorHSL(void);
  /*! Creates a color object with the specified values.
   */
  inline ColorHSL(float h, float s, float l);
  ColorHSL(const ColorRGB& rgb);
  inline ColorHSL(const ColorRGBA& rgba);
  inline ColorHSL(const ColorHSV& hsv);
  /*! Clamps each component to the range [0, 1].
   */
  void clamp(void);
  inline operator float* (void);
  inline operator const float* (void) const;
  inline ColorHSL operator - (void) const;
  inline ColorHSL operator + (float value) const;
  inline ColorHSL operator - (float value) const;
  inline ColorHSL operator * (float value) const;
  inline ColorHSL operator / (float value) const;
  inline ColorHSL operator += (float value);
  inline ColorHSL& operator -= (float value);
  inline ColorHSL& operator *= (float value);
  inline ColorHSL& operator /= (float value);
  inline ColorHSL operator + (const ColorHSL& color) const;
  inline ColorHSL operator - (const ColorHSL& color) const;
  inline ColorHSL operator * (const ColorHSL& color) const;
  inline ColorHSL operator / (const ColorHSL& color) const;
  inline ColorHSL operator += (const ColorHSL& color);
  inline ColorHSL& operator -= (const ColorHSL& color);
  inline ColorHSL& operator *= (const ColorHSL& color);
  inline ColorHSL& operator /= (const ColorHSL& color);
  inline bool operator == (const ColorHSL& color) const;
  inline bool operator != (const ColorHSL& color) const;
  inline void setDefaults(void);
  inline void set(float newH, float newS, float newL);
  float h;
  float s;
  float l;
  static const ColorHSL WHITE;
  static const ColorHSL BLACK;
};

///////////////////////////////////////////////////////////////////////

class ColorHSV
{
public:
  /*! Default constructor.
   *  @remarks This does not initialize the object's fields.
   */
  inline ColorHSV(void);
  /*! Creates a color object with the specified values.
   */
  inline ColorHSV(float h, float s, float v);
  ColorHSV(const ColorRGB& rgb);
  inline ColorHSV(const ColorRGBA& rgba);
  inline ColorHSV(const ColorHSL& hsv);
  /*! Clamps each component to the range [0, 1].
   */
  void clamp(void);
  inline operator float* (void);
  inline operator const float* (void) const;
  inline ColorHSV operator - (void) const;
  inline ColorHSV operator + (float value) const;
  inline ColorHSV operator - (float value) const;
  inline ColorHSV operator * (float value) const;
  inline ColorHSV operator / (float value) const;
  inline ColorHSV operator += (float value);
  inline ColorHSV& operator -= (float value);
  inline ColorHSV& operator *= (float value);
  inline ColorHSV& operator /= (float value);
  inline ColorHSV operator + (const ColorHSV& color) const;
  inline ColorHSV operator - (const ColorHSV& color) const;
  inline ColorHSV operator * (const ColorHSV& color) const;
  inline ColorHSV operator / (const ColorHSV& color) const;
  inline ColorHSV operator += (const ColorHSV& color);
  inline ColorHSV& operator -= (const ColorHSV& color);
  inline ColorHSV& operator *= (const ColorHSV& color);
  inline ColorHSV& operator /= (const ColorHSV& color);
  inline bool operator == (const ColorHSV& color) const;
  inline bool operator != (const ColorHSV& color) const;
  inline void setDefaults(void);
  inline void set(float h, float s, float v);
  float h;
  float s;
  float v;
  static const ColorHSV WHITE;
  static const ColorHSV BLACK;
};

///////////////////////////////////////////////////////////////////////

inline ColorRGB::ColorRGB(void)
{
}

inline ColorRGB::ColorRGB(float sr, float sg, float sb):
  r(sr),
  g(sg),
  b(sb)
{
}

inline ColorRGB ColorRGB::min(const ColorRGB& color) const
{
  return ColorRGB(fminf(r, color.r), fminf(g, color.g), fminf(b, color.b));
}

inline ColorRGB ColorRGB::max(const ColorRGB& color) const
{
  return ColorRGB(fmaxf(r, color.r), fmaxf(g, color.g), fmaxf(b, color.b));
}

inline ColorRGB::operator float* (void)
{
  return &(r);
}

inline ColorRGB::operator const float* (void) const
{
  return &(r);
}

inline ColorRGB ColorRGB::operator - (void) const
{
  return ColorRGB(-r, -g, -b);
}

inline ColorRGB ColorRGB::operator + (float value) const
{
  return ColorRGB(r + value, g + value, b + value);
}

inline ColorRGB ColorRGB::operator - (float value) const
{
  return ColorRGB(r - value, g - value, b - value);
}

inline ColorRGB ColorRGB::operator * (float value) const
{
  return ColorRGB(r * value, g * value, b * value);
}

inline ColorRGB ColorRGB::operator / (float value) const
{
  return ColorRGB(r / value, g / value, b / value);
}

inline ColorRGB ColorRGB::operator += (float value)
{
  r += value;
  g += value;
  b += value;
  return *this;
}

inline ColorRGB& ColorRGB::operator -= (float value)
{
  r -= value;
  g -= value;
  b -= value;
  return *this;
}

inline ColorRGB& ColorRGB::operator *= (float value)
{
  r *= value;
  g *= value;
  b *= value;
  return *this;
}

inline ColorRGB& ColorRGB::operator /= (float value)
{
  r /= value;
  g /= value;
  b /= value;
  return *this;
}

inline ColorRGB ColorRGB::operator + (const ColorRGB& color) const
{
  return ColorRGB(r + color.r, g + color.g, b + color.b);
}

inline ColorRGB ColorRGB::operator - (const ColorRGB& color) const
{
  return ColorRGB(r - color.r, g - color.g, b - color.b);
}

inline ColorRGB ColorRGB::operator * (const ColorRGB& color) const
{
  return ColorRGB(r * color.r, g * color.g, b * color.b);
}

inline ColorRGB ColorRGB::operator / (const ColorRGB& color) const
{
  return ColorRGB(r / color.r, g / color.g, b / color.b);
}

inline ColorRGB ColorRGB::operator += (const ColorRGB& color)
{
  r += color.r;
  g += color.g;
  b += color.b;
  return *this;
}

inline ColorRGB& ColorRGB::operator -= (const ColorRGB& color)
{
  r -= color.r;
  g -= color.g;
  b -= color.b;
  return *this;
}

inline ColorRGB& ColorRGB::operator *= (const ColorRGB& color)
{
  r *= color.r;
  g *= color.g;
  b *= color.b;
  return *this;
}

inline ColorRGB& ColorRGB::operator /= (const ColorRGB& color)
{
  r /= color.r;
  g /= color.g;
  b /= color.b;
  return *this;
}

inline bool ColorRGB::operator == (const ColorRGB& color) const
{
  return r == color.r && g == color.g && b == color.b;
}

inline bool ColorRGB::operator != (const ColorRGB& color) const
{
  return r != color.r || g != color.g || b != color.b;
}

inline float ColorRGB::getLength(void) const
{
  return r * r + g * g + b * b;
}

inline float ColorRGB::getSquaredLength(void) const
{
  return sqrtf(r * r + g * g + b * b);
}

inline void ColorRGB::setDefaults(void)
{
  r = 0.f;
  g = 0.f;
  b = 0.f;
}

inline void ColorRGB::set(float sr, float sg, float sb)
{
  r = sr;
  g = sg;
  b = sb;
}

///////////////////////////////////////////////////////////////////////

inline ColorRGBA::ColorRGBA(void)
{
}

inline ColorRGBA::ColorRGBA(float sr, float sg, float sb, float sa):
  r(sr),
  g(sg),
  b(sb),
  a(sa)
{
}

inline ColorRGBA::ColorRGBA(const ColorRGB& rgb, float sa):
  r(rgb.r),
  g(rgb.g),
  b(rgb.b),
  a(sa)
{
}

inline ColorRGBA ColorRGBA::min(const ColorRGBA& color) const
{
  return ColorRGBA(fminf(r, color.r),
                   fminf(g, color.g),
		   fminf(b, color.b),
		   fminf(a, color.a));
}

inline ColorRGBA ColorRGBA::max(const ColorRGBA& color) const
{
  return ColorRGBA(fmaxf(r, color.r),
                   fmaxf(g, color.g),
		   fmaxf(b, color.b),
		   fmaxf(a, color.a));
}

inline ColorRGBA::operator float* (void)
{
  return &(r);
}

inline ColorRGBA::operator const float* (void) const
{
  return &(r);
}

inline ColorRGBA ColorRGBA::operator - (void) const
{
  return ColorRGBA(-r, -g, -b, -a);
}

inline ColorRGBA ColorRGBA::operator + (float value) const
{
  return ColorRGBA(r + value, g + value, b + value, a + value);
}

inline ColorRGBA ColorRGBA::operator - (float value) const
{
  return ColorRGBA(r - value, g - value, b - value, a - value);
}

inline ColorRGBA ColorRGBA::operator * (float value) const
{
  return ColorRGBA(r * value, g * value, b * value, a * value);
}

inline ColorRGBA ColorRGBA::operator / (float value) const
{
  return ColorRGBA(r / value, g / value, b / value, a / value);
}

inline ColorRGBA ColorRGBA::operator += (float value)
{
  r += value;
  g += value;
  b += value;
  a += value;
  return *this;
}

inline ColorRGBA& ColorRGBA::operator -= (float value)
{
  r -= value;
  g -= value;
  b -= value;
  a -= value;
  return *this;
}

inline ColorRGBA& ColorRGBA::operator *= (float value)
{
  r *= value;
  g *= value;
  b *= value;
  a *= value;
  return *this;
}

inline ColorRGBA& ColorRGBA::operator /= (float value)
{
  r /= value;
  g /= value;
  b /= value;
  a /= value;
  return *this;
}

inline ColorRGBA ColorRGBA::operator + (const ColorRGBA& color) const
{
  return ColorRGBA(r + color.r, g + color.g, b + color.b, a + color.a);
}

inline ColorRGBA ColorRGBA::operator - (const ColorRGBA& color) const
{
  return ColorRGBA(r - color.r, g - color.g, b - color.b, a - color.a);
}

inline ColorRGBA ColorRGBA::operator * (const ColorRGBA& color) const
{
  return ColorRGBA(r * color.r, g * color.g, b * color.b, a * color.a);
}

inline ColorRGBA ColorRGBA::operator / (const ColorRGBA& color) const
{
  return ColorRGBA(r / color.r, g / color.g, b / color.b, a / color.a);
}

inline ColorRGBA ColorRGBA::operator += (const ColorRGBA& color)
{
  r += color.r;
  g += color.g;
  b += color.b;
  a += color.a;
  return *this;
}

inline ColorRGBA& ColorRGBA::operator -= (const ColorRGBA& color)
{
  r -= color.r;
  g -= color.g;
  b -= color.b;
  a -= color.a;
  return *this;
}

inline ColorRGBA& ColorRGBA::operator *= (const ColorRGBA& color)
{
  r *= color.r;
  g *= color.g;
  b *= color.b;
  a *= color.a;
  return *this;
}

inline ColorRGBA& ColorRGBA::operator /= (const ColorRGBA& color)
{
  r /= color.r;
  g /= color.g;
  b /= color.b;
  a /= color.a;
  return *this;
}

inline bool ColorRGBA::operator == (const ColorRGBA& color) const
{
  return r == color.r && g == color.g && b == color.b && a == color.a;
}

inline bool ColorRGBA::operator != (const ColorRGBA& color) const
{
  return r != color.r || g != color.g || b != color.b || a != color.a;
}

inline float ColorRGBA::getLength(void) const
{
  return r * r + g * g + b * b;
}

inline float ColorRGBA::getSquaredLength(void) const
{
  return sqrtf(r * r + g * g + b * b);
}

inline void ColorRGBA::setDefaults(void)
{
  r = 0.f;
  g = 0.f;
  b = 0.f;
  a = 1.f;
}

inline void ColorRGBA::set(const ColorRGB& rgb, float sa)
{
  r = rgb.r;
  g = rgb.g;
  b = rgb.b;
  a = sa;
}

inline void ColorRGBA::set(float sr, float sg, float sb, float sa)
{
  r = sr;
  g = sg;
  b = sb;
  a = sa;
}

///////////////////////////////////////////////////////////////////////

inline ColorHSL::ColorHSL(void)
{
}

inline ColorHSL::ColorHSL(float sh, float ss, float sl):
  h(sh),
  s(ss),
  l(sl)
{
}

inline ColorHSL::operator float* (void)
{
  return &(h);
}

inline ColorHSL::operator const float* (void) const
{
  return &(h);
}

inline ColorHSL ColorHSL::operator - (void) const
{
  return ColorHSL(-h, -s, -l);
}

inline ColorHSL ColorHSL::operator + (float value) const
{
  return ColorHSL(h + value, s + value, l + value);
}

inline ColorHSL ColorHSL::operator - (float value) const
{
  return ColorHSL(h - value, s - value, l - value);
}

inline ColorHSL ColorHSL::operator * (float value) const
{
  return ColorHSL(h * value, s * value, l * value);
}

inline ColorHSL ColorHSL::operator / (float value) const
{
  return ColorHSL(h / value, s / value, l / value);
}

inline ColorHSL ColorHSL::operator += (float value)
{
  h += value;
  s += value;
  l += value;
  return *this;
}

inline ColorHSL& ColorHSL::operator -= (float value)
{
  h -= value;
  s -= value;
  l -= value;
  return *this;
}

inline ColorHSL& ColorHSL::operator *= (float value)
{
  h *= value;
  s *= value;
  l *= value;
  return *this;
}

inline ColorHSL& ColorHSL::operator /= (float value)
{
  h /= value;
  s /= value;
  l /= value;
  return *this;
}

inline ColorHSL ColorHSL::operator + (const ColorHSL& color) const
{
  return ColorHSL(h + color.h, s + color.s, l + color.l);
}

inline ColorHSL ColorHSL::operator - (const ColorHSL& color) const
{
  return ColorHSL(h - color.h, s - color.s, l - color.l);
}

inline ColorHSL ColorHSL::operator * (const ColorHSL& color) const
{
  return ColorHSL(h * color.h, s * color.s, l * color.l);
}

inline ColorHSL ColorHSL::operator / (const ColorHSL& color) const
{
  return ColorHSL(h / color.h, s / color.s, l / color.l);
}

inline ColorHSL ColorHSL::operator += (const ColorHSL& color)
{
  h += color.h;
  s += color.s;
  l += color.l;
  return *this;
}

inline ColorHSL& ColorHSL::operator -= (const ColorHSL& color)
{
  h -= color.h;
  s -= color.s;
  l -= color.l;
  return *this;
}

inline ColorHSL& ColorHSL::operator *= (const ColorHSL& color)
{
  h *= color.h;
  s *= color.s;
  l *= color.l;
  return *this;
}

inline ColorHSL& ColorHSL::operator /= (const ColorHSL& color)
{
  h /= color.h;
  s /= color.s;
  l /= color.l;
  return *this;
}

inline bool ColorHSL::operator == (const ColorHSL& color) const
{
  return h == color.h && s == color.s && l == color.l;
}

inline bool ColorHSL::operator != (const ColorHSL& color) const
{
  return h != color.h || s != color.s || l != color.l;
}

inline void ColorHSL::setDefaults(void)
{
  h = 0.f;
  s = 0.f;
  l = 0.f;
}

inline void ColorHSL::set(float sh, float ss, float sl)
{
  h = sh;
  s = ss;
  l = sl;
}

///////////////////////////////////////////////////////////////////////

inline ColorHSV::ColorHSV(void)
{
}

inline ColorHSV::ColorHSV(float sh, float ss, float sv):
  h(sh),
  s(ss),
  v(sv)
{
}

inline ColorHSV::operator float* (void)
{
  return &(h);
}

inline ColorHSV::operator const float* (void) const
{
  return &(h);
}

inline ColorHSV ColorHSV::operator - (void) const
{
  return ColorHSV(-h, -s, -v);
}

inline ColorHSV ColorHSV::operator + (float value) const
{
  return ColorHSV(h + value, s + value, v + value);
}

inline ColorHSV ColorHSV::operator - (float value) const
{
  return ColorHSV(h - value, s - value, v - value);
}

inline ColorHSV ColorHSV::operator * (float value) const
{
  return ColorHSV(h * value, s * value, v * value);
}

inline ColorHSV ColorHSV::operator / (float value) const
{
  return ColorHSV(h / value, s / value, v / value);
}

inline ColorHSV ColorHSV::operator += (float value)
{
  h += value;
  s += value;
  v += value;
  return *this;
}

inline ColorHSV& ColorHSV::operator -= (float value)
{
  h -= value;
  s -= value;
  v -= value;
  return *this;
}

inline ColorHSV& ColorHSV::operator *= (float value)
{
  h *= value;
  s *= value;
  v *= value;
  return *this;
}

inline ColorHSV& ColorHSV::operator /= (float value)
{
  h /= value;
  s /= value;
  v /= value;
  return *this;
}

inline ColorHSV ColorHSV::operator + (const ColorHSV& color) const
{
  return ColorHSV(h + color.h, s + color.s, v + color.v);
}

inline ColorHSV ColorHSV::operator - (const ColorHSV& color) const
{
  return ColorHSV(h - color.h, s - color.s, v - color.v);
}

inline ColorHSV ColorHSV::operator * (const ColorHSV& color) const
{
  return ColorHSV(h * color.h, s * color.s, v * color.v);
}

inline ColorHSV ColorHSV::operator / (const ColorHSV& color) const
{
  return ColorHSV(h / color.h, s / color.s, v / color.v);
}

inline ColorHSV ColorHSV::operator += (const ColorHSV& color)
{
  h += color.h;
  s += color.s;
  v += color.v;
  return *this;
}

inline ColorHSV& ColorHSV::operator -= (const ColorHSV& color)
{
  h -= color.h;
  s -= color.s;
  v -= color.v;
  return *this;
}

inline ColorHSV& ColorHSV::operator *= (const ColorHSV& color)
{
  h *= color.h;
  s *= color.s;
  v *= color.v;
  return *this;
}

inline ColorHSV& ColorHSV::operator /= (const ColorHSV& color)
{
  h /= color.h;
  s /= color.s;
  v /= color.v;
  return *this;
}

inline bool ColorHSV::operator == (const ColorHSV& color) const
{
  return h == color.h && s == color.s && v == color.v;
}

inline bool ColorHSV::operator != (const ColorHSV& color) const
{
  return h != color.h || s != color.s || v != color.v;
}

inline void ColorHSV::setDefaults(void)
{
  h = 0.f;
  s = 0.f;
  v = 0.f;
}

inline void ColorHSV::set(float sh, float ss, float sv)
{
  h = sh;
  s = ss;
  v = sv;
}

///////////////////////////////////////////////////////////////////////

} /*namespace moira*/

///////////////////////////////////////////////////////////////////////
#endif /*MOIRA_COLOR_H*/
///////////////////////////////////////////////////////////////////////
