
#include <wendy/Wendy.h>

using namespace moira;
using namespace wendy;

#include "Glow.h"
#include "Stars.h"
#include "Frame.h"

FrameEffect::FrameEffect(demo::EffectType& type, const String& name):
  demo::Effect(type, name),
  sequence(*this, "Sequence"),
  currentSequence(0)
{
  sequence.addSymbol("Inside", INSIDE);
  sequence.addSymbol("Outside out", OUTSIDE_OUT);
  sequence.addSymbol("Outside orbit", OUTSIDE_ORBIT);
  sequence.addSymbol("Outside in", OUTSIDE_IN);
}

bool FrameEffect::init(void)
{
  glow = Glow::createInstance(256);
  if (!glow)
    return false;

  frameCanvas = GL::TextureCanvas::createInstance(GL::Context::get()->getWidth(),
					          GL::Context::get()->getHeight(),
					          "frame");
  if (!frameCanvas)
    return false;

  Ref<Stars> stars = Stars::createInstance(50.f);
  if (!stars)
    return false;

  render::Mesh* dome = render::Mesh::readInstance("dome");
  if (!dome)
    return false;

  mesh = render::Mesh::readInstance("frame");
  if (!mesh)
    return false;

  render::MeshNode* meshNode = new render::MeshNode();
  meshNode->setMesh(mesh);
  scene.addNode(*meshNode);

  domeNode = new render::MeshNode();
  domeNode->setMesh(dome);
  scene.addNode(*domeNode);

  starsNode = new StarsNode();
  starsNode->setStars(stars);
  scene.addNode(*starsNode);

  camera.setFOV(60.f);

  cameraNode = new render::CameraNode();
  cameraNode->setCameraName(camera.getName());
  scene.addNode(*cameraNode);

  generateTracks();

  return true;
}

void FrameEffect::update(Time deltaTime)
{
  const unsigned int newSequence = sequence.getSequenceIndex(getTimeElapsed());
  if (currentSequence != newSequence)
  {
    generateTracks();
    currentSequence = newSequence;
  }

  if (sequence.getValue(getTimeElapsed()) != INSIDE)
  {
    const Time elapsed = getTimeElapsed();
    const Time start = sequence.getSequenceStart(elapsed);
    const Time duration = sequence.getSequenceDuration(elapsed);

    float progress = (float) (elapsed - start) / (float) duration;

    Vector3 position = track(progress);

    cameraNode->getLocalTransform().position = position;
    cameraNode->getLocalTransform().rotation.setVectorRotation(position.normalize());

    starsNode->getLocalTransform().position = position;
    domeNode->getLocalTransform().position = position;

    scene.setTimeElapsed(elapsed - start);
  }
}

void FrameEffect::prepare(void) const
{
  prepareChildren();

  if (sequence.getValue(getTimeElapsed()) != INSIDE)
  {
    frameCanvas->begin();
    renderChildren();
    frameCanvas->end();

    render::Queue queue(camera);
    scene.enqueue(queue);
    glow->prepare(queue);
  }
}

void FrameEffect::render(void) const
{
  if (sequence.getValue(getTimeElapsed()) == INSIDE)
    renderChildren();
  else
  {
    render::Queue queue(camera);
    scene.enqueue(queue);
    queue.render();

    glow->render();
  }
}

void FrameEffect::generateTracks(void)
{
  track.points.clear();

  float magicDistance = 0.5f / tanf((camera.getFOV() / 2.f) * M_PI / 180.f);

  switch (sequence.getValue(getTimeElapsed()))
  {
    case OUTSIDE_OUT:
    {
      BezierPoint3 point;

      point.position.set(0.f, 0.f, 2.0f + magicDistance);
      point.direction.set(0.f, 0.f, 0.5f);
      track.points.push_back(point);

      point.position.z += 1.f;
      track.points.push_back(point);

      point.position.z += 1.f;
      track.points.push_back(point);

      break;
    }

    case OUTSIDE_ORBIT:
    {
      RandomVolume volume(Vector3(-3.f, -2.f, -2.f),
			  Vector3(3.f, 2.f, 5.f));

      unsigned int seconds = std::max(1, (int) sequence.getSequenceDuration(getTimeElapsed()));

      BezierPoint3 point;

      point.position.set(0.f, 0.f, 4.0f + magicDistance);
      point.direction.set(0.f, 0.f, 0.5f);
      track.points.push_back(point);

      for (unsigned int i = 0;  i < seconds * 2;  i++)
      {
	Vector3 position;

	do
	{
	  position = volume.generate();
	}
	while (position.length() < 3.f);

	point.position = position;
	position.normalize();

	Vector3 direction;

	do
	{
	  direction = (volume.generate() - volume.generate()).normalize();
	}
	while (fabsf(direction.dotProduct(position)) > 0.5f);

	point.direction = direction;

	track.points.push_back(point);
      }

      point.position.set(0.f, 0.f, 4.0f + magicDistance);
      point.direction.set(0.f, 0.f, -0.5f);
      track.points.push_back(point);

      break;
    }

    case OUTSIDE_IN:
    {
      BezierPoint3 point;

      point.position.set(0.f, 0.f, 4.f + magicDistance);
      point.direction.set(0.f, 0.f, -0.5f);
      track.points.push_back(point);

      point.position.set(0.f, 0.f, 2.f + magicDistance);
      point.direction.set(0.f, 0.f, -0.5f);
      track.points.push_back(point);

      break;
    }
  }
}

