#include "Quat.h"
#include "assert.h"

void QUAT::operator ()(float x, float y, float z, float scalar)
{
	vector(x, y, z);
	this ->scalar = scalar;
}

QUAT QUAT::operator + (QUAT source)
{
	QUAT ret;
	ret.vector = vector + source.vector;
	ret.scalar = scalar + source.scalar;
	return ret;
}

QUAT QUAT::operator - (QUAT source)
{
	QUAT ret;
	ret.vector = vector - source.vector;
	ret.scalar = scalar - source.scalar;
	return ret;
}

QUAT QUAT::operator * (QUAT source)
{
	QUAT ret;
	G3DVECTOR	v1, v2, v3;
	v1 = vector * source.scalar;
	v2 = source.vector * scalar;
	v3 = vector * source.vector;
	ret.vector = v1 + v2 + v3;
	ret.scalar = (scalar * source.scalar) - vector.DotProduct(source.vector);
	return ret;
}

QUAT QUAT::operator * (float source)
{
	QUAT ret;
	ret.vector = vector * source;
	ret.scalar = scalar * source;
	return ret;
}

QUAT QUAT::GetCoupleQuat()
{
	QUAT ret;
	ret.vector = vector;
	ret.vector.Reverse();
	ret.scalar = scalar;
	return ret;
}

void QUAT::RotateToQuat(float x, float y, float z, float angle)
{
	vector(x, y, z);
	vector = vector * sinf(angle/2.0f);
	scalar = cosf(angle/2.0f);
	Normalize();
}

void QUAT::QuatToRotate(G3DVECTOR *axis, float *angle)
{
	(*angle) = acosf(this ->scalar) * 2.0f;

	float _sinf = sinf((*angle)/2.0f);

	if (_sinf == 0)
	{
		(*axis) = this ->vector;
		(*axis).Normalize();
		return;
	}

	(*axis) = this ->vector * (1.0f/_sinf);
	(*axis).Normalize();
}

void QUAT::Normalize()
{
	float len = GetLength();
	if (len != 0.0f)
	{
		float proportion = 1.0f/len;
		vector = vector * proportion;
		scalar = scalar * proportion;
	}
}

float QUAT::GetLength()
{
	return sqrtf((scalar * scalar) + vector.DotProduct(vector));
}

float QUAT::DotProduct(QUAT q1, QUAT q2)
{
	return (q1.vector.x * q2.vector.x) + (q1.vector.y * q2.vector.y) + (q1.vector.z * q2.vector.z) + (q1.scalar * q2.scalar);
}

float QUAT::GetAngle(QUAT q1, QUAT q2)
{
	float	angle;

	angle = DotProduct(q1, q2);
	if (angle > 1.0f) angle = 1.0f;
	if (angle < -1.0f) angle = -1.0f;

	angle = acosf(angle);

	return angle;
}

G3DVECTOR QUAT::Rotate(G3DVECTOR source)
{
	QUAT ret;
	QUAT coup;
	coup = GetCoupleQuat();
	ret.vector = source;
	ret.scalar = 0.0f;
	ret = (*this) * ret;
	ret = ret * coup;
	return ret.vector;
}

QUAT QUAT::lerp(float time, QUAT q1, QUAT q2)
{
	QUAT ret;

	ret = (q1 * (1.0f - time)) + (q2 * time);
	ret.Normalize();

	return ret;
}

QUAT QUAT::slerp(float time, QUAT q1, QUAT q2)
{
	float angle = GetAngle(q1, q2);
	return ((q1 * sinf(angle * (1 - time))) + (q2 * sin(angle * time))) * (1/sin(angle));
}

QUAT QUAT::squad(float time, QUAT q1, QUAT q2, QUAT a, QUAT b)
{
	QUAT	p1, p2;

	p1 = slerp(time, q1, q2);
	p2 = slerp(time, a, b);

	return slerp(2*time*(1-time), p1, p2);
}
