#include "tunnel.hpp"
#include "utils.hpp"

#define N_TRIANGLES 100
#define MIN_Z 0.0
#define MAX_Z 1.0



Tunnel::Tunnel() {
  _t0 = -1.0;
  _triangles = new Face3[N_TRIANGLES];

  /* we start with all the triangles very close to the viewer.
     this way they will be found to be 'too close' by zoom_triangles(),
     and they'll be replaced with cool ones asap */

  for (int i = 0; i < N_TRIANGLES; i++) {
    double z = (double) i * (MAX_Z - MIN_Z) / N_TRIANGLES + MIN_Z;
    _triangles[i].p0.z = z;
    _triangles[i].p1.z = z;
    _triangles[i].p2.z = z;
  }
}

double prev_time;

void Tunnel::effect(double time) {
  if (_t0 <= 0.0) {
    _t0 = time;
  }

  // TODO: Set dz to value based on time (or don't use dz at all)
  double local_time = time - _t0;

  for (int i = 0; i < N_TRIANGLES; i++) {
		calc_triangle(&_triangles[i], local_time, i);
  }

  _time = time;
  draw();
}

void Tunnel::calc_triangle(Face3* triangle, double time, int i) {
  double z = fmod(time * 1.0 + i * (1.0 / N_TRIANGLES), 1.0);
  double angle4 = (time + i) * 0.02;
  double angle0 = (time + i) * 2.0 * M_PI * (0.3 + sin(angle4) / 10.0);

  double angle5 = (time + i) * 0.01;
  double angle1 = (time + i) * sin(angle5) * 0.004;

  double x = 0.5 + 0.5 * cos (-angle0) + 0.25 * cos (angle4);
  double y = 0.5 + 0.5 * sin (-angle0) + 0.25 * sin (angle4);

  triangle->p0.x = x + 0.3 * cos (angle1 + M_PI * 0.0 / 3.0);
  triangle->p0.y = y + 0.3 * sin (angle1 + M_PI * 0.0 / 3.0);
  triangle->p0.z = z;

  triangle->p1.x = x + 0.3 * cos (angle1 + M_PI * 2.0 / 3.0);
  triangle->p1.y = y + 0.3 * sin (angle1 + M_PI * 2.0 / 3.0);
  triangle->p1.z = z;

  triangle->p2.x = x + 0.3 * cos (angle1 + M_PI * 4.0 / 3.0);
  triangle->p2.y = y + 0.3 * sin (angle1 + M_PI * 4.0 / 3.0);
  triangle->p2.z = z;
}

void Tunnel::draw() {
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();

  glTranslatef(-1.0f, -0.1f, 0.0f);

  glClear(GL_DEPTH_BUFFER_BIT);
  glDisable(GL_DEPTH_TEST);

  glBlendFunc(GL_SRC_ALPHA_SATURATE, GL_ONE);
  glEnable(GL_BLEND);
  glDisable(GL_LIGHTING);
  glColor4f(0.9, 0.9, 1.0, 0.1);

  for (int i = 0; i < N_TRIANGLES; i++) {
	  glRotatef(i / 10.0, 0.0, 0.0, 1.0);
	  glBegin(GL_TRIANGLES);
	  glNormal3f(0.0, 0.0, -1.0);
	  float c = _triangles[i].p0.z;

	  float maxdepth = .7;
	 
	  if (c > maxdepth)
		  c = maxdepth - ((c-maxdepth) / (1.0-maxdepth));

	  if (c < 0)
		  c = 0;


    float r = cos(i+_time*.354) * .5 + .5;
    float g = cos(i+_time*.143) * .5 + .5;
    float b = cos(i+_time*.576) * .5 + .5;

	  glColor4f(r,g,b,c);
	  glVertex3f(_triangles[i].p0.x, _triangles[i].p0.y, _triangles[i].p0.z * 10.0 - 10.0);
	  glVertex3f(_triangles[i].p1.x, _triangles[i].p1.y, _triangles[i].p1.z * 10.0 - 10.0);
	  glVertex3f(_triangles[i].p2.x, _triangles[i].p2.y, _triangles[i].p2.z * 10.0 - 10.0);
	  glEnd();
  }

  glMatrixMode(GL_MODELVIEW);
  glPopMatrix();
}
