/**
 * tubes.cpp
 */

#include <cmath>

#include "tubes.hpp"
#include "utils.hpp"

/** Length of the tube, horizontally */
const float LENGTH = 8.0 * 2.0;

/** Number of coordinates on a circle */
const int N_CIRCLE_COORDS = 22;

/** Number of circles */
const int N_CIRCLES = 20 * 2;

/** For each pair of adjacent circles we store triangles */
const int N_TUBE_FACES = (N_CIRCLES - 1) * (N_CIRCLE_COORDS * 2);

Vector3 tubeCoords[N_CIRCLES * N_CIRCLE_COORDS];
IndexFace3 tubeFaces[N_TUBE_FACES];

Tubes::Tubes() {
  // No implementation
}

/**
 * Calculates coordinates on a tube, by calculating a list of distorted circles,
 * and then connecting these.
 *
 * @param time the time in milliseconds
 *
 * TODO: There is a small problem with the connecting seam, which is not continuous.
 *       Consider making the arbitrary function C0/C1/C2 continuous.
 */
void Tubes::effect(double time) {
  // Calculate coordinates
  for (int circle = 0; circle < N_CIRCLES; circle++) {
    float xPos = ((float) circle / N_CIRCLES);
    for (int i = 0; i < N_CIRCLE_COORDS; i++) {
      float angle = i * 2.0 * M_PI / N_CIRCLE_COORDS;
      Vector3* c = &tubeCoords[circle * N_CIRCLE_COORDS + i];
      calcCoord(*c, xPos, angle, time);
    }
  }
  // Set faces
  for (int circle = 0; circle < N_CIRCLES - 1; circle++) {
    for (int i = 0; i < (N_CIRCLE_COORDS - 1); i++) {
      int index = circle * N_CIRCLE_COORDS * 2 + i * 2;

      int curr = i;
      int next = ((curr + 1) % (N_CIRCLE_COORDS - 1));

      tubeFaces[index].v0 = circle * N_CIRCLE_COORDS + next;
      tubeFaces[index].v1 = circle * N_CIRCLE_COORDS + curr;
      tubeFaces[index].v2 = (circle + 1) * N_CIRCLE_COORDS + curr;
      calc_face_normal(tubeFaces[index], tubeCoords);

      tubeFaces[index + 1].v0 = (circle + 1) * N_CIRCLE_COORDS + next;
      tubeFaces[index + 1].v1 = circle * N_CIRCLE_COORDS + next;
      tubeFaces[index + 1].v2 = (circle + 1) * N_CIRCLE_COORDS + curr;
      calc_face_normal(tubeFaces[index + 1], tubeCoords);
    }
  }

  draw();
}

/**
 * Calculates a coordinate, based on x, f, t
 * @param x the horizontal component, derived from the circle count, range 0..1
 * @param f the radial component, derived from the circle angle, range 0..2*pi
 * @param t the time component, derived from the time, range 0..inf
 */
void Tubes::calcCoord(Vector3& c, float x, float f, float t) {
  x = (x - 0.5) * LENGTH;
  c.x = x + 0.5 * cos(cos(x + f) * sin(f * 0.5 * sin(t) + f * 3.2) + sin(t));
  c.y = cos(f + t)
    + 0.2 * cos(cos(x + f * 3.0 * cos(t)) * sin(f * 0.1 + t) + t);
  c.z = sin(f + t)
    + 0.2 * sin(sin(x - f * 1.7 - sin(t)) * sin(f * 0.3) + t);
}

void Tubes::draw() {
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();

  glEnable(GL_LIGHTING);
  glLoadIdentity();
  glTranslatef(-1.0, -1.0, -7.0);

  draw_faces(tubeFaces, N_TUBE_FACES, tubeCoords);

  glPopMatrix();
}

