/**
 * eff5.cpp - background image
 *
 * Preparation effects for a simple demo.
 * Requires: OpenGL, SDL.
 *
 * @version n/a
 */

// Visual C workarounds
#ifdef _MSC_VER
#define M_PI 3.1415926535
#include <windows.h>
#endif

#include <SDL/SDL.h>
#include <SDL/SDL_image.h>
#include <GL/gl.h>
#include <GL/glu.h>

#include <cmath>
#include <cstdlib>
#include <iostream>
#include <vector>

#include "graphics.hpp"
#include "utils.hpp"

#include "background.hpp"
#include "font.hpp"
#include "cubes.hpp"
#include "tubes.hpp"
#include "hermite.hpp"
#include "cubebitmap.hpp"
#include "water.hpp"
#include "scroller.hpp"
#include "tunnel.hpp"
#include "feedback.hpp"
#include "dance.hpp"

#include <fmod.h>
#include <fmod_errors.h>

/** Set to 1 to enable full screen mode */
const int FULLSCREEN = 1;

/** Set to 1 to disable sound playback */
const int NO_SOUND = 0;

#define WIDTH 640
#define HEIGHT 480

/** Reference to the FMOD music */
FMUSIC_MODULE *mod = NULL;

double _START = 0.0;

/**
 * Gets the current time in SDL ticks.
 */
float getTime() {
  return SDL_GetTicks() / 1000.0f - _START;
  //  return FMUSIC_GetTime(mod) / 1000.0f;
}

void  resetTimer() {
  _START = SDL_GetTicks() / 1000.0f;
}

/**
 * Initializes SDL (Simple Direct media Layer)
 */
void init_sdl() {
  SDL_Init(SDL_INIT_VIDEO);
  SDL_GL_SetAttribute(SDL_GL_RED_SIZE, 8);
  SDL_GL_SetAttribute(SDL_GL_GREEN_SIZE, 8);
  SDL_GL_SetAttribute(SDL_GL_BLUE_SIZE, 8);
  SDL_GL_SetAttribute(SDL_GL_ALPHA_SIZE, 8);
  SDL_GL_SetAttribute(SDL_GL_DEPTH_SIZE, 24);
  SDL_GL_SetAttribute(SDL_GL_DOUBLEBUFFER, 1);

#ifdef SDL_1_2_10
  SDL_GL_SetAttribute(SDL_GL_SWAP_CONTROL, 1);
#endif /* SDL_1_2_10 */
  SDL_ShowCursor(0);
  
  SDL_Surface *screen = 0;
  
  if ((screen=SDL_SetVideoMode(WIDTH, HEIGHT, 0, SDL_OPENGL | (FULLSCREEN ? SDL_FULLSCREEN : 0) )) == 0) {
    dprintf("Could not initialize SDL/OpenGL: %s\n", SDL_GetError());
    SDL_Quit();
    return;
  }

#ifdef _MSC_VER
  ShowCursor(0);
#endif
}

/**
 * Initializes OpenGL
 */
void init_opengl() {
  glShadeModel(GL_SMOOTH);
  glClearDepth(1.0f);
  glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST );

  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluPerspective(45.0f, (double) WIDTH / HEIGHT, 0.1f, 100.0f);
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();

  set_gl_lighting();
}

/**
 * Initializes FMOD and starts playing the specified music file.
 * @param file the filename of the music to play, typically "chip.xm"
 */
void init_fmod(char* file) {
  dprintf("Initializing music...\n");
  
  if (FSOUND_GetVersion() < FMOD_VERSION) {
    dprintf("Error: You are using the wrong DLL version!  You should be using FMOD %.02f\n", FMOD_VERSION);
    exit(1);
  }

  if (NO_SOUND) {
    FSOUND_SetOutput(FSOUND_OUTPUT_NOSOUND);
  }
  
  if (!FSOUND_Init(44100, 32, 0)) {
    dprintf("FMOD init error: %s\n", FMOD_ErrorString(FSOUND_GetError()));
    exit(1);
  }

  mod = FMUSIC_LoadSong(file);
  if (!mod) {
    dprintf("FMOD load error: %s\n", FMOD_ErrorString(FSOUND_GetError()));
    exit(1);
  }

  FMUSIC_SetLooping(mod, 0);
  FMUSIC_PlaySong(mod);     
}

void draw_curve(Vector3 v0, Vector3 v1, Vector3 v2, Vector3 v3) {
  Hermite hermite(v0, v1, v2, v3);
  
  glColor3f(0.5, 0.7, 1.0);
  glBegin(GL_LINES);
  for (int i = 0; i < 99; i++) {
    Vector3 v0 = hermite.interpolate((i + 0) / 100.0);
    Vector3 v1 = hermite.interpolate((i + 1) / 100.0);
    glVertex3f(v0.x, v0.y, v0.z);
    glVertex3f(v1.x, v1.y, v1.z);
  }
  glEnd();
}


/**
 * Display nice stuffs.
 */
void effects() {
  bool quit = false;
  SDL_Event event;

  Cubes cubes(9, 9, 9);
  CubeBitmap cube_bitmap(17, 17);
  Background background;
  Water water;
  Scroller scroller;
  Scroller scroller2(1);
  Scroller scroller3(2);
  Tubes tube;
  Tunnel tunnel;
  Feedback feedback;
  Dance dance;



  double start_time = getTime();
  int frames = 0;
  int prev_pattern = -1;
  while (!quit) {

    char title[255];
    sprintf(title,"%f", getTime());

       SDL_WM_SetCaption(title, 0 );

    glPushAttrib(~0);

//     for (int i = 0; i < 100000; i++) {
//       getTime();
//     }

    frames++;
    set_gl_lighting();

	int prev_pos;
	int next_pos;
    
    while(SDL_PollEvent(&event)){ 
      switch(event.type){ 
      case SDL_KEYDOWN:
	switch (event.key.keysym.sym) {
	case SDLK_LEFT:
	  prev_pos = FMUSIC_GetOrder(mod) - 1;
	  FMUSIC_SetOrder(mod, prev_pos);
	  break;
	case SDLK_RIGHT:
	  next_pos = FMUSIC_GetOrder(mod) + 1;
	  FMUSIC_SetOrder(mod, next_pos);
	  break;
	case SDLK_ESCAPE:
	  quit = true;
	  break;
	default:
	  // Other keys not handled
	  break;
	}
      }
    }

    int pattern = FMUSIC_GetOrder(mod);
    int row = FMUSIC_GetRow(mod);

    if (prev_pattern != pattern) {
      dprintf("pattern = %d\n", pattern);
    }
    prev_pattern = pattern;

    reset_gl();

    /* background image */
    background.effect(getTime(), pattern);

     glClear(GL_DEPTH_BUFFER_BIT);

    /* font */
    if ((pattern >= 0) && (pattern <= 2)) {
      scroller.effect(getTime());
    }

    /* cubes */
    if ((pattern >= 3) && (pattern <= 6)) {
      cubes.effect(getTime(), pattern - 3, row);
    }

    /* tunnel */
    if ((pattern >= 7) && (pattern <= 8)) {
      tunnel.effect(getTime());
    }

    /* cube bitmap */
    if ((pattern >= 9) && (pattern <= 11)) {
      // TODO: For the cube pixmap effect we temporarily do not
      //       want to see the background image.
//       glClearColor(0.2f, 0.1f, 0.4f, 0.0f);
//       glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
      cube_bitmap.effect(getTime(), pattern - 9);
    }

    /* feedback plasma */
    if ((pattern >= 12) && (pattern <= 13)) {
      feedback.effect(getTime());
      dance.effect(getTime(), pattern - 12);
      init_opengl();
      reset_gl();
      scroller3.draw3(getTime());
    }

    /* tube */
    if ((pattern >= 14) && (pattern <= 16)) {
      init_opengl();
      reset_gl();
      tube.effect(getTime());
      dance.effect(getTime(), pattern - 13);
      init_opengl();
      reset_gl();
      scroller2.draw2(getTime());
    }

//     if (pattern >= 16) {
//       dance.effect(getTime(), 2);
//     }

//     /* water effect */
//     if ((pattern >= 12) && (pattern <= 13)) {
//       water.effect(getTime());
//     }

//     /* curve line */
//     if ((pattern >= 14) && (pattern <= 15)) {
//       wuppie();
//     }

    /* the end */
    if ((pattern == 16) && (row == 31)) {
      break;
    }

    SDL_GL_SwapBuffers();
    glPopAttrib();
  }
  dprintf("FPS = %f\n", frames / (getTime() - start_time));
}

void cop(double start, int flag) {
  int width = 640;
  int height = 40;
  glBegin(GL_QUADS);
  for (int i = 0; i < height;i++) {
    double y = start + i;
    double f = sin((double) i / height * 1.0 * M_PI) / 2.0 + 0.5;
    f = f * f * f;
    if (getTime() > 13.0) {
      f = f * (15.0 - getTime()) / 2.0;
    }
    if (flag == 1) {
      glColor3f(0, f, 0);
    }
    if (flag == 2) {
      glColor3f(f, f, f);
    }
    if (flag == 3) {
      glColor3f(f, 0, 0);
    }
    glVertex3f(0.0, y, 0.0);
    glVertex3f(width, y, 0.0);
    glVertex3f(width, y + 1.0, 0.0);
    glVertex3f(0.0, y + 1.0, 0.0);
  }
  glEnd();
}

void coppers() {

  double time = getTime();

  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();
  glTranslatef(0.0f, 0.0f, -1.0);

  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();
  glOrtho(0.0, 640.0, 480.0, 0.0, 0.0, 1000);
  glDisable(GL_LIGHTING);

  // first
  double start1 = 240.0 + sin(time * 2.0) * 200.0;
  cop(start1, 1);
  // 2nd
  double start2 = 240.0 + sin(time * 2.0 + 0.2) * 200.0;
  cop(start2, 2);
  // 3
  double start3 = 240.0 + sin(time * 2.0 + 0.4) * 200.0;
  cop(start3, 3);

  glPopMatrix();

  glMatrixMode(GL_MODELVIEW);
  glPopMatrix();

}

void cracktro() {
  SDL_Event event;
  int quit = 0;
  Scroller s(3);
  do { 
    glClear(GL_DEPTH_BUFFER_BIT | GL_COLOR_BUFFER_BIT);
      while(SDL_PollEvent(&event)){ 
         switch(event.type){ 
         case SDL_KEYDOWN:
           switch (event.key.keysym.sym) {
        case SDLK_ESCAPE:
          quit = true;
          break;
        default:
          // Other keys not handled
          break;
        }
      }
    }
    glPushAttrib(~0);
    coppers();
    s.draw4(getTime());

    SDL_GL_SwapBuffers();
    glPopAttrib();
  } while ((getTime() < 15.0) && (!quit));
}

/**
 * Main routine
 */
int main(int argc, char *argv[]) {
  init_sdl();
  init_opengl();

  if (FULLSCREEN) {
    while (getTime() < 1.0) {
    }
  }

  resetTimer();
  init_fmod("DXN-FTGR.XM");
  FMUSIC_SetOrder(mod, 13);

  cracktro();
  init_opengl();

  init_fmod("chip.xm");

  resetTimer();
  // Hack: Skip to pattern for testing an effect
  //  FMUSIC_SetOrder(mod, 14);
  //     FMUSIC_SetOrder(mod, 7); // tunnel
  //    FMUSIC_SetOrder(mod, 3); // cubes
  //  FMUSIC_SetOrder(mod, 12); // feedback
  //  FMUSIC_SetOrder(mod, 13); // feedback
//  FMUSIC_SetOrder(mod, 1); // cubes
//   FMUSIC_SetOrder(mod, 9); // cube plasmas
//   FMUSIC_SetOrder(mod, 14); // tube

  effects();
  SDL_Quit();

  return 0;
}
