///////////////////////////////////////////////////////////////////////////////
/// \file  CRedNeuronal.h
/// \brief Clase que define el comportamiento de una red neuronal.
///
/// <b>F.Creacin:</b>28 de octubre de 2004.       <br>
/// <b>Autor:     </b>Francisco Jos Gallego Durn.<br>
///////////////////////////////////////////////////////////////////////////////
#ifndef __SR_CREDNEURONAL_H__
#define __SR_CREDNEURONAL_H__

/// PRAGMAS
///
/// Para deshabilitar el warning que produce la STL cuando introducimos
/// un tipo de la STL dentro de otro. Slo se queja de que los nombres
/// son demasiado largos para el depurador (por culpa de los pairs).
#pragma warning (disable : 4786)

/// INCLUDES
///
#include <vector>
#include <string>
#include <map>
#include <math.h>
#include <iostream>
#include "tipos.h"

using namespace std;

/// ESTRUCTURAS Y TIPOS
///

/// <b>Tipos de ejecucin de la red:</b><br>
/// Instantanea:  Ejecuta globalmente la red para obtener una respuesta
///               instantnea sin tener en cuenta situaciones anteriores.<br>
/// PasoDinamico: Ejecuta un nico paso de la red obteniendo una respuesta
///               y dejando los valores listos para el siguiente paso.<br>
typedef enum { Ejecucion_Instantanea,
               Ejecucion_PasoDinamico } TEjecucion;

/// <b>Tipos de neuronas</b>
///
typedef enum { Neurona_Entrada,
               Neurona_Salida,
               Neurona_Oculta,
               Neurona_Sesgo,
               Neurona_SinTipo } TNeurona;

/// <b>Tipo Par ordenado para Capas <doubles, ints></b><br>
///
/// Servir para insertar fcilmente valores en la tabla de capas.
///
typedef pair<double, int> TParCapas;


struct SNeurona;

/// <b>Estructura para definir un enlace</b>
///
struct SEnlace
{
    // Neuronas de entrada y salida del enlace
    SNeurona *pNeuronaEntrada, *pNeuronaSalida;

    // Peso del enlace
    double dPeso;

    // Es recurrente?
    bool bRecurrente;

    // Constructor por defecto
    SEnlace (double peso, SNeurona* pNIn,
             SNeurona* pNOut, bool rec): dPeso(peso), pNeuronaEntrada (pNIn),
                                         pNeuronaSalida(pNOut), bRecurrente (rec) {}
};

/// <b>Estructura para definir el comportamiento de una neurona</b>
///
struct SNeurona
{
    // Enlaces de entrada y salida de la neurona
    vector<SEnlace> vEnlacesEntrada, vEnlacesSalida;

    // Tipo de Neurona e identificador
    TNeurona TipoNeurona;
    int    nID;

    // Valores de clculo interno
    double dCurvaturaSigmoidea;
    double dSumaEntradas;
    double dSalida;
    double dX, dY, dSplitX, dSplitY;

    // Constructor por defecto
    SNeurona (TNeurona tipo, int id,
              double   x, double y,
              double   curva): TipoNeurona(tipo), nID (id), dSumaEntradas(0.0),
                               dSplitX(0.0), dSplitY(0.0), dSalida(0.0),
                               dX (0.0), dY(0.0), dCurvaturaSigmoidea (curva) {}
};

/// <b>CLASE:</b> CRedNeuronal<br>
///
/// Gestin y control de instancias que representan las redes neuronales reales (los fenotipos)<br>
/// de los individuos de la poblacin.
///
class CRedNeuronal
{
public:
    // Constructores y Destructor
    CRedNeuronal();
    ~CRedNeuronal();

    // Metodos de utilidad
    void Limpiar();
    const TVecDoubles&  Actualizar(const TVecDoubles& entradas, const TEjecucion TipoEjec);

    // Metodos Add
    const SNeurona*     AddNeurona(TNeurona TipoN, int nID, double dSplitX, double dSplitY, double dCurva);
    void                AddEnlace (const SNeurona* NeuronaEntrada, const SNeurona* NeuronaSalida, double dPeso, bool bRec);

    // Mtodos Get
    const vector<SNeurona *>& GetRed () const { return m_vNeuronas; }
private:
    // Selector de funcin de activacin
    double  FuncionActivacion (double SumaEntradas, double curvatura);

    // Atributos
    map<double, int>   m_mTablaCapas;   ///< Tabla hash para contar las capas que tiene la red y saber la profundidad
    TVecDoubles        m_vdSalidas;     ///< Vector que almacenar las salidas de una ejecucin de la red
    vector<SNeurona *> m_vNeuronas;     ///< Lista de neuronas que forman la red
};

/// OPERADORES DE ENTRADA Y SALIDA POR FLUJOS
///
ostream& operator<<(ostream& out, const CRedNeuronal& red);
istream& operator>>(istream&  in,       CRedNeuronal& red);

#endif
