///////////////////////////////////////////////////////////////////////////////
/// \file  CInnovacion.h
/// \brief Clase que define el comportamiento de la Base de Datos de innovaciones
///        en una poblacin de genomas.
///
/// <b>F.Creacin:</b>02 de mayo de 2005.          <br>
/// <b>Autor:     </b>Francisco Jos Gallego Durn.<br>
///////////////////////////////////////////////////////////////////////////////
#ifndef __SR_CINNOVACION_H__
#define __SR_CINNOVACION_H__

/// PRAGMAS
///
/// Para deshabilitar el warning que produce la STL cuando introducimos
/// un tipo de la STL dentro de otro. Slo se queja de que los nombres
/// son demasiado largos para el depurador (por culpa de los pairs).
#pragma warning (disable : 4786)

/// INCLUDES
///
#include <vector>
#include "Genes.h"
#include "CGenoma.h"
#include "CRedNeuronal.h"

class CGenoma;

using namespace std;

/// STRUCTS Y TIPOS
///

/// <b> Tipo de innovacin </b><br>
///
/// Enumeracin para describir las clases de innovacin que se producen. <br>
///
typedef enum
{
    INO_Neurona,
    INO_Enlace
} TInnovacion;

/// <b> Estructura de una innovacin </b><br>
///
/// Estructura completa de datos para describir todos los elementos de una innovacin.<br>
///
struct SInnovacion
{
    int         nID;              ///< Nmero identificador de la innovacin
    TInnovacion tipo;             ///< Tipo de innovacin

    int         nIDNeuronaInicio; ///< Neurona de la que parte una innovacin Enlace
    int         nIDNeuronaFin;    ///< Neurona a la que llega una innovacin Enlace

    int         nIDNeurona;       ///< Identificador de la neurona que representa una innovacin
    TNeurona    tipoNeurona;      ///< Tipo de la neurona que representa una innovacin
    double      dSplitY;          ///< Coordenada vertical   normalizada de representacin de la neurona
    double      dSplitX;          ///< Coordenada horizontal normalizada de representacin de la neurona

    /// <b> Constructor por defecto que no hace nada </b>
    SInnovacion() {}

    /// <b> Constructor de Innovaciones Enlace </b>
    SInnovacion(SGenEnlace gen, int id): nIDNeuronaInicio(gen.nIDNeuronaInicio), tipo (INO_Enlace),
                                         nIDNeuronaFin (gen.nIDNeuronaFin),      nIDNeurona (0),
                                         nID (id), dSplitX(0), dSplitY(0), tipoNeurona (Neurona_SinTipo) {}

    /// <b> Constructor de Innovaciones Neurona </b>
    SInnovacion(SGenNeurona gen, int id, int nIDNeu): nID (id), nIDNeurona (nIDNeu), dSplitX(gen.dSplitX),
                                                      dSplitY(gen.dSplitY), tipoNeurona (gen.Tipo), tipo (INO_Neurona),
                                                      nIDNeuronaInicio (-1), nIDNeuronaFin (-1)    {}

    /// <b> Constructor de Innovaciones Completo </b>
    SInnovacion(int dsd, int hasta, TInnovacion t, int id,
                TNeurona tNeu, double x, double y): nIDNeuronaInicio (dsd), nIDNeuronaFin (hasta),
                                                    tipo (t), nID (id), nIDNeurona (0),
                                                    tipoNeurona (tNeu), dSplitX(x), dSplitY(y)      {}
};

/// <b> Tipo vector de innovaciones </b><br>
///
/// Vector donde se guardarn todas las innovaciones que se produzcan en la poblacin<br>
///
typedef vector<SInnovacion> TInnovaciones;


/// <b> CLASE: </b> CInnovacion<br>
///
/// Clase que gestiona la base de datos de innovaciones en una poblacin de genomas.
///
class CInnovacion
{
public:
    CInnovacion(const CGenoma& genoma);

    // Mtodos SET
    int         SetSigIDInnovacion   (int num = 0) { return (m_nSigIDInnovacion += num); }
    void        SetSigIDNeurona      (int n)       { m_nSigIDNeurona = n; }

    // Mtodos de trabajo
    int         ComprobarInnovacion  (int nIDNeuronaIni, int nIDNeuronaFin, TInnovacion t);
    int         CrearNuevaInnovacion (SInnovacion inno);
    int         CrearNuevaInnovacion (int ini, int fin, TInnovacion t,
                                      TNeurona tNeu=Neurona_SinTipo, double x=0.0, double y=0.0);
    void        Limpiar              ();
    SGenNeurona CrearNeuronaDesdeID  (int nIDNeurona);

    // Mtodos GET
    int                  GetSigIDInnovacion ()          const { return m_nSigIDInnovacion;                }
    int                  GetSigIDNeurona    ()          const { return m_nSigIDNeurona;                   }
    int                  GetIDNeurona       (int nInno) const { return m_vInnovaciones[nInno].nIDNeurona; }
    const TInnovaciones& GetBDInnovaciones  ()          const { return m_vInnovaciones;                   }
private:
    TInnovaciones m_vInnovaciones;      ///< Vector donde se guardarn todas las innovaciones de la poblacin.
    int           m_nSigIDInnovacion;   ///< Siguiente identificador de innovacin disponible.
    int           m_nSigIDNeurona;      ///< Siguiente identificador de neurona disponibla.
};

/// OPERADORES DE ENTRADA Y SALIDA DE FLUJOS
///
ostream& operator<<(ostream& out, const CInnovacion& Inno);
istream& operator>>(istream& in,        CInnovacion& Inno);

#endif
