///////////////////////////////////////////////////////////////////////////////
/// \file  CAlgoritmoGentico.h
/// \brief Archivo de cabecera que define la implementacin de un algoritmo
///        gentico desarrollado con NEAT.
///
/// <b>Fecha Creacin: </b>19 de abril de 2005
/// <b>Autor:          </b>Francisco Jos Gallego Durn
///////////////////////////////////////////////////////////////////////////////
#ifndef __SR_CALGORITMOGENETICO_H__
#define __SR_CALGORITMOGENETICO_H__

/// PRAGMAS
///
/// Para deshabilitar el warning que produce la STL cuando introducimos
/// un tipo de la STL dentro de otro. Slo se queja de que los nombres
/// son demasiado largos para el depurador (por culpa de los pairs).
#pragma warning (disable : 4786)

/// INCLUDES
///
#include <vector>
#include <algorithm>
#include <limits>
#include <iostream>
#include <sstream>
#include "CGenoma.h"
#include "CEspecie.h"
#include "CInnovacion.h"
#include "IIndividuoIA.h"
#include "../DebugLib/DebugLib.h"

using namespace std;

/// DEFINES Y MACROS
///

/// CONSTANTES
///

/// STRUCTS Y TIPOS
///
typedef vector<const CRedNeuronal*> TFenotipos;
typedef vector<CEspecie> TEspecies;

struct SOrdenPtrGenoma
{
    double         dIdoneidad;
    const CGenoma* genoma;

    SOrdenPtrGenoma (double dI, const CGenoma* g) : dIdoneidad (dI), genoma(g) {}

    friend bool operator<(const SOrdenPtrGenoma& g1, const SOrdenPtrGenoma& g2)
    { return (g1.dIdoneidad > g2.dIdoneidad); }
};
///
/// <b>CLASE: </b> CAlgoritmoGenetico
///
/// Clase que gestiona la realizacin de las epocas del algoritmo gentico y <br>
/// la evolucin de los individuos.
///
class CAlgoritmoGenetico
{
public:
    CAlgoritmoGenetico();
    ~CAlgoritmoGenetico() { Limpiar (); }

    // Mtodos de trabajo
    void Limpiar           ();
    void Inicializar       (int nEntradas, int nSalidas);
    void CrearMente        (IIndividuoIA* Individuo);
    void Epoca             (TPoblacionIA& vPoblacion);
    void SalvarPoblacion   (const TPoblacionIA& vPoblacion, const char* ficheroSalida, char keyChar) const;
    void RecuperarPoblacion(TPoblacionIA& vPoblacion, const char* ficheroEntrada, char keyChar);

    // Mtodos GET
    int        GetGeneracion      () const  { return m_nGeneracion; }
    TFenotipos GetMejoresFenotipos();

    // Mtodos SET
    void       SetParametrosPorDefecto          ();
    void       SetEspeciacion                   (bool v)           { m_bEspeciacionActivada = v; }
    void       SetRatioDeCruce                  (double d)         { m_dRatioDeCruce = d; }
    void       SetRatioDeMutacion               (double d)         { m_dRatioDeMutacion = d; }
    void       SetRatioDeMutacionSigmoidea      (double d)         { m_dRatioDeMutacionSigmoidea = d; }
    void       SetMaxNumeroDeEspecies           (unsigned int max) { m_nMaxNumEspecies = max; }
    void       SetMaxNumeroDeNeuronas           (unsigned int max) { m_dMaxNumNeuronas = max; }
    void       SetNumeroIndividuosElite         (unsigned int n)   { m_nIndividuosElite = n; }
    void       SetMaxEpocasPermitidasSinMejora  (unsigned int max) { m_nMaxEpocasPermitidasSinMejora = max; }
    void       SetUmbralCompatibilidadEspecie   (double umbral)    { m_dUmbralCompatibilidad = umbral; }
    void       SetProbabilidadAnyadirNeurona    (double prob)      { m_dProbabilidadAnyadirNeurona = prob; }
    void       SetMaxNumeroIntentosEnlaceAntiguo(unsigned int max) { m_nMaxNumIntentosEnlaceAntiguo; }
    void       SetProbabilidadAnyadirEnlace     (double prob)      { m_dProbabilidadAnyadirEnlace = prob; }
    void       SetProbabilidadAnyadirBucle      (double prob)      { m_dProbabilidadAnyadirBucle = prob; }
    void       SetProbabilidadSustituirPeso     (double prob)      { m_dProbabilidadSustituirPeso = prob; }
    void       SetMaxIntentosAnyadirBucle       (unsigned int max) { m_nMaxIntentosAnyadirBucle = max; }
    void       SetMaxIntentosAnyadirEnlace      (unsigned int max) { m_nMaxIntentosAnyadirEnlace = max; }
    void       SetMaxPerturbacionPeso           (double dmax)      { m_dMaxPerturbacionPeso = dmax; }
    void       SetMaxPerturbacionSigmoidea      (double dmax)      { m_dMaxPerturbacionSigmoidea = dmax; }

private:
    int          m_nIDSigGenoma;            ///< Identificador que se asociar al siguiente genoma que se cree.
    int          m_nIDSigEspecie;           ///< Nmero identificador de la siguiente especie a crear.
    int          m_nGeneracion;             ///< Contador de generaciones transcurridas en la evolucin.
    int          m_nMejorIndividuo;         ///< ndice del mejor individuo de la poblacin.
    int          m_nEntradasRedes;          ///< Nmero de entradas para las redes neuronales.
    int          m_nSalidasRedes;           ///< Nmero de salidas para las redes neuronales.
    bool         m_bMejorIdoneidadValida;   ///< Antes de la primera generacin, no existe mejor idoneidad, empieza a ser vlida cuando se coge la primera.
    double       m_dMejorIdoneidad;         ///< Mejor idoneidad alcanzada por la evolucin hasta la fecha.
    double       m_dIdoneidadTotalAjustada; ///< Suma de las idoneidades ajustadas de la poblacin.
    double       m_dIdoneidadMediaAjustada; ///< Media de las idoneidadea ajustadas.
    CInnovacion* m_pBDInnovaciones;         ///< Puntero a la Base de datos de innovaciones.
    vector<CGenoma> m_vMejoresIndividuos;   ///< Copia de los mejores individuos de la poca anterior
    TEspecies    m_vEspecies;               ///< Vector de especies
    TPoblacionIA*m_vPoblacion;              ///< Puntero al vector de poblacin

    // Parmetros del algoritmo
    unsigned int m_nIndividuosElite;                ///< Nmero de individuos que forman la lite de la especie
    unsigned int m_nMaxEpocasPermitidasSinMejora;   ///< Nmero mximo de pocas que se le permite a una especie evolucionar sin conseguir una mejora
    unsigned int m_nMaxNumEspecies;                 ///< Nmero mximo de especies simultneas posibles
    unsigned int m_dMaxNumNeuronas;                 ///< Mximo nmero de neuronas permitidas para un nico genoma
    unsigned int m_nMaxNumIntentosEnlaceAntiguo;    ///< Mximo nmero de intentos para encontrar un enlace preexistente donde se pueda aadir una nueva neurona
    unsigned int m_nMaxIntentosAnyadirBucle;        ///< Nmero mximo de intentos para encontrar una neurona que admita un bucle
    unsigned int m_nMaxIntentosAnyadirEnlace;       ///< Nmero mximo de intentos para encontrar un par de neuronas donde se pueda aadir un nuevo enlace
    bool         m_bEspeciacionActivada;            ///< Indica si va a haber especies o no durante la evolucin gentica
    double       m_dUmbralCompatibilidad;           ///< El umbral a partir del cual dos individuos se consideran de la misma especie
    double       m_dRatioDeCruce;                   ///< La probabilidad [0,1] de que se produzca un cruce entre 2 individuos preseleccionados
    double       m_dRatioDeMutacion;                ///< La probabilidad [0,1] de que se produzca una mutacin en un peso determinado de un genoma
    double       m_dRatioDeMutacionSigmoidea;       ///< La probabilidad [0,1] de que se produzca una mutacin en el coeficiente de la funcin de activacin sigmoidea
    double       m_dProbabilidadAnyadirNeurona;     ///< La probabilidad [0,1] de que una neurona sea aadida en 1 generacin determinada
    double       m_dProbabilidadAnyadirEnlace;      ///< La probabilidad [0,1] de que se aada un nuevo enlace entre 2 neuronas del genoma
    double       m_dProbabilidadAnyadirBucle;       ///< La probabilidad [0,1] de que se aada un nuevo enlace de una neurona consigo misma (un bucle)
    double       m_dProbabilidadSustituirPeso;      ///< La probabilidad [0,1] de que la mutacin que se produce en un peso sea la sustitucin del mismo por otro nuevo
    double       m_dMaxPerturbacionPeso;            ///< Mximo rango de perturbacin de un peso cuando es mutado (radio del intervalo)
    double       m_dMaxPerturbacionSigmoidea;       ///< Mximo rango de perturbacin del coeficiente de la funcin de activacin sigmoidea cuando es mutado (radio del intervalo)

    /// LOGS DE DEPURACION
/**/ CLog *logEspecies;
/**/ CLog *logGenes;
/**/ CLog *logAG;


    // Mtodos privados
              void OrdenarYSalvarIndividuos       ();
              void ResetearYEliminar              ();
              void AjustarIdoneidadEspecies       ();
              void EspeciarYCalcularHijosAExpandir();
    const CGenoma* TournamentSelection            (unsigned int nComparaciones); // RONALDO: Signed -> Unsigned (20/03/2006)
         CGenoma * Cruce                          (const CGenoma& madre, const CGenoma& padre);

    void AddIDNeurona   (int nID, vector<int> &v) { if (find(v.begin(),v.end(),nID) == v.end()) v.push_back(nID); }
};

#endif

