///////////////////////////////////////////////////////////////////////////////
/// \file  CEspecie.h
/// \brief Clase que agrupa a todos los miembros de una misma especie y controla
///        el ciclo vital del grupo.
///
/// <b>F.Creacin:</b>03 de mayo de 2005.          <br>
/// <b>Autor:     </b>Francisco Jos Gallego Durn.<br>
///////////////////////////////////////////////////////////////////////////////
#ifndef __SR_CESPECIE_H__
#define __SR_CESPECIE_H__

/// PRAGMAS
///
/// Para deshabilitar el warning que produce la STL cuando introducimos
/// un tipo de la STL dentro de otro. Slo se queja de que los nombres
/// son demasiado largos para el depurador (por culpa de los pairs).
#pragma warning (disable : 4786)

/// INCLUDES
///
#include <limits>
#include <vector>
#include "CGenoma.h"

using namespace std;

/// STRUCTS Y TIPOS
///

/// <b> Tipo utilizado para simbolizar el vector de miembros integrantes de una especie </b>
typedef vector<const CGenoma *> TIntegrantes;

/// <b> CLASE: </b> CEspecie <br>
///
/// Clase que agrupa a todos los individuos de una misma especie, gestionando todas<br>
/// las tareas del grupo y controlando su ciclo vital.<br>
///
class CEspecie
{
public:
    CEspecie (const CGenoma &PrimerMiembro, int nIDEspecie);

    // Mtodos SET y ADD
    void AddMiembro (const CGenoma& nuevo);

    // Mtodos de trabajo
    void Limpiar                ();
    void Purgar                 ();
    void AjustarIdoneidades     ();
    void CalcularHijosAExpandir ();

    // Mtodos GET
    const CGenoma& GetSuperviviente  () const;
    const CGenoma& GetLider          () const { return m_IndividuoLider;               }
    double         GetHijosAExpandir () const { return m_dHijosAExpandir;              }
    int            GetNumMiembros    () const { return m_vIndividuos.size();           }
    int            GetEpocasSinMejora() const { return m_nEpocasSinMejora;             }
    int            GetID             () const { return m_nID;                          }
    double         GetIdoneidadLider () const { return m_IndividuoLider.GetIdoneidad();}
    double         GetMejorIdoneidad () const { return m_dMejorIdoneidad;              }
    int            GetEdad           () const { return m_nEdad;                        }

    // Operador para poder ordenar las especies de ms idonea a menos
    friend bool operator<(const CEspecie& Izq, const CEspecie& Dcha)
    {
        return (Izq.m_dMejorIdoneidad > Dcha.m_dMejorIdoneidad);
    }

    // Mtodos SET para los parmetros
    static void SetEdadEspecieJoven (unsigned int e) { ms_nEdadEspecieJoven = e; }
    static void SetEdadEspecieVieja (unsigned int e) { ms_nEdadEspecieVieja = e; }
    static void SetBonusEspecieJoven(double d)       { ms_dBonusEspecieJoven = d; }
    static void SetPenaEspecieVieja (double d)       { ms_dPenaEspecieVieja  = d; }
    static void SetRatioSuperviviencia(double r)     { ms_dRatioSupervivencia = r; }

private:
             int m_nID;             ///< Nmero identificador de la especie
             int m_nEpocasSinMejora;///< Nmero de epocas que han pasado sin que la especie haya mejorado.
             int m_nEdad;           ///< Edad de la especie en pocas.
          double m_dMejorIdoneidad; ///< Mejor puntuacin de idoneidad alcanzada por esta especie
          double m_dHijosAExpandir; ///< Nmero de hijos que debe expandir esta especie para la siguiente poca.
         CGenoma m_IndividuoLider;  ///< El individuo que define la especie. Los dems deben compararse a l para la pertenencia.
    TIntegrantes m_vIndividuos;     ///< Vector de punteros a individuos que pertenecen a esta especie.

    /// Parmetros de control de las especies
    ///
    static unsigned int ms_nEdadEspecieJoven;   ///< Edad hasta la cual se considera que una especie es joven y, por tanto, debe ser potenciada
    static unsigned int ms_nEdadEspecieVieja;   ///< Edad a partir de la cual se considera vieja a la especie y debe ser penalizada
    static double       ms_dBonusEspecieJoven;  ///< Factor por el que se multiplica a una especie joven para potenciar su desarrollo (se multiplica su fitness)
    static double       ms_dPenaEspecieVieja;   ///< Factor por el que se multiplica a una especie vieja para penalizarla (se multiplica su fitness)
    static double       ms_dRatioSupervivencia; ///< Porcentaje de individuos de una especie que pasan a la siguiente generacin inalterados (Elitismo por especie)
};

#endif
