///////////////////////////////////////////////////////////////////////////////
/// \file  CRTBot.cpp
/// \brief Clase preparada para la implementacin de un Bot del Real Time Battle
///
/// \author Francisco Jos Gallego Durn.
/// \date   30 de Junio de 2006
///////////////////////////////////////////////////////////////////////////////
#include "CRTBot.h"

///
/// <b> Constructor por defecto de CRTBot </b><br>
///
/// Construye un robot de Real Time Battle basado en una red neuronal <br>
/// que viene codificada en un string. <br>
///
/// \param botname   Nombre que tendr el bot
/// \param sRed      Red neuronal codificada en un string, para construir un CRedNeuronal de NEAT.
/// \param serverin  Canal de comunicacin para entrada de informacin via streams de C++ con el servidor.
/// \param serverout Canal de comunicacin para salida de informacin via streams de C++ con el servidor.
/// \param entradas  Nmero de entradas que tendr la red neuronal
///
CRTBot::CRTBot (string botname, string sRed, istream& serverin, ostream& serverout, unsigned int entradas)
{
    // Parmetros
    m_sNombreDelBot = botname;
    m_ServerIn      = &serverin;
    m_ServerOut     = &serverout;
    m_bTerminado    = false;

    // Inicializamos la red neuronal
    stringstream ss;
    ss.str(sRed);
    ss >> m_RedNeuronal;

    // Preparamos los vectores
    m_nNumEntradas = entradas;
    m_vEntradas.clear();
    for(int i=0; i < m_nNumEntradas; i++) m_vEntradas.push_back(0.0);
}

///
/// <b> Destructor del Bot </b>
///
CRTBot::~CRTBot () { }

///
/// <b> Mtodo para saber si el Bot ha terminado de ejecutar o no </b>
///
/// \returns true si el bot ha terminado de ejecutar, false en caso contrario
///
bool
CRTBot::HaTerminado ()  { return m_bTerminado; }

///
/// <b> Mtodo de ejecucin del Bot </b>
///
/// A travs de este mtodo, el Bot consulta a su red neuronal qu es lo que debe hacer en este <br>
/// instante y se lo transmite al servidor de Real Time Battle. <br>
///
void
CRTBot::TomarDecisiones()
{
    TObjetoRotable objeto;

    // Procesamos
    m_vSalidas = m_RedNeuronal.Actualizar(m_vEntradas, Ejecucion_PasoDinamico);

    // Ejecutamos
    Girar (OBJ_ROBOT,  (m_vSalidas[0]*2.0)-1.0);                         // entre -1 y 1
    Girar (OBJ_RADAR,  (m_vSalidas[1]*2.0)-1.0);
    Girar (OBJ_CANYON, (m_vSalidas[2]*2.0)-1.0);
    if (m_vSalidas[3] >= 0.5) Acelerar (((m_vSalidas[3]-0.5)*3.0)-0.5);  // entre -0.5 y 1
    else                      Frenar   (m_vSalidas[3]*2.0);              // entre 0 y 1
    Disparar (m_vSalidas[4]);                                            // entre 0 y 1

    // Entradas a 0
    for (int i=0; i < m_vEntradas.size(); i++) m_vEntradas[i] = 0.0;
}

///
/// <b> Mtodo de inicializacin </b>
///
/// Este mtodo sirve para que el bot mande la secuencia de inicializacin al servidor, <br>
/// preparndose as para la batalla.
///
void
CRTBot::Inicializar()
{
  (*m_ServerOut) << "Name " << m_sNombreDelBot << endl;
  (*m_ServerOut) << "Colour FF0000 0000FF"     << endl;
  (*m_ServerOut).flush();
}

///
/// <b> Obtencin de valores del radar </b>
///
/// Este mtodo debe ser llamado cada vez que haya un mensaje de RADAR procedente del servidor, <br>
/// esperando para ser procesado.
///
void
CRTBot::Radar()
{
    const double dMaxDistancia = 50.0;
    const double dMaxAngulo    = 6.28;
    double distancia, angulo;
    int    objeto;
    (*m_ServerIn) >> distancia >> objeto >> angulo;

    // Convierte la seal del radar en impulsos para la red neuronal
    if (objeto >= 0 && objeto < 5) SetEntrada (objeto, distancia);
    SetEntrada (IN_ANGULORADAR, angulo);
}


///
/// <b> Energa restante </b>
///
/// Mediante este mtodo se procesa el mensaje Energy, que indica al robot su energa restante.
///
void
CRTBot::Energy()
{
    double energylevel;
    (*m_ServerIn) >> energylevel;

    // Convierte la seal de info en impulsos para la red neuronal
    SetEntrada (IN_ENERGIABOT, energylevel);
}

///
/// <b> Informacin adicional </b>
///
/// Este mtodo procesa el mensaje Info, que viene siempre despus del mensaje de Radar, y aporta <br>
/// ms informacin al robot.
///
void
CRTBot::Info()
{
    double tiempo, velocidad, angulocanyon;
    (*m_ServerIn) >> tiempo >> velocidad >> angulocanyon;

    // Convierte la seal de info en impulsos para la red neuronal
    SetEntrada (IN_ANGULOCANYON, angulocanyon);
    SetEntrada (IN_VELOCIDAD   ,    velocidad);
    SetEntrada (IN_TIEMPO      ,       tiempo);
}

///
/// <b> Informacin sobre el robot detectado con el radar </b>
///
/// Con este mtodo se procesa el mensaje RobotInfo, que nos indica datos sobre el robot <br>
/// que se acaba de detectar con el radar.
///
void
CRTBot::RobotInfo()
{
    double energylevel;
    int    teammate;
    (*m_ServerIn) >> energylevel >> teammate;

    // Convierte la seal de info en impulsos para la red neuronal
    SetEntrada (IN_ENERGIAOTRO, energylevel);
    SetEntrada (IN_COMPANYERO , teammate);
}

///
/// <b> Establece el valor de una entrada de la red neuronal </b>
///
/// \param nEntrada Nmero de la entrada a establecer
/// \param valor    Valor double al que ser establecida
///
void
CRTBot::SetEntrada (unsigned int nEntrada, double valor)
{
    if (nEntrada < m_vEntradas.size()) m_vEntradas[nEntrada] = valor;
}

///
/// <b> Fuerza al Bot a terminar su ejecucin </b>
///
void
CRTBot::TerminarBot() { m_bTerminado = true; }

///
/// <b> Robot efectuando un disparo </b>
///
/// Mediante este mtodo el bot realiza un disparo con la fuerza deseada.
///
/// \param valor Fuerza con la que el robot desea disparar, normalizada de 0 a 1. Valores inferiores a 0.5 significan no disparar.
///
void
CRTBot::Disparar(double valor)
{
    const double dRobotMaxFireEnergy = 30.0;
    if (valor >= 0.5)
        *(m_ServerOut) << "Shoot " << (((valor-0.5)*2.0)*dRobotMaxFireEnergy) << endl;
}

///
/// <b> Robot acelerando </b>
///
/// Con este mtodo el robot acelera su marcha en sentido positivo o negativo.
///
/// \param valor Incremento de velocidad, entre -0.5 y 1.0
///
void
CRTBot::Acelerar(double valor)
{
    const double dAceleracionMax = 2.0;
    if (valor >= 0) valor *= dAceleracionMax;
    if (valor >= -0.5)
        *(m_ServerOut) << "Accelerate " << valor << endl;
}

///
/// <b> Robot frenando </b>
///
/// El robot pisa el freno con la fuerza que desea.
///
/// \param valor Fuerza con la que se pisa el freno, de 0 a 1
///
void
CRTBot::Frenar (double valor)
{
    if (valor > 0) *(m_ServerOut) << "Brake " << valor << endl;
}

///
/// <b> Girar articulacin </b>
///
/// Este mtodo sirve para girar al propio robot, su can y su radar con la aceleracin <br>
/// de giro deseada.<br>
///
/// \param TObjetoRotable Uno de los objetos rotables posibles (RADAR, CAON, ROBOT...)
/// \param valor          Aceleracin que se le quiere dar al objeto rotable, normalizada de 0 a 1
///
void
CRTBot::Girar(TObjetoRotable objeto, double valor)
{
/*    const double dMaxRotacionRobot  = 0.785398;
    const double dMaxRotacionRadar  = 2.0944;
    const double dMaxRotacionCanyon = 1.5708;
*/
    const double dMaxRotacionComun = 0.785398;
    valor *= dMaxRotacionComun;
/*    switch (objeto)
    {
        case OBJ_ROBOT:  { valor *= dMaxRotacionRobot;  break; }
        case OBJ_RADAR:  { valor *= dMaxRotacionRadar;  break; }
        case OBJ_CANYON: { valor *= dMaxRotacionCanyon; break; }      
    }
*/
    *(m_ServerOut) << "Rotate " << objeto << " " << valor << endl;
}



