#!/usr/bin/env python
# -*- encoding: iso-8859-1 -*-

import sys
import math

######################################################################
# Constants del joc
######################################################################

ROBOT_OPTIONS = {
    'SEND_SIGNAL':                0,
    'SEND_ROTATION_REACHED':    1,
    'SIGNAL':                    2,
    'USE_NON_BLOCKING':            3}

OBJECT_TYPE = {
    'NO_OBJECT':            -1,
    'ROBOT':                0,
    'SHOT':                    1,
    'WALL':                    2,
    'COOKIE':                3,
    'MINE':                    4,
    'LAST_OBJECT_TYPE':        5}

ROBOT_PARTS = {
    'ROBOT':    1,
    'CANNON':    2,
    'RADAR':    4}

GAME_OPTIONS = {
    'ROBOT_MAX_ROTATE':                0,
    'ROBOT_CANNON_MAX_ROTATE':        1,
    'ROBOT_RADAR_MAX_ROTATE':        2,
    'ROBOT_MAX_ACCELERATION':        3,
    'ROBOT_MIN_ACCELERATION':        4,
    'ROBOT_START_ENERGY':            5,
    'ROBOT_MAX_ENERGY':                6,
    'ROBOT_ENERGY_LEVELS':            7,
    'SHOT_SPEED':                    8,
    'SHOT_MIN_ENERGY':                9,
    'SHOT_MAX_ENERGY':                10,
    'SHOT_ENERGY_INCREASE_SPEED':    11,
    'TIMEOUT':                        12,
    'DEBUG_LEVEL':                    13,
    'SEND_ROBOT_COORDINATES':        14}

######################################################################
# Joc
######################################################################

class RealTimeBattleGame:
    def __init__ (self, rtb):
        self.rtb = rtb

        self.remainingRobots = 0
        self.elapsedTime = 0
        self.options = {}

        self.rtb.eventManager.addListener("RobotsLeft", self.remainingRobotsListener)
        self.rtb.eventManager.addListener("Info", self.infoListener)
        self.rtb.eventManager.addListener("GameOption", self.gameOptionListener)

    def remainingRobotsListener (self, remainingRobots):
        self.remainingRobots = int(remainingRobots)

    def infoListener (self, elapsedTime, speed, cannonAngle):
        self.elapsedTime = float(elapsedTime)

    def gameOptionListener (self, option, value):
        self.rtb.debug("%d = %f" % (option, value))
        value = value.replace(",", ".")
        self.options[int(option)] = float(value)

    def getOption (self, optionName):
        option = GAME_OPTIONS[optionName]
        return self.options[option]

######################################################################
# RTB
######################################################################

class RealTimeBattle:
    def __init__ (self):
        self.input = sys.stdin
        self.output = sys.stdout
        self.eventManager = EventManager()
        self.game = None

        self.eventManager.addListener("GameStarts", self.gameStartsListener)
        self.eventManager.addListener("GameFinishes", self.gameFinishesListener)
        self.eventManager.addListener("ExitRobot", self.exitRobotListener)
        self.eventManager.addListener("Warning", self.warningListener)

    def sendMessage (self, message):
        self.output.write(message + "\n")
        self.output.flush()

    def processEvents (self):
        while 1:
            message = self.input.readline()

            # EOF?
            if len(message) == 0: break

            # Eliminem els salts de línia
            while len(message) > 0 and message[-1] == "\n": message = message[:-1]

            # Processem el missatge
            messageParts = message.split()
            if len(messageParts) == 0: continue
            event = messageParts[0]
            parameters = messageParts[1:]

            # self.debug("%s(%s)" % (event, ", ".join(parameters)))

            # Mirem si hi ha algun mètode registrat per l'event
            if not self.eventManager.events.has_key(event): self.debug("No s'ha tractat l'event %s" % event)
            self.eventManager.executeListeners(event, parameters)

    def debug (self, message):
        self.sendMessage("Debug %s" % message)

    def gameStartsListener (self):
        self.game = RealTimeBattleGame(self)

    def gameFinishesListener (self):
        self.game = None

    def exitRobotListener (self):
        sys.exit(0)

    def warningListener (self, message):
        self.debug("WARNING: %s" % message)

######################################################################
# Gestor d'events
######################################################################

class EventManager:
    def __init__ (self):
        self.events = {}

    def addListener (self, event, listener):
        if not self.events.has_key(event): self.events[event] = []
        self.events[event].append(listener)

    def getListeners (self, event):
        if self.events.has_key(event): return self.events[event]
        return []

    def executeListeners (self, event, parameters):
        for listener in self.getListeners(event):
            try:
                listener(*parameters)
            except TypeError:
                pass

######################################################################
# Robot
######################################################################

class Robot:
    def __init__ (self):
        self.rtb = RealTimeBattle()
        self.name = "Sense nom"
        self.color = "FF3300"
        self.cannon = RobotCannon(self)
        self.radar = RobotRadar(self)
        self.energy = 0
        self.speed = 0
        self.collisionObjectType = OBJECT_TYPE['NO_OBJECT']
        self.collisionAngle = 0

        self.rtb.sendMessage("RobotOption %d %d" % (ROBOT_OPTIONS["USE_NON_BLOCKING"], 0))

        self.rtb.eventManager.addListener("Initialize", self.initializeListener)
        self.rtb.eventManager.addListener("Energy", self.energyListener)
        self.rtb.eventManager.addListener("Info", self.infoListener)
        self.rtb.eventManager.addListener("Collision", self.collisionListener)
        self.rtb.eventManager.addListener("YourName", self.yourNameListener)
        self.rtb.eventManager.addListener("YourColour", self.yourColourListener)

    def initializeListener (self, first):
        if first:
            self.rtb.sendMessage("Name %s" % self.name)
            self.rtb.sendMessage("Colour %s" % self.color)

    def energyListener (self, energy):
        self.energy = float(energy)

    def accelerate (self, acceleration):
        self.brake(0)
        self.rtb.sendMessage("Accelerate %f" % acceleration)

    def brake (self, portion):
        self.rtb.sendMessage("Brake %f" % portion)

    def rotateAmount (self, angle, velocity = None):
        if velocity == None: velocity = self.rtb.game.getOption('ROBOT_MAX_ROTATE')
        self.rtb.sendMessage("RotateAmount %d %f %f" % (ROBOT_PARTS['ROBOT'], velocity, angle))

    def rotate (self, angularVelocity):
        self.rtb.sendMessage("Rotate %d %f" % (ROBOT_PARTS['ROBOT'], angularVelocity))

    def infoListener (self, timeElapsed, speed, cannonAngle):
        self.speed = float(speed)

    def collisionListener (self, objectType, angle):
        self.collisionObjectType = int(objectType)
        self.collisionAngle = float(angle)

    def start (self):
        self.rtb.processEvents()

    def yourNameListener (self, name):
        self.name = name

    def yourColourListener (self, color):
        self.color = color

######################################################################
# Canó del robot
######################################################################

class RobotCannon:
    def __init__ (self, robot):
        self.robot = robot
        self.angle = 0
        self.robot.rtb.eventManager.addListener("Info", self.infoListener)

    def shoot (self, energy):
        self.robot.rtb.sendMessage("Shoot %f" % energy)

    def infoListener (self, time, speed, angle):
        self.angle = float(angle)

    def sweep (self, angle, velocity = None):
        if velocity == None: velocity = 1
        self.robot.rtb.sendMessage("Sweep %d %f %f %f" % (ROBOT_PARTS['CANNON'], velocity, angle / -2, angle / 2))

######################################################################
# Radar del Robot
######################################################################

class RobotRadar:
    def __init__ (self, robot):
        self.robot = robot
        self.objectDistance = 0
        self.objectType = OBJECT_TYPE['NO_OBJECT']
        self.angle = 0
        self.robot.rtb.eventManager.addListener("Radar", self.radarListener)

    def radarListener (self, objectDistance, objectType, angle):    
        self.objectDistance = float(objectDistance)
        self.objectType = int(objectType)
        self.angle = float(angle)

    def sweep (self, angle, velocity = None):
        # self.robot.rtb.debug("ROBOT_RADAR_MAX_ROTATE = %f" % self.robot.rtb.game.getOption('ROBOT_RADAR_MAX_ROTATE'))
        if velocity == None: velocity = 1
        # if velocity == None: velocity = self.robot.rtb.game.getOption('ROBOT_RADAR_MAX_ROTATE')
        self.robot.rtb.sendMessage("Sweep %d %f %f %f" % (ROBOT_PARTS['RADAR'], velocity, angle / -2, angle / 2))

    def rotateTo (self, angle, velocity):
        self.robot.rtb.sendMessage("RotateTo %d %f %f %f" % (ROBOT_PARTS['RADAR'], velocity, angle))

class Almogaver (Robot):
    def __init__ (self):
        Robot.__init__(self)

        self.name = "r00z"
        self.color = "FFCC00"
        self.seen = 0
        self.radar.lastAngle = 0
        self.rotation = 1
        self.direction = 1
        self.shots = 0

        self.rtb.eventManager.addListener("GameStarts", self.gameStartsListener)
        self.rtb.eventManager.addListener("Info", self.infoListener)

    def gameStartsListener (self):
        self.accelerate(2)
        self.rotate(2)

    def infoListener (self, *arguments):
        if self.radar.angle < self.radar.lastAngle:
            self.rotation = 1
        else:
            self.rotation = -1

        if self.radar.objectType == OBJECT_TYPE['ROBOT']:
            self.seen = 10
            self.shots = (30 - self.radar.objectDistance)
            self.rotate(0)
            self.accelerate(2)
        elif self.energy < 80 and self.radar.objectType == OBJECT_TYPE['COOKIE']:
            self.rotate(0)
            self.accelerate(2)
        elif self.radar.objectType == OBJECT_TYPE['MINE'] and self.radar.objectDistance < 5:
            self.cannon.shoot(1)
            self.brake(10)
        elif self.radar.objectType == OBJECT_TYPE['SHOT']:
            #  self.rotate(0.1)
            pass
        elif self.collisionObjectType == OBJECT_TYPE['SHOT']:
            self.rotate(self.direction * 2)
            self.accelerate(2)
        else:
            if self.seen > 0:
                self.seen -= 1
                self.direction = -self.rotation
                # self.rotate(self.direction * 1)
            else:
                self.rotate(self.direction * 2)
            self.brake(1000)
            if self.seen > 0:
                self.radar.sweep(math.pi / 5)
                self.radar.sweep(math.pi / 5)
            else:
                self.cannon.sweep(math.pi / 4)
                self.cannon.sweep(math.pi / 4)

        if self.shots > 0:
            self.shots -= 1
            self.cannon.shoot(2)

if __name__ == '__main__':
    Almogaver().start()
