#include <GL/gl.h>
#include <GL/glu.h>
#include <SDL/SDL.h>
#include <SDL/SDL_image.h>

#include <cstdlib>
#include <cmath>
#include <iostream>

#include "graphics.hpp"
#include "vector.hpp"

/** 
 * Calculates the normal for the specified face and stores it in normal.
 *
 *  |  i  j  k |
 *  | x0 y0 z0 |
 *  | x1 y1 z1 |
 *
 *  i -> y0z1 - z0y1
 *  j -> x1z0 - x0z1
 *  k -> x0y1 - x1y0
 *
 * @param normal the output normal to calculate for the specified face
 * @param face the input face, pointing to coords
 * @param coords array of coordinates
 */ 
void calc_face_normal(IndexFace3& face, Vector3* coords) {
  int v0 = face.v0;
  int v1 = face.v1;
  int v2 = face.v2;

  float x0 = coords[v2].x - coords[v0].x;
  float y0 = coords[v2].y - coords[v0].y;
  float z0 = coords[v2].z - coords[v0].z;

  float x1 = coords[v1].x - coords[v0].x;
  float y1 = coords[v1].y - coords[v0].y;
  float z1 = coords[v1].z - coords[v0].z;

  float nx = y0 * z1 - z0 * y1;
  float ny = x1 * z0 - x0 * z1;
  float nz = x0 * y1 - x1 * y0;

  // Normalize the result
  float length = sqrt(nx * nx + ny * ny + nz * nz);
  if (length < 0.000001) {
    length = 100.0;
  } else {
    length = 1.0 / length;
  }

  face.normal.x = nx * length;
  face.normal.y = ny * length;
  face.normal.z = nz * length;
}

/**
 * Determines if x is a power of 2.
 * @param x the value to check
 * @return 1 if x is a power of 2, 0 if it is not.
 */
int isPowerOf2(int x) {
  return (x & (x - 1)) == 0;
}

void add_alpha(char* target, int width, int height, char* source) {
  for (int i = 0; i < height; i++) {
    for (int j = 0; j < width; j++) {
      int offset = i * width + j;
      int red = source[offset * 3 + 0];
      int green = source[offset * 3 + 1];
      int blue = source[offset * 3 + 2];
      target[offset * 4 + 0] = red;
      target[offset * 4 + 1] = green;
      target[offset * 4 + 2] = blue;
      int alpha = 0x00;
      if ((red == 0) && (green == 0) && (blue == 0)) {
        alpha = 0xff;
      }
      target[offset * 4 + 3] = alpha;
    }
  }
}

/**
 * Loads a texture from a file and installs it in OpenGL.
 * @param filename the file to load, typically "foo.png"
 * @return an OpenGL texture identifier
 */
GLuint load_texture_from_file(const char *filename, int mask) {
  GLuint m_glTexture;
  glGenTextures(1, &m_glTexture);

  glBindTexture(GL_TEXTURE_2D, m_glTexture);

  SDL_Surface *textureData = IMG_Load(filename);
  if (!textureData)
    {
      char buf[100];
      sprintf(buf,"error loading texture '%s'", filename );
      perror(buf);
      exit(1);
    }
  SDL_LockSurface(textureData);

  if (textureData->w != textureData->h) {
    printf("Warning: Texture width (%d) and height (%d) are not the same.  This might be slow on some cards?.\n", 
	   textureData->w, textureData->h);
  }
  if (!isPowerOf2(textureData->w)) {
    printf("Warning: Texture width (%d) is not a power of 2.  This might be slow on some cards.\n", textureData->w);
  }
  if (!isPowerOf2(textureData->h)) {
    printf("Warning: Texture height (%d) is not a power of 2.  This might be slow on some cards.\n", textureData->h);
  }

  char* data = (char*) malloc(4 * textureData->w * textureData->h);
  add_alpha(data, textureData->w, textureData->h, (char*) textureData->pixels);

  glTexImage2D(GL_TEXTURE_2D,
               0,
               GL_RGBA,
               textureData->w,
               textureData->h,
               0,
               GL_RGBA,
               GL_UNSIGNED_BYTE,
	       data);

  SDL_UnlockSurface(textureData);
  SDL_FreeSurface(textureData);

//   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
//   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);

  return m_glTexture;
}

void draw_faces(IndexFace3* faces, int nFaces, Vector3* coords) {
  glBegin(GL_TRIANGLES);
  for (int i = 0; i < nFaces; i++) {
    int v0 = faces[i].v0;
    int v1 = faces[i].v1;
    int v2 = faces[i].v2;

    Vector3 c0 = coords[v0];
    Vector3 c1 = coords[v1];
    Vector3 c2 = coords[v2];

    glNormal3f(faces[i].normal.x, faces[i].normal.y, faces[i].normal.z);
    glVertex3f(c0.x, c0.y, c0.z);
    glVertex3f(c1.x, c1.y, c1.z);
    glVertex3f(c2.x, c2.y, c2.z);
  }
  glEnd();
}

/**
 * Reset OpenGL settings to the default drawing mode.
 */
void reset_gl() {
  glLoadIdentity();

  glEnable(GL_DEPTH_TEST);
  glDepthFunc(GL_LEQUAL);

  glDisable(GL_CULL_FACE);
  glDisable(GL_BLEND);

  glDisable(GL_POLYGON_SMOOTH);
}

/**
 * Set OpenGL settings for polygon smoothing.
 */
void set_gl_polygon_smoothing() {
  glEnable(GL_POLYGON_SMOOTH);

  glCullFace(GL_BACK);
  glEnable(GL_CULL_FACE);

  glBlendFunc(GL_SRC_ALPHA_SATURATE, GL_ONE);
  glEnable(GL_BLEND);
  glDisable(GL_DEPTH_TEST);
}

void set_gl_lighting() {
  glEnable(GL_LIGHT1);
  glEnable(GL_LIGHTING);

  GLfloat lightAmbient[] = { 0.3f, 0.2f, 0.5f, 1.0f };
  GLfloat lightDiffuse[] = { 0.5f, 0.4f, 0.7f, 1.0f };
  GLfloat lightPosition[] = { 0.0f, 0.0f, 2.0f, 1.0f };

  glLightfv(GL_LIGHT1, GL_AMBIENT, lightAmbient);
  glLightfv(GL_LIGHT1, GL_DIFFUSE, lightDiffuse);
  glLightfv(GL_LIGHT1, GL_POSITION, lightPosition);

  glDisable(GL_COLOR_MATERIAL);
  glColor4f(1.0, 1.0, 1.0, 1.0);
  glEnable(GL_LIGHTING);

  glShadeModel(GL_SMOOTH);
  glClearDepth(1.0f);
  glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST );

  glAlphaFunc(GL_ALWAYS, 0.0);
  glDisable(GL_BLEND);
}
