
from effect import Effect
from core import GLSLShader, Texture


def crossProduct(v0, v1):

    x = v0[1]*v1[2] - v0[2] * v1[1]
    y = -(v0[0]*v1[2] - v0[2] * v1[0])
    z = v0[0]*v1[1] - v0[1] * v1[0]

    return (x,y,z)

def subtract(v0, v1):
    return (v0[0] - v1[0], v0[1] - v1[1], v0[2] - v1[2])
    

def add(v0, v1):
    return (v0[0] + v1[0], v0[1] + v1[1], v0[2] + v1[2])


def scale(v, scale):
    return [x * scale for x in v]
    

FO_NONE, FO_EXTRUDE, FO_FAN = range(3)

class Face:
    def __init__(self, verts = [], faceOp = FO_NONE, depth = 0):
        self.verts = verts
        self.depth = depth
        self.faceOp = faceOp

    def calcNormal(self):
        v0 = subtract(self.verts[1], self.verts[0])
        v1 = subtract(self.verts[2], self.verts[0])
        return crossProduct(v0, v1)

    def calcCenter(self):
        scale = 1.0 / len(self.verts)
        center = (0,0,0)
        for vertex in self.verts:
            center = add(center, vertex)

        return center.scale(center, scale)

    def draw(self):

        faceCenter = self.calcCenter()
        vertexCount = len(face.verts)

        for i in xrange(vertexCount):
            v0 = face.verts[i]
            v1 = face.verts[i % vertexCount]
            glVertex3f(*v0)
            glVertex3f(*v1)
            glVertex3f(*faceCenter)

class MeshGenerator:

    def __init__(self):
        self.faces = []

    def addFace(self, face):
        self.faces.append(face)

    def extrudeFace(self, face, length = .1, scale = 1.0):

        vertexCount = len(face.verts)
        faceNormal = face.calcNormal()

        # process sides
        for i in xrange(vertexCount):
            v0 = face.verts[i]
            v1 = face.verts[i % vertexCount]
            v2 = add(v1, faceNormal)
            v3 = add(v0, faceNormal)

            self.addFace(Face(verts = [v0, v1, v2, v3]), depth = face.depth + 1)

        # process top
        topVertices = []
        for vertex in face.verts:
            topVertices.append(add(vertex, faceNormal), depth = face.depth + 1)

        self.addFace(Face(verts = topVertices))
            
    def fanFace(self, face, length = 0):

        vertexCount = len(face.verts)
        faceCenter = face.calcCenter()
        faceNormal = face.calcNormal()

        for i in xrange(vertexCount):
            v0 = face.verts[i]
            v1 = face.verts[i % vertexCount]
            v2 = add(faceCenter, scale(faceNormal, length))

            self.addFace(Face(verts = [v0, v1, v2]), depth = face.depth + 1)


    def generatePlane(self):

        v0 = (-1, 0, 1)
        v1 = (1, 0, 1)
        v2 = (1, 0, -1)
        v3 = (-1, 0, -1)
        
        self.addFace(Face(verts = [v0, v1, v2, v3]))

    def drawMesh(self):
        glBegin(GL_TRIANGLES)
        for face in self.faces:
            face.draw()
        glEnd()


class Extruder(Effect):

    def __init__(self):
        self._mesh = MeshGenerator()
        self._mesh.generatePlane()

    def draw(self, time, delta, start, stop):
        self._mesh.drawMesh()

        
