# util.py

"""Utitily function for Dice3DS.

Defines some routines for calculating normals and transforming points.

"""

import Numeric as N


def translate_points(pointarray,matrix):
	"""Translate points in pointarray by the given matrix.

	    tpointarray = translate_points(pointarray,matrix)

        Takes array of points and a homogenous (4D) transformation
        matrix right out of the 3DS DOM.

	Returns a pointarray with the points transformed by the matrix.

	"""
	
	n = len(pointarray)
	pt = N.ones((n,4),N.Float)
	pt[:,:3] = pointarray
	tpt = N.transpose(N.matrixmultiply(matrix,N.transpose(pt)))
	return N.array(tpt[:,:3]/tpt[:,3:4])



def calculate_normals_no_smoothing(pointarray,facearray,smarray=None):
	"""Calculate normals all perpendicular to the faces.

	    points,norms = calculate_normals_no_smoothing(
	            pointarray,facearray,smarray=None)

        Takes an array of points and faces right out the the 3DS DOM.
        It accepts a smoothing array, but ignores it.

	Returns a Numeric.array of points, one per row, and a
        Numeric.array of the corresponding normals.  The points are
        returned as a list of consecutive triangles; the first three
        rows make up the first triangle, the second three rows make up
        the second triangle, and so on.  (This is because a single
        point will end up with different normals.)
	
	The normal vectors are calculated by calculating the normal at
	each face.  There is no smoothing.

	"""

	# prepare to calculate normals. define some arrays

	m = len(facearray)
	p = N.zeros((m,3),N.Float)
	fnorms = N.zeros((m*3,3),N.Float)
	points = N.take(pointarray,facearray.flat)

	# calculate normals for each face
	
	A = N.take(pointarray,facearray[:,0])
	B = N.take(pointarray,facearray[:,1])
	C = N.take(pointarray,facearray[:,2])
	b = A - C
	c = B - A		
	fnorms[2::3,0] = c[:,2]*b[:,1]-c[:,1]*b[:,2]
	fnorms[2::3,1] = c[:,0]*b[:,2]-c[:,2]*b[:,0]
	fnorms[2::3,2] = c[:,1]*b[:,0]-c[:,0]*b[:,1]
	fnorms[0::3] = fnorms[1::3] = fnorms[2::3]

	# we're done

	return points, fnorms



def calculate_normals_by_cross_product(pointarray,facearray,smarray):
	"""Calculate normals by smoothing, weighting by cross-product.

	    points,norms = calculate_normals_by_cross_product(
	            pointarray,facearray,smarray)

        Takes an array of points, faces, and a smoothing group right
        out the the 3DS DOM.

	Returns a Numeric.array of points, one per row, and a
        Numeric.array of the corresponding normals.  The points are
        returned as a list of consecutive triangles; the first three
        rows make up the first triangle, the second three rows make up
        the second triangle, and so on.  (This is because a single
        point can end up with different normals.)

	To calculate the normal of a given vertex on a given face,
	this function averages the normal vector for all faces which
	have share that vertex, and a smoothing group.

	The normals being averaged are weighted by the cross-product
	used to obtain the face's formal, which is proportional to the
	area of the face.

	"""

	# prepare to calculate normals. define some arrays

	m = len(facearray)
	p = N.zeros((m,3),N.Float)
	rnorms = N.zeros((m*3,3),N.Float)
	fnorms = N.zeros((m*3,3),N.Float)
	points = N.take(pointarray,facearray.flat)
	exarray = N.zeros(3*m,N.UnsignedInt32)
	if smarray:
		exarray[0::3] = exarray[1::3] = exarray[2::3] = smarray

	# calculate scaled normals (according to angle subtended)

	A = N.take(pointarray,facearray[:,0])
	B = N.take(pointarray,facearray[:,1])
	C = N.take(pointarray,facearray[:,2])
	a = C - B
	b = A - C
	c = B - A		
	rnorms[0::3,0] = c[:,2]*b[:,1]-c[:,1]*b[:,2]
	rnorms[0::3,1] = c[:,0]*b[:,2]-c[:,2]*b[:,0]
	rnorms[0::3,2] = c[:,1]*b[:,0]-c[:,0]*b[:,1]
	rnorms[1::3,0] = a[:,2]*c[:,1]-a[:,1]*c[:,2]
	rnorms[1::3,1] = a[:,0]*c[:,2]-a[:,2]*c[:,0]
	rnorms[1::3,2] = a[:,1]*c[:,0]-a[:,0]*c[:,1]
	rnorms[2::3,0] = b[:,2]*a[:,1]-b[:,1]*a[:,2]
	rnorms[2::3,1] = b[:,0]*a[:,2]-b[:,2]*a[:,0]
	rnorms[2::3,2] = b[:,1]*a[:,0]-b[:,0]*a[:,1]

	# normalize vectors according to passed in smoothing group

	vecmap = {}
	for i in xrange(3*m):
		tp = tuple(points[i])
		if vecmap.has_key(tp):
			vecmap[tp].append(i)
		else:
			vecmap[tp] = [i]

	for i in xrange(3*m):
		rgroup = vecmap[tuple(points[i])]
		group = [ x for x in rgroup if exarray[i]&exarray[x] ]
		if not group:
			group = [i]
		q = N.sum(N.take(rnorms,group))
		lq = N.sqrt(N.dot(q,q))
		if lq: q /= lq
		fnorms[i] = q

	# we're done

	return points, fnorms



def calculate_normals_by_angle_subtended(pointarray,facearray,smarray):
	"""Calculate normals by smoothing, weighting by angle subtended.

	    points,norms = calculate_normals_by_angle_subtended(
	            pointarray,facearray,smarray)

        Takes an array of points, faces, and a smoothing group right
        out the the 3DS DOM.

	Returns a Numeric.array of points, one per row, and a
        Numeric.array of the corresponding normals.  The points are
        returned as a list of consecutive triangles; the first three
        rows make up the first triangle, the second three rows make up
        the second triangle, and so on.  (This is because a single
        point can end up with different normals.)

	To calculate the normal of a given vertex on a given face,
	this function averages the normal vector for all faces which
	have share that vertex, and a smoothing group.

	The normals being averaged are weighted by the angle
	subtended.

	"""

	# define this enorm function as a convenience

	def _enorm(vecarray):
		return N.maximum(N.sqrt(vecarray[:,0]**2
					+ vecarray[:,1]**2
					+ vecarray[:,2]**2),
				 1.0e-10)
			     
	# prepare to calculate normals. define some arrays

	m = len(facearray)
	p = N.zeros((m,3),N.Float)
	rnorms = N.zeros((m*3,3),N.Float)
	fnorms = N.zeros((m*3,3),N.Float)
	points = N.take(pointarray,facearray.flat)
	exarray = N.zeros(3*m,N.UnsignedInt32)
	if smarray:
		exarray[0::3] = exarray[1::3] = exarray[2::3] = smarray

	# calculate scaled normals (according to angle subtended)

	A = N.take(pointarray,facearray[:,0])
	B = N.take(pointarray,facearray[:,1])
	C = N.take(pointarray,facearray[:,2])
	a = C - B
	b = A - C
	c = B - A		
	p[:,0] = c[:,2]*b[:,1]-c[:,1]*b[:,2]
	p[:,1] = c[:,0]*b[:,2]-c[:,2]*b[:,0]
	p[:,2] = c[:,1]*b[:,0]-c[:,0]*b[:,1]
	la = _enorm(a)
	lb = _enorm(b)
	lc = _enorm(c)
	lp = _enorm(p)
	sinA = N.clip(lp/lb/lc,-1.0,1.0)
	sinB = N.clip(lp/la/lc,-1.0,1.0)
	sinC = N.clip(lp/la/lb,-1.0,1.0)
	sinA2 = sinA*sinA
	sinB2 = sinB*sinB
	sinC2 = sinC*sinC
	angA = N.arcsin(sinA)
	angB = N.arcsin(sinB)
	angC = N.arcsin(sinC)
	angA = N.where(sinA2 > sinB2 + sinC2, N.pi - angA, angA)
	angB = N.where(sinB2 > sinA2 + sinC2, N.pi - angB, angB)
	angC = N.where(sinC2 > sinA2 + sinB2, N.pi - angC, angC)
	rnorms[0::3] = p*(angA/lp)[:,N.NewAxis]
	rnorms[1::3] = p*(angB/lp)[:,N.NewAxis]
	rnorms[2::3] = p*(angC/lp)[:,N.NewAxis]

	# normalize vectors according to passed in smoothing group

	vecmap = {}
	for i in xrange(3*m):
		tp = tuple(points[i])
		if vecmap.has_key(tp):
			vecmap[tp].append(i)
		else:
			vecmap[tp] = [i]

	for i in xrange(3*m):
		rgroup = vecmap[tuple(points[i])]
		group = [ x for x in rgroup if exarray[i]&exarray[x] ]
		if not group:
			group = [i]
		q = N.sum(N.take(rnorms,group))
		lq = N.sqrt(N.dot(q,q))
		if lq: q /= lq
		fnorms[i] = q

	# we're done

	return points, fnorms
