# glmodel.py

"""Classes for rendering 3DS models in OpenGL.

Defines some classes (based on Dice3DS.example.basicmodel) with some
additional methods to draw the model in OpenGL, or create a display
list to do so. Requires PyOpenGL.

"""

from Dice3DS import dom3ds
from Dice3DS.example import basicmodel, gltexture
from Dice3DS.example.GL import *


class GLMaterial(basicmodel.BasicMaterial):
	"""Subclass of BasicMaterial that sets OpenGL material properties."""

	def __init__(self,*args,**kwargs):
		super(GLMaterial,self).__init__(*args,**kwargs)

	def set_material(self,has_tverts):
		if self.twosided:
			side = GL_FRONT_AND_BACK
		else:
			side = GL_FRONT
		glMaterialfv(side,GL_AMBIENT,self.ambient)
		glMaterialfv(side,GL_DIFFUSE,self.diffuse)
		glMaterialfv(side,GL_SPECULAR,self.specular)
		glMaterialf(side,GL_SHININESS,self.shininess*128.0)
		if self.texture and has_tverts:
			self.texture.enable()
			self.texture.bind()

	def unset_material(self,has_tverts):
		if self.texture and has_tverts:
			self.texture.disable()


class GLMesh(basicmodel.BasicMesh):
	"""Subclass of BasicMesh that renders the mesh in OpenGL."""

	def __init__(self,*args,**kwargs):
		super(GLMesh,self).__init__(*args,**kwargs)

	def render_normals(self):
		glPushAttrib(GL_LIGHTING_BIT)
		glDisable(GL_LIGHTING)
		glColor3f(1.0,1.0,0.0)
		glBegin(GL_LINES)
		for i in xrange(len(self.points)):
			glVertex3fv(self.points[i])
			glVertex3fv(self.points[i] + 0.1*self.norms[i])
		glEnd()
		glPopAttrib()

	def render_nomaterials(self):
		glMaterialfv(GL_FRONT,GL_AMBIENT,(0.2,0.2,0.2,1.0))
		glMaterialfv(GL_FRONT,GL_DIFFUSE,(0.8,0.8,0.8,1.0))
		glMaterialfv(GL_FRONT,GL_SPECULAR,(0.0,0.0,0.0,1.0))
		glMaterialf(GL_FRONT,GL_SHININESS,0.0)
		glBegin(GL_TRIANGLES)
		for i in xrange(len(self.points)):
			glNormal3fv(self.norms[i])
			glVertex3fv(self.points[i])
		glEnd()

	def render_materials(self):
		has_tverts = bool(self.tverts)
		for material,faces in self.matarrays:
			if (not material.texture or not has_tverts
			    or material.texture.dim == 0):
				def tcfunc(i):
					pass
			elif material.texture.dim == GL_TEXTURE_2D:
				def tcfunc(i):
					glTexCoord2fv(self.tverts[i])
			elif material.texture.dim == GL_TEXTURE_1D:
				def tcfunc(i):
					glTexCoord1f(self.tverts[i,0])
			else:
				assert False
			material.set_material(has_tverts)
			glBegin(GL_TRIANGLES)
			for f in faces:
				tcfunc(f*3)
				glNormal3fv(self.norms[f*3])
				glVertex3fv(self.points[f*3])
				tcfunc(f*3+1)
				glNormal3fv(self.norms[f*3+1])
				glVertex3fv(self.points[f*3+1])
				tcfunc(f*3+2)
				glNormal3fv(self.norms[f*3+2])
				glVertex3fv(self.points[f*3+2])
			glEnd()
			material.unset_material(has_tverts)

	def render(self):
		if self.matarrays:
			self.render_materials()
		else:
			self.render_nomaterials()


class GLModel(basicmodel.BasicModel):
	"""Subclass of BasicModel that renders the model in OpenGL.

	Provides two methods:

	    render() - issue the OpenGL commands to draw this model
	    create_dl() - create an OpenGL display list of this model,
	            and return the handle.

	"""

	meshclass = GLMesh
	matclass = GLMaterial

	def __init__(self,*args,**kwargs):
		super(GLModel,self).__init__(*args,**kwargs)

	def render(self):
		for m in self.meshes:
			m.render()

	def create_dl(self):
		dl = glGenLists(1)
		if dl == 0:
			raise GLError, "cannot allocate display list"
		glNewList(dl,GL_COMPILE)
		self.render()
		glEndList()
		return dl
