# basicmodel.py

"""Basic abstract classes representing a 3DS model.

Defines some classes that represent objects and materials of a 3DS
file in a more convienient form. It has methods to convert from the
DOM format. The classes can serve as base classes for more advanced
uses.

"""

import Numeric as N
from Dice3DS import dom3ds, util


def _colorf(color,alpha,default):
	if color is None:
		return (default,default,default,alpha)
	if type(color) is dom3ds.COLOR_24:
		return (color.red/255.0,color.green/255.0,
			color.blue/255.0,alpha)
	return (color.red,color.green,color.blue,alpha)


def _pctf(pct,default):
	if pct is None:
		return default
	if type(pct) is dom3ds.INT_PERCENTAGE:
		return pct.value/100.0
	return pct.value


class BasicMaterial(object):
	"""Represents a material from a 3DS file.

	This class, instances of which a BasicModel instance is
	usually responsible for creating, lists the material
	information.

	    mat.name - name of the mateiral
	    mat.ambient - ambient color
	    mat.diffuse - diffuse color
	    mat.specular - specular color
	    mat.shininess - shininess exponent
	    mat.texture - texture object
	    mat.twosided - whether to paint both sides

	"""

	def __init__(self,name,ambient,diffuse,specular,
		     shininess,texture,twosided):
		self.name = name
		self.ambient = ambient
		self.diffuse = diffuse
		self.specular = specular
		self.shininess = shininess
		self.texture = texture
		self.twosided = twosided


class BasicMesh(object):
	"""Represents a single mesh from a 3DS file.

	This class, instances of which a BasicModel instance is
	usually responsible for creating, takes a mesh data passed in
	from a 3DS DOM, reduces the data somewhat, and calculates
	normals.

	    mesh.name - name of the mesh
	    mesh.matarrays - list of materials and their corresponding
	        faces
	    mesh.points - Numeric array of poitns, one row per
	        vertex, each set of three adjacent rows representing
		a triangle
	    mesh.norms - corresponding array of normal vectors
	    mesh.tverts - correspondint array of texture coordinates

	"""
	
	def __init__(self,name,facearray,pointarray,
		     smarray,tvertarray,matrix,matarrays,
		     nfunc=util.calculate_normals_by_angle_subtended):

		self.name = name
		self.matarrays = matarrays

                if matrix is None:
                        matrix = N.identity(4,N.Float)
		pointarray = util.translate_points(pointarray,matrix)
		self.points,self.norms = nfunc(pointarray,facearray,smarray)

		if tvertarray:
			self.tverts = N.take(tvertarray,facearray.flat)
		else:
			self.tverts = None


class BasicModel(object):
	"""Represents a basic model from a 3DS file.

	This is an example of a more usable and direct representation
	of a 3DS model.  It can sometimes be hard to operate on data
	while it's sitting in a 3DS DOM; this class and the related
	classes BasicMesh and BasicMaterial make it more accessible.

	"""

	meshclass = BasicMesh
	matclass = BasicMaterial

	def __init__(self,dom,load_texture_func):
		self.load_texture_func = load_texture_func
		try:
			self.extract_materials(dom)
			self.extract_meshes(dom)
		finally:
			del self.load_texture_func

	def extract_materials(self,dom):
		self.materials = {}
		for mat in dom.mdata.materials:
			m = self.create_material(mat)
			self.materials[m.name] = m

	def extract_meshes(self,dom):
		self.meshes = []
		for nobj in dom.mdata.objects:
			obj = nobj.obj
			if type(obj) is not dom3ds.N_TRI_OBJECT:
				continue
                        if obj.faces is None:
                                continue
			if obj.faces.nfaces < 1:
				continue
			self.meshes.append(self.create_mesh(nobj.name,obj))

	def create_material(self,mat):
		name = mat.name.value
		alpha = 1.0 - _pctf(mat.transparency
				    and mat.transparency.pct,0.0)
		ambient = _colorf(mat.ambient
				  and mat.ambient.color,alpha,0.2)
		diffuse = _colorf(mat.diffuse
				  and mat.diffuse.color,alpha,0.8)
		specular = _colorf(mat.specular
				   and mat.specular.color,alpha,0.0)
		shininess = _pctf(mat.shininess
				  and mat.shininess.pct,0.0)
		texture = (mat.texmap and self.load_texture_func(
			mat.texmap.filename.value))
		twosided = bool(mat.two_side)
		return self.matclass(name,ambient,diffuse,specular,
				     shininess,texture,twosided)

	def create_mesh(self,name,nto):
		facearray = N.array(nto.faces.array[:,:3])
		smarray = nto.faces.smoothing and nto.faces.smoothing.array
		pointarray = nto.points.array
		tvertarray = nto.texverts and nto.texverts.array
		matrix = nto.matrix and nto.matrix.array
		matarrays = []
		for m in nto.faces.materials:
			matarrays.append((self.materials[m.name],m.array))
		return self.meshclass(name,facearray,pointarray,smarray,
				      tvertarray,matrix,matarrays)

	def center_of_gravity(self):
		cg = N.zeros((3,),N.Float)
		n = 0
		for mesh in self.meshes:
			cg += N.sum(mesh.points)
			n += len(mesh.points)
		return cg/n

	def bounding_box(self):
		ul = N.maximum.reduce([ N.maximum.reduce(mesh.points)
					for mesh in self.meshes ])
		ll = N.minimum.reduce([ N.minimum.reduce(mesh.points)
					for mesh in self.meshes ])
		return ul,ll
