using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Data;
using System.Text;
using System.Windows.Forms;

namespace StarifficEditor
{
    public delegate void SelectedObjectChanged(LevelObject obj);

    public partial class EditorControl : UserControl
    {
        public Level level;

        private LevelObject m_selected_obj;
        public LevelObject SelectedObject
        {
            get { return m_selected_obj; }
            set
            {
                if (m_selected_obj == value)
                    return;

                m_selected_obj = value;
                if (SelectedObjectChangedEvent != null)
                    SelectedObjectChangedEvent(value);
                Invalidate();
            }
        }

        private bool m_wireframe;
        public bool Wireframe
        {
            get { return m_wireframe; }
            set { m_wireframe = value; Invalidate(); }
        }

        private bool m_preview;
        public bool Preview
        {
            get { return m_preview; }
            set { m_preview = value; Invalidate(); }
        }

        private bool m_show_screen = false;
        public bool ShowScreen
        {
            get { return m_show_screen; }
            set { m_show_screen = value; Invalidate(); }
        }

        public event SelectedObjectChanged SelectedObjectChangedEvent;

        private enum MouseState
        {
            None,
            Selecting,
            Dragging,
            ShowingContextMenu,
        }

        private float m_top_x = 0;
        private float m_top_y = 0;

        private MouseState m_mouse_state = MouseState.None;
        private PointF m_last_mouse = new PointF();
        private bool m_mouse_on_control;

        private PointF m_mouse_pos_start;

        public EditorControl()
        {
            InitializeComponent();

            SetStyle(ControlStyles.UserPaint, true);
            SetStyle(ControlStyles.AllPaintingInWmPaint, true);
            SetStyle(ControlStyles.OptimizedDoubleBuffer, true);

            level = new Level();
            level.level_changed_event += new LevelChanged(LevelChangeEvent);

            LevelChangeEvent();
        }

        public PointF GetCenterMapCoord()
        {
            return MapMouseToLevel(new Point(Width / 2, Height / 2));
        }

        private void LevelChangeEvent()
        {
            DoLevelLayout();
            Invalidate();
        }

        private void DoLevelLayout()
        {
            RectangleF bbox = level.BoundingBox;
            float extra = 50.0f;

            m_top_x = bbox.Left - extra;
            m_top_y = bbox.Top - extra;
            AutoScrollMinSize = new Size((int)(bbox.Right - m_top_x + extra), (int)(bbox.Bottom - m_top_y + extra));
        }

        private void EditorControl_Paint(object sender, PaintEventArgs e)
        {
            Graphics g = e.Graphics;

            g.Clear(Color.Black);
            g.Transform = new Matrix(1, 0, 0, 1, -m_top_x + AutoScrollPosition.X, -m_top_y + AutoScrollPosition.Y);

            // Draw the objects
            for (int i = level.objects.Count - 1; i >= 0; --i)
            {
                LevelObject obj = level.objects[i];

                bool selected = (m_selected_obj == obj);
                if (m_preview)
                    selected = true;
                obj.Draw(g, selected, m_wireframe);
            }

            // Draw the screen region
            if (ShowScreen && m_mouse_on_control)
            {
                Pen p = new Pen(new HatchBrush(HatchStyle.DarkDownwardDiagonal, Color.White, Color.Black));
//                p.DashStyle = DashStyle.Dash;
                float w = level.options.ScreenWidth;
                float h = level.options.ScreenHeight;

                g.DrawRectangle(p, m_last_mouse.X - w / 2, m_last_mouse.Y - h / 2, w, h);
            }
        }

        private PointF MapMouseToLevel(Point pos)
        {
            return new PointF( m_top_x + pos.X - AutoScrollPosition.X, m_top_y + pos.Y - AutoScrollPosition.Y );
        }

        private void EditorControl_MouseMove(object sender, MouseEventArgs e)
        {
            HandleMove(e);
        }

        private void EditorControl_MouseDown(object sender, MouseEventArgs e)
        {
            HandleMove(e);

            switch (m_mouse_state)
            {
                case MouseState.None:
                    if (e.Button == MouseButtons.Left)
                    { // Select / deselect / drag
                        m_mouse_state = MouseState.Selecting;
                        if (SelectedObject != null)
                        { // Clicked on object?
                            if (SelectedObject.IsPointOnObject(m_last_mouse))
                                m_mouse_state = MouseState.Dragging;
                            else
                                SelectedObject = null;
                        }

                        // Select new?
                        if (SelectedObject == null)
                        {
                            SelectedObject = level.FindObject(m_last_mouse);
                        }
                    }

                    if (e.Button == MouseButtons.Right)
                    { // If clicked on a selected object, pop up a menu strip
                        if (SelectedObject != null)
                        {
                            if (SelectedObject.IsPointOnObject(m_last_mouse) == false)
                                SelectedObject = null;
                        }
                        m_mouse_pos_start = m_last_mouse;
                        m_mouse_state = MouseState.ShowingContextMenu;
                    }
                    break;
            }
        }

        private void EditorControl_MouseUp(object sender, MouseEventArgs e)
        {
            HandleMove(e);

            switch (m_mouse_state)
            {
                case MouseState.Dragging:
                    if (e.Button != MouseButtons.Left)
                        return;

                    if (m_mouse_state == MouseState.Dragging)
                    {
                        level.NotifyChanges();
                    }
                    break;

                case MouseState.Selecting:
                    if (e.Button != MouseButtons.Left)
                        return;
                    break;

                case MouseState.ShowingContextMenu:
                    if (e.Button != MouseButtons.Right)
                        return;
                    ShowContextMenu(m_mouse_pos_start, e.Location);
                    break;
            }
            m_mouse_state = MouseState.None;
        }

        private void HandleMove(MouseEventArgs e)
        {
            // Update the moving
            PointF new_pos = MapMouseToLevel(e.Location);
            if (new_pos == m_last_mouse)
                return;
            PointF old_pos = m_last_mouse;

            m_last_mouse = new_pos;

            // Dragging object?
            if (m_mouse_state == MouseState.Dragging)
            {
                if (SelectedObject != null)
                {
                    SelectedObject.X += new_pos.X - old_pos.X;
                    SelectedObject.Y += new_pos.Y - old_pos.Y;
                    Invalidate();
                }
            }

            // Redraw because we're showing the screen area?
            if (ShowScreen)
                Invalidate();
        }

        private void EditorControl_MouseEnter(object sender, EventArgs e)
        {
            m_mouse_on_control = true;
            Invalidate();
        }

        private void EditorControl_MouseLeave(object sender, EventArgs e)
        {
            m_mouse_on_control = false;
            Invalidate();
        }

        private void ShowContextMenu(PointF pt, Point menu_pos)
        {
            LevelObject[] objs = level.FindObjects(pt);
            if (objs.Length <= 0)
                return;

            ContextMenuStrip menu = new ContextMenuStrip();

            if (SelectedObject != null)
            {
                menu.Items.Add("Bring to front", null, delegate { BringToFrontOrBack(true); });
                menu.Items.Add("Send to back", null, delegate { BringToFrontOrBack(false); });
                menu.Items.Add("-");
                menu.Items.Add("Clone", null, delegate { CloneCurrentObject(); });
                menu.Items.Add("-");
            }

            foreach (LevelObject obj in objs)
            {
                LevelObject o = obj; // This is to work around the "obj" into the inner scope so that the created delegate uses a unique reference for each item
                menu.Items.Add("Select: \"" + obj.ToString() + "\"", null, delegate { SelectedObject = o; });
            }

            menu.Show(this, menu_pos);
        }

        private void CloneCurrentObject()
        {
            if (SelectedObject == null)
                return;
            LevelObject obj = (LevelObject) SelectedObject.Clone();

            // Insert to the same level
            int at_idx = 0;
            int cur_idx = level.objects.IndexOf(SelectedObject);
            if (cur_idx != -1)
                at_idx = cur_idx;

            // Insert
            level.objects.Insert(at_idx, obj);
            level.NotifyChanges();

            SelectedObject = obj;
        }

        private void BringToFrontOrBack(bool front)
        {
            if (SelectedObject == null)
                return;
            int idx = level.objects.IndexOf(SelectedObject);
            if (idx < 0)
                return;

            LevelObject obj = SelectedObject;

            level.objects.RemoveAt(idx);
            if (front)
                level.objects.Insert(0, obj);
            else
                level.objects.Insert(level.objects.Count, obj);
            level.NotifyChanges();
        }
    }
}
