
/**
 * Definitions for a 2D verlet. 
 */
public class Verlet
{
	public int[] pos;
	public int[] old_pos;

	public Verlet()
	{
		pos = Vec2.create();
		old_pos = Vec2.create();
	}

	/**
	 * Resets the verlets position, velocity and acceleration.
	 * 
	 * @param new_pos The new position.
	 */
	public void resetPos( int[] new_pos )
	{
	    Vec2.set( pos, new_pos );
	    Vec2.set( old_pos, new_pos );
	}
	
	/**
	 * Sets the verlets velocity.
	 * 
	 * @param vel The velocity to set.
	 */
	public void setVelocity( int[] vel )
	{
	    Vec2.sub( old_pos, pos, vel );
	}
	
	/**
	 * Adds velocity to the verlet.
	 * 
	 * @param vel The velocity to add.
	 */
	public void addVelocity( int[] vel )
	{
	    Vec2.sub( old_pos, vel );
	}
	
	/**
	 * Returns the current approximate velocity (per frame).
	 * 
	 * @param dest_vel The destination vector.
	 */
	public void getVelocity( int[] dest_vel )
	{
	    Vec2.sub( dest_vel, pos, old_pos );
	}

	/**
	 * Moves the verlets position without changing it's velocity.
	 * 
	 * @param delta_vec The relative delta vector.
	 */
	public void movePosRel( int[] delta_vec )
	{
	    Vec2.add( pos, delta_vec );
	    Vec2.add( old_pos, delta_vec );
	}

    /**
     * Applies friction to the verlet.
     * 
     * @param tangent The tangential vector (unit length).
     * @param amt Amount of force to put into it.
     */
    public void applyFriction( int[] tangent, int amt )
    {
        getVelocity( g_temp_vec );
        amt = FP.mul( amt, Vec2.dot( tangent, g_temp_vec ) );
        Vec2.addScaled( g_temp_vec, g_temp_vec, tangent, -amt );
        
        setVelocity( g_temp_vec );
    }
    
    /**
     * Applies a bump to the verlet.
     * 
     * @param nx Normal x.
     * @param ny Normal y.
     * @param amt Bump amount.
     */
    public void bump(int nx, int ny, int amt)
    {
        getVelocity( g_temp_vec );
        
/*      // Get the amount to move it (-2*dot(normal,dir))
        int mamt = -2 * (FP.mul(nx,g_temp_vec[0]) + FP.mul(ny,g_temp_vec[1]));
        if (mamt <= 0)
            return;
        
        // Make the mirrored & scaled velocity
        nx = FP.mul(mamt, nx);
        ny = FP.mul(mamt, ny);
        
        g_temp_vec[0] = FP.mul(g_temp_vec[0] + nx, amt);
        g_temp_vec[1] = FP.mul(g_temp_vec[1] + ny, amt);
*/
        
        // Add the impact
        g_temp_vec[0] += FP.mul(nx, amt);
        g_temp_vec[1] += FP.mul(ny, amt);
        
        // Set it
        setVelocity( g_temp_vec );
    }
    
	/**
	 * Makes one integration step.
	 *
	 * @param time_step_2		The time step to the power of 2 (t^2).
	 * @param accel				The acceleration vector.
	 */
	public void integrate( int time_step_2, int[] accel )
	{
		for ( int i = 0; i < 2; ++i )
		{
			int old = old_pos[ i ];

			// X(next) = 2*X(cur) - X(old) + A*(T^2)
			// X(old) = X(cur)
			
			old_pos[ i ] = pos[ i ];
			pos[ i ] = (pos[ i ] << 1) - old;
			if ( accel[ i ] != 0 )
				pos[ i ] += FP.mul( accel[ i ], time_step_2 );
		}
	}
    
    private static final int[] g_temp_vec = new int[ 2 ];
}
