import java.io.*;
import java.util.*;

import javax.microedition.lcdui.*;

public class Level
{
    public static final int DIR_RIGHT = 0;
    public static final int DIR_DOWN = 1;
    public static final int DIR_LEFT = 2;
    public static final int DIR_UP = 3;
    
    /** Screen width in level coords. */
    public int screen_width;
    /** Screen height in level coords. */
    public int screen_height;
    /** Screen aspect ratio. */
    public int screen_aspect;
    
    public BoundingBox level_bbox;
    public LevelObject[] objects;
    
    public Star star;
    public Goal goal;
    
    // Draw the particles to the base
    public ParticleSystem particles;
    
    private int m_fetch_cache_number;
    
    // Play info
    public int time_bronze;
    public int time_silver;
    public int time_gold;
    public int current_top_score = FP.toFP(-1);
    
    public int play_time;
    public int collectibles_left;
    
    public int gravity;
    public int gravity_dir = DIR_DOWN;
    public int[] gravity_vec = Vec2.create( 0, gravity );
    
    // Grid info
    private int m_grid_x, m_grid_y, m_grid_slot_size;
    private int m_grid_w, m_grid_h;
    private LevelObject[][][] m_grid;
    
    // Grid search info
    private int m_search_x1, m_search_x2, m_search_y1, m_search_y2;
    
    // Screen info
    private BoundingBox m_screen_box = new BoundingBox();
    private Vector m_screen_objs = new Vector();

	public Level(byte[] data)
	{
		try
		{
			InputStream in = new ByteArrayInputStream(data);
			loadLevel(new DataInputStream(in));
			in.close();
		}
		catch (Throwable t)
		{
            t.printStackTrace();
			throw new Error("Error loading the level: " + t);
		}
	}
	
	public Level(String res_name, int top_score)
	{
		try
		{
			InputStream in = Res.getResourceAsStream(res_name);
			loadLevel(new DataInputStream(in));
			in.close();
            
            current_top_score = top_score;
		}
		catch (Throwable t)
		{
            t.printStackTrace();
			throw new Error("Error loading the level: " + t);
		}
	}
	
    /**
     * Sets a new gravity direction.
     * 
     * @param dir The new direction.
     */
    public void setGravity( int dir )
    {
        gravity_dir = dir;
        
        switch (dir)
        {
        case DIR_RIGHT:
            gravity_vec[0] = gravity;
            gravity_vec[1] = 0;
            break;

        case DIR_DOWN:
            gravity_vec[0] = 0;
            gravity_vec[1] = gravity;
            break;

        case DIR_LEFT:
            gravity_vec[0] = -gravity;
            gravity_vec[1] = 0;
            break;

        case DIR_UP:
            gravity_vec[0] = 0;
            gravity_vec[1] = -gravity;
            break;
        }
    }
    
	/**
	 * Fetches a list of level objects possibly intersecting with the given bounding box.
	 * 
	 * @param bbox The bounding box to test against.
	 * @params dest_list The destination list.
	 */
	public void regionFetch( BoundingBox bbox, Vector dest_list )
	{
		dest_list.removeAllElements();
		
        // Find the search region
        if ( !getBoundingBoxRegion( bbox ) )
            return;
        
		// Get a new new cache number
		int cache = ++m_fetch_cache_number;
		if ( cache < 0 )
		{ // Reset the cache numbers on overflow
			cache = m_fetch_cache_number = 1;
			
            for ( int i = 0; i < objects.length; ++i )
                objects[ i ].region_fetch_cache = 0;
		}
		
//        System.out.println("reg " + m_search_x1 + "," + m_search_y1 + "   " + m_search_x2+","+m_search_y2 );
//        System.out.println("find " + bbox);
        
		// Search
        for ( int y = m_search_y1; y <= m_search_y2; ++y )
        {
            LevelObject[][] list = m_grid[y];
            for ( int x = m_search_x1; x <= m_search_x2; ++x )
            {
                LevelObject[] objs = list[x];
                for ( int i = 0; i < objs.length; ++i )
                {
                    LevelObject obj = objs[i];
                    
                    // Check for if this has been processed already
                    if ( obj.region_fetch_cache == cache )
                        continue;
                    obj.region_fetch_cache = cache;
                    
                    // Test for intersection
                    if ( bbox.intersects( obj.bounding_box ) )
                        dest_list.addElement( obj );
                }
            }
        }
	}
	
    /**
     * Updates the internal m_search_* variables so that they span over the given bounding box.
     * 
     * @param box The bounding box to search for.
     * @return False if the bbox is not in the area.
     */
    private boolean getBoundingBoxRegion( BoundingBox box )
    {
        m_search_x1 = FP.toInt( FP.div( box.x1 - m_grid_x, m_grid_slot_size ) );
        m_search_y1 = FP.toInt( FP.div( box.y1 - m_grid_y, m_grid_slot_size ) );
        m_search_x2 = FP.toCeilInt( FP.div( box.x2 - m_grid_x, m_grid_slot_size ) );
        m_search_y2 = FP.toCeilInt( FP.div( box.y2 - m_grid_y, m_grid_slot_size ) );

        if (m_search_x1 >= m_grid_w || m_search_y1 >= m_grid_h || m_search_x2 < 0 || m_search_y2 < 0)
            return false;
        
        if (m_search_x1 < 0) m_search_x1 = 0;
        if (m_search_y1 < 0) m_search_y1 = 0;
        if (m_search_x2 >= m_grid_w) m_search_x2 = m_grid_w - 1;
        if (m_search_y2 >= m_grid_h) m_search_y2 = m_grid_h - 1;
        
        if (m_search_x1 > m_search_x2 || m_search_y1 > m_search_y2)
            return false;
        
        return true;
    }
    
    /**
     * Updates the level.
     * 
     * @param fp_secs The time to update, in fixed point seconds.
     */
    public void update( int fp_secs )
    {
        int iters = 1;
        int time_iter = fp_secs / iters;
        
        // Calculate the played time
        if ( !star.is_at_home )
            play_time += fp_secs;
        
        // Simulate the objects
        while ( iters-- > 0 )
        {
            for ( int i = 0; i < objects.length; ++i )
                objects[ i ].update( time_iter );
        }
        
        // Update the particles
        particles.update( fp_secs, gravity_vec );
                
        // Out of level?
        if ( !star.is_at_home && !star.is_dead )
        {
            if (level_bbox.contains(star.getX(), star.getY()) == false)
                star.kill();
        }
    }

    /**
     * Renders the current state to the given graphics.
     * 
     * @param g The destination graphics.
     */
    public void render( Graphics g, int top_x, int top_y, int scale_x, int scale_y, int vis_w, int vis_h, int scr_w, int scr_h )
    {
        // Make the screen bounding box
        m_screen_box.x1 = top_x;
        m_screen_box.y1 = top_y;
        m_screen_box.x2 = top_x + vis_w;
        m_screen_box.y2 = top_y + vis_h;
        
        // Get the objects to render
        regionFetch( m_screen_box, m_screen_objs );
        m_screen_objs.addElement( star );
        
        // Sort the objects
        sort( m_screen_objs );
        
        // Render the objects
        int size = m_screen_objs.size();
        for ( int i = 0; i < size; ++i )
        {
            LevelObject obj = (LevelObject) m_screen_objs.elementAt( i );
            obj.render( g, top_x, top_y, scale_x, scale_y );
        }
        
        // DEBUG: Renders all the objects
//        for ( int i = 0; i < objects.length; ++i )
//            objects[i].render( g, top_x, top_y, scale_x, scale_y );
        
        // Render the particles
        particles.render( g, top_x, top_y, scale_x, scale_y );
        
        // Show the time
        Font f = Font.getFont( Font.FACE_PROPORTIONAL, Font.STYLE_BOLD | Font.STYLE_ITALIC, Font.SIZE_MEDIUM );
        g.setFont( f );
        
        String time_str = MiscUtils.timeToString( play_time );
        
        g.setColor( 0xff111111 );
        g.fillRect( 2, 2, 2 + f.stringWidth( time_str ), 2 + f.getHeight() );
        
        g.setColor( 0xffffffff );
        g.drawString( time_str, 2, 2, Graphics.TOP | Graphics.LEFT );
        
        // Show the bronze/silver/gold time
        if ( true )
        {
            int time = current_top_score;
            if ( time < 0 ) time = time_bronze + FP.toFP(99);
//          time = Math.min( time, play_time );
            
            if ( time <= time_gold )
                time_str = "Best: " + MiscUtils.timeToString( time );
            else if ( time <= time_silver )
                time_str = "Gold: " + MiscUtils.timeToString( time_gold );
            else if ( time <= time_bronze )
                time_str = "Silver: " + MiscUtils.timeToString( time_silver );
            else
                time_str = "Bronze: " + MiscUtils.timeToString( time_bronze );
        }
        
        if ( time_str != null )
        {
            int str_w = f.stringWidth( time_str );
            
            g.setColor( 0xff111111 );
            g.fillRect( scr_w - str_w - 2, 2, 2 + str_w, 2 + f.getHeight() );
            
            g.setColor( 0xffffffff );
            g.drawString( time_str, scr_w - str_w - 2, 2, Graphics.TOP | Graphics.LEFT );
        }
        
        // Show the amount of collectibles left
        if ( collectibles_left > 0 )
        {
            String str = "Stars: " + collectibles_left;
            int y = f.getHeight();
            
            g.setColor( 0xff111111 );
            g.fillRect( 2, y+2, 2 + f.stringWidth( str ), 2 + f.getHeight() );
            
            g.setColor( 0xffffffff );
            g.drawString( str, 2, y+2, Graphics.TOP | Graphics.LEFT );
        }
    }
    
    /**
     * Does a comb sort (combsort11) to the given list of level objects.
     */
    private static void sort( Vector objs )
    {
        int size = objs.size();
        int gap = size;
        
        boolean swapped;
        do
        {
            swapped = false;
            
            // Get the new gap size
            gap = (10 * gap) / 13;
            if ( gap == 9 || gap==10 )
                gap = 11;
            else if ( gap < 1 )
                gap = 1;
            
            // Process the objects with this gap
            for ( int i = 0; i < (size - gap); ++i )
            {
                LevelObject o1 = (LevelObject) objs.elementAt( i );
                LevelObject o2 = (LevelObject) objs.elementAt( i + gap );
                
                if ( o1.draw_index > o2.draw_index )
                {
                    swapped = true;
                    objs.setElementAt( o1, i + gap );
                    objs.setElementAt( o2, i );
                }
            }
        } while ( swapped || gap > 1 );
    }

    /**
     * Loads the level from the given input stream.
     * 
     * @param in The input stream.
     */
    private void loadLevel(DataInputStream in) throws IOException
    {
        // Load the screen sizes
        screen_width = in.readInt();
        screen_height = in.readInt();
        screen_aspect = in.readInt();
        
        // Load and set the gravity
        gravity = in.readInt();
        setGravity( DIR_DOWN );
        
        // Load the level times
        time_bronze = in.readInt();
        time_silver = in.readInt();
        time_gold = in.readInt();

    	// Load the total bounding box for the level
        {
            int x = in.readInt();
            int y = in.readInt();
            int w = in.readInt();
            int h = in.readInt();
            level_bbox = new BoundingBox( x, y, w, h );
        }
        
        // Load the level objects
        int level_obj_count = in.readShort();
        objects = new LevelObject[ level_obj_count ];
        for ( int i = 0; i < level_obj_count; ++i )
        {
            int type = in.readByte() & 0xff;
            LevelObject obj = null;
            
            // Create the object
            switch ( type )
            {
            case 0:     obj = new Star(); break;
            case 1:     obj = new Goal(); break;
            case 10:    obj = new WallRectangle(); break;
            case 11:    obj = new WallSphere(); break;
            case 20:    obj = new MeshObject(); break;
            case 30:    obj = new ItemObject(); break;
            }
            if ( obj == null )
                throw new Error( "Unknown level object: " + type );
            
            obj.level = this;
            obj.draw_index = i;

            // Read the bounding box
            {
                int x = in.readInt();
                int y = in.readInt();
                int w = in.readInt();
                int h = in.readInt();
                obj.bounding_box = new BoundingBox( x, y, w, h );
            }
            // Star?
            if (obj instanceof Star)
            {
                if (star != null)
                    throw new Error( "More than one starting position found!" );
                star = (Star)obj;
            }
            // Goal?
            if (obj instanceof Goal)
            {
                if (goal != null)
                    throw new Error( "More than one ending position found!" );
                goal = (Goal)obj;
            }
            
            // Load the object
            obj.load( in );
            objects[ i ] = obj;
            
            // Was a collectible?
            if ((obj instanceof ItemObject) && ((ItemObject)obj).getType() == ItemObject.ITEM_COLLECTIBLE )
            {
                ++collectibles_left;
            }
        }
        
        // Load the grid
        m_grid_x = in.readInt();
        m_grid_y = in.readInt();
        m_grid_slot_size = in.readInt();
        m_grid_w = in.readShort();
        m_grid_h = in.readShort();
        
        m_grid = new LevelObject[m_grid_h][m_grid_w][];
        for ( int y = 0; y < m_grid_h; ++y )
            for ( int x = 0; x < m_grid_w; ++x )
            {
                LevelObject[] objs = new LevelObject[ in.readShort() ];
                for ( int i = 0; i < objs.length; ++i )
                {
                    int idx = in.readShort();
                    objs[ i ] = this.objects[ idx ];
                }
                m_grid[y][x] = objs;
            }
        
        // Create the particle system
        particles = new ParticleSystem( 32 );
        
        // Found a starting/ending location?
        if (star == null)
            throw new Error( "No starting location found!" );
        if (goal == null)
            throw new Error( "No ending location found!" );
    }
}
